/* [lwpproc.c wk 16.01.94] LWP controlling functions
 *	Copyright (c) 1994 by Werner Koch (dd9jn)
 * This file is part of WkLib.
 *
 * WkLib is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * WkLib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <wk/tailor.h>
RCSID("$Id: lwpproc.c,v 1.6 1995/03/08 16:56:42 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <wk/lwp.h>

#if defined(OS20)
    #define INCL_NOCOMMON 1
    #define INCL_DOSPROCESS 1
    #define INCL_DOSSEMAPHORES 1
    #define INCL_DOSERRORS 1
    #include <os2.h>
    #define THREADAPI _System
#else
    #define THREADAPI
#endif



/************************************************
 ************  local variables	*****************
 ************************************************/

static t_sema threadCounter;

#if OS20
static void THREADAPI RunThread( t_thread tib );
#endif

/************************************************
 ************  local functions	 ****************
 ************************************************/
#if OS20
static HMTX serializeSemaphore;

static void RequestSerialization(void)
{
    int rc;
    if( (rc = DosRequestMutexSem( serializeSemaphore , 1000 )) )
        Error(8,"process: error requesting internal serialization: rc=%d", rc);
}

static void ReleaseSerialization(void)
{
    int rc;
    if( (rc = DosReleaseMutexSem( serializeSemaphore )) )
        Error(4,"process: error releasing internal serialization: rc=%d", rc);
}

#else	/* all other OS */
#define RequestSerialization()	   do {    } while(0)
#define ReleaseSerialization()	   do {    } while(0)
#endif


#if OS20 /* currently only for OS2 2.x */
static void THREADAPI RunThread( t_thread tib )
{
    WaitSema(tib->active);
    tib->rc = tib->fnc(tib->parm);
    SignalSema(tib->active);
    SignalSema(threadCounter);
  #if OS20
    DosExit(EXIT_THREAD, 0);
  #else /* falls into an endless loop */
    /* remove from dispatcher list */
       Bug("module closed - function construction ahead");
    RelinquishThread();
    BUG();
  #endif
}
#endif

/************************************************
 ************  public functions  ****************
 ************************************************/


/****************
 * Create a new thread and return a ThreadIdentificationBlock.
 * Parameters are:
 *	fnc ::= the function which this thread shoulf call
 *	parm::= a pointer which is passed to the function
 *	stackWords ::= Number of words to be allocated for
 *		       the stack of the new thread; if the function
 *		       assumes a too low value it will use a default
 *		       value.
 */

t_thread CreateThread( int (*fnc)(void*), void *parm, size_t stackWords )
{
    t_thread tib;

    if( stackWords < 100 )
	stackWords = 1024;

    tib = NULL;
    RequestSerialization();
    if( !threadCounter )
	threadCounter = CreateSema(1);

    if( !(tib = calloc(1, sizeof *tib )) )
	goto failure;
    tib->parm = parm;
    tib->active = CreateSema( 0 );

  #if OS20
    {
	TID osTID;		/* tid internal to the OS    */
	if( DosCreateThread( &osTID,
			 (PFNTHREAD)RunThread,
			 (ULONG)tib,
			 0, /* Immediate Execution, default stack initial. */
			 stackWords*sizeof(int) ) )
	    goto failure;   /* creation error */
	DecrementSema(threadCounter);
    }
  #else
  #endif
    ReleaseSerialization();
    SignalSema(tib->active);   /* thaw the thread */
    return tib;

  failure:
    ReleaseSerialization();
    if( tib ) {
	free(tib);
    }
    return NULL;
}


/****************
 * Tests, wether the given thread is running, that is the thread
 * has begun execution but has not ended
 * Returns: 0 := Thread is not active
 *	    1 := Thread is running
 *	    2 := Thread is blocked  (not yet used)
 */

int GetThreadState( t_thread tib )
{
    int value;

    value = QuerySema( tib->active );
    if( value < 0 )
	return 0;
    return 1;
}


void RelinquishThread()
{
  #if OS20
    DosSleep(0); /* give up my timeslice */
  #else
    Bug("dont know how to process with another thread");
  #endif
}



/****************
 * Blocks the current thread until the given thread has terminated.
 * You may specify NULL as argument; in this case the thread is blocked
 * until all threads have terminated - this should only be called by
 * the main-thread.
 */

void WaitThread( t_thread tib )
{
    if( tib )
        Bug( "Waiting for specifiy thread is not yet supported");
    WaitSema(threadCounter);
}


#if 0 /* don't know wether these functions make sense */
/****************
 * Block a thread on a semaphore.
 * You should never block your own thread, use WaitSema directly
 * in this case. There is no guarantee that this function
 * will detect that it is the own thread.
 */

void BlockThread( t_thread tib, t_sema sema )
{
    BlockThread( tib, sema, 0 );
}

/****************
 * Block a thread on a semaphore; timeout after timeout milliseconds..
 * Returns: 0 := okay
 *	    not 0 := timed out
 */

int BlockThread( t_thread tib, t_sema sema, ulong timeout )
{
    int rc;
    if( !TestSema(sema) )
	return 0; /* no need for blocking */

  #if OS20
  #else
    #error Cannot block thread!
  #endif
    rc = SemaWaitTimed(sema, timeout);
  #if OS20
  #else
    #error Cannot thaw thread!
  #endif

}
#endif

/*** end of file ***/
