/* [memdbg.c wk 9.1.90] Memory Debug functions
 *	Copyright (c) 1988-93 by Werner Koch (dd9jn)
 *  This file is part of WkLib.
 *
 *  WkLib is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkLib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * helps in detection of heap allocation bugs
 * see also: xalloc.c
 *
 * History:
 * 14.11.91 wk	changed memcount to ulong, because OS/2 supports more mallocs
 *  8.07.92 wk	changed printf format to %p
 * 15.02.93 wk	added xrealloc()
 * 03.03.93 wk	NULL-ptr check for xstrdup()
 */

#include <wk/tailor.h>
RCSID("$Id: memdbg.c,v 1.16 1996/01/25 20:16:46 wernerk Exp $")
#include <stdio.h>
#include <stdlib.h>
    /* Microsoft C 6.0 Users: Warning:
     * You should at least patch the file "include/string.h"
     * because there are no precautions agains duplicate includes -
     * add the usual stuff at front and end. This module here
     * will not compile because the second prototype would be
     * modified with __FILE__ and __LINE__ to ..._debug(..
     * Sorry, I see no other easy way to do it.
     */
#include <string.h>
#if defined(OS20) && defined(MULTI_THREADED)
    #define INCL_NOCOMMON 1
    #define INCL_DOSSEMAPHORES 1
    #include <os2.h>
#endif

#ifndef MEM_DEBUG
    #define MEM_DEBUG 1
#endif /* we always need it here */

#include <wk/lib.h>
#include <wk/io.h>
#include <wk/string.h>

int _wklib_memdbgcheck;

/***** constants *****/
#define BEFOREVAL	0x12345678
#define AFTERVAL	0x87654321


#if MSDOS || OS2
#define BADVAL		0xFF
#define MALLOCVAL	0xEE
#else
#define BADVAL		0x7A
#define MALLOCVAL	0xEE
#endif

#if ALIGNMENT_REQUIRED
static long afterval = AFTERVAL; /* so we can get the address */
#endif


/***** globals *****/
static int initialized = 0;
static int atexitOkay = 0;	/* atexit function was called */
static int briefDebugInfo;	/* mode of debuginfo */
static int inReset = 0 ;	/* local Semaphore */
static ulong allocCount;	/* # of unfreed allocs */
static long maxalloced; 	/* max # of bytes allocated */
static long numalloced; 	/* cur # of allocated bytes */

#if MSDOS
   #define ferr  stdout 	/* cannot redirect stderr on MS-DOS */
#else
   #define ferr  stderr
#endif

Wklib_Serializer(memdbg);

/* we use a linked list to store infos about each alloc */
static struct s_alloclist {
    struct s_m {
	  struct s_alloclist *m_next;
	  struct s_alloclist *m_prev;
	  char *m_file; 		/* source filename */
	  int m_line;			/* source line number */
	  unsigned m_nbytes;		/* allocated size */
	  long m_beforeval;		/* underrun sentinel */
	} m;
	char data[1];		    /* where storage is allocated */
} alloclist = /* setup the head */
    { {  NULL, NULL, "unnamed", 7777, 0, BEFOREVAL }, { 0 } };

#define next		m.m_next
#define prev		m.m_prev
#define file		m.m_file
#define line		m.m_line
#define nbytes		m.m_nbytes
#define beforeval	m.m_beforeval


/******* protos **************/
static void _near Init(void);
static void _near DoAbort(int);
static void _near PrintAl( struct s_alloclist *);
static void _near FileLine( char *, int);

/****** defs *******/

/* Disable mapping macros */
#undef	malloc
#undef	calloc
#undef	strdup
#undef	realloc
#undef	free

#undef	xmalloc
#undef	xcalloc
#undef	xrealloc
#undef	xstrdup

/* Convert from a void * to a alloclist struct and reverse */
#define PTR2AL(p)  ((struct s_alloclist *) ((char*)p-sizeof(struct s_m)))
#define AL2PTR(al) ((void *) &((al)->data[0]))


/****** functions ************/

/*******************************************
 * memdbg control functions
 *******************************************
 */


/***************************
 * Reset Mem dbg System
 * may be called as often as needed
 * if envVar DEBUGINFO is set, this function will be called via
 * atexit ( which will be installed the first time a alloc-function is
 * used ) this function does not terminate the Program
 * ( as an prerequiste for an atexit function )
 */

void memdbg_reset()
{
    register struct s_alloclist *al;

    if( initialized ) {

	inReset = 1 ;	/* suppress exit() in DoAbort */
	memdbg_check();
	inReset = 0 ;
	if( !briefDebugInfo )
	    for(al = alloclist.next; al; al = al->next) {
		fprintf(ferr,"Unfreed ptr: ");
		PrintAl(al);
	    }
	if( allocCount )
	    fprintf(ferr,"Warning: %lu unfreed ptrs\n", allocCount );
	fprintf(ferr,"Max amount ever allocated == %ld bytes\n",
		     maxalloced);
	fflush(ferr); /* just in case, that a crash follows */
	initialized = 0;
    }
}


/***************************
 *  Check allocated Memory
 */

void memdbg_check()
{
    register struct s_alloclist *al;

    Init();
    for( al = alloclist.next; al ; al = al->next )
	memdbg_checkptr( AL2PTR(al) );
}

/***************************
 * Check one allocated Pointer
 */

void memdbg_checkptr(register void *p )
{
    register struct s_alloclist *al;

    Init();
    for( al = alloclist.next; al; al = al->next) {
	if( p >= (void *)&(al->data[0]) &&
	    p < (void *)((char*)al + sizeof(struct s_alloclist)-1+al->nbytes))
	    break;
    }
    if( !al ) {
	DoAbort( 2 );
	return ;
    }

    al = PTR2AL(p);
    if( al->beforeval != BEFOREVAL ) {
	fprintf(ferr,"Ptr %p underrun\n",p);
	goto errLabel;
    }

  #if ALIGNMENT_REQUIRED
    if( memcmp(&al->data[al->nbytes], &afterval, sizeof(AFTERVAL)) )
  #else
    if( *(long *) &al->data[al->nbytes] != AFTERVAL )
  #endif
    {
	fprintf(ferr,"Ptr %p overrun\n",p);
	goto errLabel;
    }
    return;

  errLabel:
    PrintAl( al );
    DoAbort(3);
    return;
}


/****************
 * Informationen ber den Heap zurckgeben
 * level 0 = aktuelle Anzahl der allokierten Bytes
 *	 1 = max allokierte Anzahl von Bytes
 */

long memdbg_info( int level )
{
    switch( level ) {
      case 0: return numalloced;
      case 1: return maxalloced;
    }
    BUG();  /*NOTREACHED*/ return 0;
}


/***************************************************************
 * Debug versions of malloc(), calloc(), free() and realloc().
 ***************************************************************
 */

void *malloc_debug( unsigned n, char *fil, int lin)
{
    void *p;

    Init();
    if( (p = calloc_debug(n,1,fil,lin)) )
	memset(p, MALLOCVAL, n);
    return p;
}


void *calloc_debug( unsigned n, unsigned m, char *fil, int lin)
{
    struct s_alloclist *al;

    Init();
    n *= m;
    al = calloc( sizeof(*al) + n + sizeof(AFTERVAL) - 1, 1);

    if( !al )
	return NULL;
    else {
	al->file = fil;
	al->line = lin;
	al->nbytes = n;
	al->beforeval = BEFOREVAL;
      #if ALIGNMENT_REQUIRED
	memcpy( &(al->data[n]), &afterval, sizeof(AFTERVAL) );
      #else
	*(long*)&(al->data[n]) = AFTERVAL;
      #endif

	/* Add al to start of allocation list */
	Wklib_BeginSerialize(memdbg);
	al->next = alloclist.next;
	al->prev = &alloclist;
	alloclist.next = al;
	if( al->next )
	    al->next->prev = al;

	allocCount++;
	numalloced += n;
	if (numalloced > maxalloced)
	    maxalloced = numalloced;
	Wklib_EndSerialize(memdbg);
	return AL2PTR( al );
    }
} /* calloc_debug() */


void free_debug(void *ptr, char *fil, int lin)
{
    struct s_alloclist *al;

    Init();
    if( !ptr )
	return;

    if( !allocCount ) {
	 fprintf(ferr, "More frees than allocs at ");
	 goto errLabel;
    }

    al = PTR2AL(ptr);
    if( al->beforeval != BEFOREVAL ) {
	 fprintf(ferr,"Ptr %p underrun\n",ptr);
	 goto errLabel2;
    }

  #if ALIGNMENT_REQUIRED
    if( memcmp(&al->data[al->nbytes], &afterval,sizeof(AFTERVAL)) )
  #else
    if( *(long *) &al->data[al->nbytes] != AFTERVAL )
  #endif
    {
	fprintf(ferr,"Ptr %p overrun\n",ptr);
	goto errLabel2;
    }
    numalloced -= al->nbytes;
    if( numalloced < 0 ) {
	fprintf(ferr, "error: numalloced = %ld, al->nbytes = %d\n",
		      numalloced, al->nbytes);
	goto errLabel2;
    }

    /* Remove al from linked list   */
    Wklib_BeginSerialize(memdbg);
    if (al->prev)
	al->prev->next = al->next;
    if (al->next)
	al->next->prev = al->prev;

    /* Stomp on the freed storage to help detect references */
    /* after the storage was freed.			    */
    memset( al, BADVAL, sizeof(*al) + al->nbytes );
    allocCount--;

    free(al);
    Wklib_EndSerialize(memdbg);
    return;

  errLabel2:
    PrintAl( al );
  errLabel:
    fprintf(ferr, "free'd from ");
    FileLine(fil, lin);
    DoAbort( 4 );
    /*NOTREACHED*/
}


void *realloc_debug( void *oldp, unsigned n, char *fil, int lin)
{
    void *p;
    struct s_alloclist *al;

    Init();
    if( !n ) {
	free_debug(oldp,fil,lin);
	p = NULL;
    }
    else if( !oldp )
	p = malloc_debug(n,fil,lin);
    else {
	p = malloc_debug(n,fil,lin);
	if( p ) {
	    al = PTR2AL(oldp);
	    if( al->nbytes < n )
		n = al->nbytes;
	    memcpy(p,oldp,n);
	    free_debug(oldp,fil,lin);
	}
    }
    return p;
}




/*************************************************
 * Debug Versions of higher leveled functions
 *************************************************
 */

char *strdup_debug(const char *s, char *fil, int lin)
{
    char *p;

    p = s ? malloc_debug( strlen(s)+1, fil, lin) : NULL;
    return p ? strcpy(p,s) : p;
}

char *xstrdup_debug(const char *s, char *fil, int lin)
{
    char *p;

    if( !s ) {
	fprintf(ferr, "Internal Error: xstrdup(NULL) ");
	FileLine(fil, lin);
	wklib_exit(4);
    }
    if( !(p = strdup_debug( s, fil, lin ) ) )
	if( !XAllocFailureHook )
	    DoAbort(0);
	else if( !XAllocFailureHook( strlen(s)+1 ) )
	    DoAbort(0);
    return p;
}

void *xmalloc_debug(unsigned n, char *fil, int lin)
{
    char *p;

    while( !(p = malloc_debug( n, fil, lin ) ) )
	if( !XAllocFailureHook )
	    DoAbort(0);
	else if( !XAllocFailureHook( n ) )
	    DoAbort(0);
    return p;
}

void *xcalloc_debug(unsigned n, unsigned m, char *fil, int lin)
{
    char *p;

    if( !(p = calloc_debug( n, m, fil, lin ) ) )
	if( !XAllocFailureHook )
	    DoAbort(0);
	else if( !XAllocFailureHook( n*m ) )
	    DoAbort(0);
    return p;
}

void *xrealloc_debug( void *ptr, unsigned size, char *fil, int lin)
{
    void *p ;

    while( !(p=realloc_debug( ptr, size, fil, lin ) ) )
	if( !XAllocFailureHook )
	    DoAbort(0);
	else if( !XAllocFailureHook( size ) )
	    DoAbort(0);

    return p ;
}


/*******************************************
 * Local Helper Functions
 *******************************************
 */

/***************************
 * Initialize memdbg system()
 */

static void _near
Init()
{
    const char *p;

    if( !initialized ) {
	if( _wklib_memdbgcheck == 1 )
	    Bug("memdbg: xalloc is linked in");
	Wklib_BeginSerialize(memdbg);
	_wklib_memdbgcheck = 2;
      #if NETWARE   /* there are no env-vars in netware */
	if( !atexitOkay )
      #else
	if( !atexitOkay && (p=getenv("DEBUGINFO") ) )
      #endif
	{
	    briefDebugInfo = !stricmp(p,"BRIEF");
	    atexitOkay++ ;
	    atexit( memdbg_reset );
	}
	allocCount = 0;
	numalloced = 0;
	maxalloced = 0;
	alloclist.next = NULL;
	initialized++;
	Wklib_EndSerialize(memdbg);
    }
}

/***********************
 * Issue message: "out of memory" and terminate Program
 * mode == 0  : out of memory ( checked )
 *	   1  : out of memory
 *	   2  : non allocated ptr
 *	   3  : ptr check failed
 *	   4  : free failed
 */

static void _near DoAbort( int mode )
{
    static char *t[] = {
	    "out of memory (in x...())",
	    "out of memory",
	    "non allocated ptr",
	    "checkptr failed",
	    "free failed"
	} ;

    fputs("Fatal error: ",ferr);
    fputs( t[mode] ,ferr);
    putc('\n', ferr);
    fflush(ferr);
    if( !inReset )
	wklib_exit(4);
    return ;
}


/*****************************
 * Print out struct s_alloclist.
 */

static void _near PrintAl( struct s_alloclist *al )
{
    fprintf(ferr, "alloc'd from `%s[%d]' nbytes: %d ptr: %p\n",
	al->file, al->line, al->nbytes, AL2PTR(al) );
}


/*********************************
 * Print out file and line number.
 */

static void _near FileLine( char *fil, int lin)
{
    fprintf(ferr,"File '%s' line %d\n",fil,lin);
    fflush(ferr);
}


/***** bottom of file ******/
