/* [mouse.c wk 25.6.91] Mouse Handler
 *	Copyright (c) 1988-93 by Werner Koch (dd9jn)
 *  This file is part of WkLib.
 *
 *  WkLib is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkLib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 * 27.08.92 wk	Defines um Mousehandling fuer bestimmte Compiler etc.
 *		ausschalten zu koennen (USE_MOUSE)
 *		( MSC v7.0 hat mousesupport nur noch in C++ )
 */

#include <wk/tailor.h>
RCSID("$Id: mouse.c,v 1.8 1995/03/08 16:56:58 wk Exp $")
#define USE_TIMERS 0	/* set to 1 to use dblClick Timers */
#define USE_SIGNAL 0	/* set to 1 to use msm_signal */

#ifdef OS2
  #undef USE_SIGNAL
  #define USE_SIGNAL 0	/* not supported for OS/2 */
#endif

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <wk/lib.h>
#if USE_TIMERS
#include <wk/timer.h>
#endif
#include <wk/mouse.h>
#include <wk/bios.h>
#if __ZTC__
  #include <msmouse.h>
  #define USE_MOUSE 1
#else
  #ifdef USE_MOUSE
    #undef USE_MOUSE
  #endif
  #undef USE_SIGNAL
  #define USE_SIGNAL 0	/* not yet supported for other compilers */
#endif

#if USE_SIGNAL
  #include <int.h>
  #include <bios.h>
#endif


/**** constants ***/
#define M_MOVE		  1   /* event masks for msm_signal */
#define M_LEFT_PRESSED	  2
#define M_LEFT_RELEASED   4
#define M_RIGHT_PRESSED   8
#define M_RIGHT_RELEASED 16

/**** types ******/
/**** globals ****/
static int mouseOkay;	    /* Flag: Eine Mouse ist aktiv */
static ulong dblClickTime;     /*time interval for double click (ms) */

#if USE_MOUSE
static int dc1Timer, dc2Timer; /* handles for double click Timers */
static int mstransX, mstransY; /* factors for coordinate translation */
static void *monitorStack;

static volatile unsigned mouseX, mouseY; /* current mouse position */
					 /* with translated coordinates */
static volatile int leftPressed;    /* left button is down */
static volatile int leftClicked;    /* left button was clicked */
static volatile int leftDblPressed; /* left button was double clicked */
static volatile int rightPressed;   /* same for right button */
static volatile int rightClicked;
static volatile int rightDblPressed;
#endif /* USE_MOUSE */

/*** local Protos ***/
#if USE_SIGNAL
static void MouseMonitor( unsigned, unsigned, unsigned, unsigned);
#endif
#if USE_MOUSE
static void CleanUp( void *);
#endif
/**** Functions ****/

#ifdef DOCUMENTATION
@Summary MouseInitalize
 #include <wk/mouse.h>

 int MouseInitialize(void);
@Description
 Initialisiert das Mouse system, kann beliebig oft aufgerufen werden.
 Alle Mousefunktionen knnen aufgerufen werden, auch wenn keine Mouse
 angeschlossen ist. ScreenInitialize ruft diese Funktion selbst auf.
 Falls die Environmentvariable WKMOUSE auf 0 gesetzt ist, so wird
 die Mouse nicht benutzt.
@Return Value
 Anzahl der MouseButtons, oder 0 wenn keine Mouse aktiv ist.
#endif /*DOCUMENTATION*/

int MouseInitialize()
{
    static int isOkay;
    int i;
  #if USE_MOUSE
    int cols, pages;
    const char *p;
  #endif

    if( !isOkay ) {
	isOkay++;

      #if (__ZTC__ && OS2) || ! USE_MOUSE
	i = 0;
      #else
        if( p = getenv("WKMOUSE") )
	    i = atoi(p) ;
	else
	    i = 1;
      #endif

	if( !i )
	    mouseOkay = 0;
      #if USE_MOUSE
	else if( msm_init() ) {
	    mouseOkay = 2;  /* we support 2 Buttons */
	    AddCleanUp( CleanUp, NULL );
	  #if USE_TIMERS
	    dc1Timer = TimerOpen(0); /* allocate an timer for left button */
	    dc2Timer = TimerOpen(0); /* and one for right button */
	  #endif
	    WKBiosGetVideoMode( &cols, &pages );
	    mstransX = cols==40 ? 16 : 8;
	    mstransY = 8;
	    dblClickTime = 400; /* ms */

	  #if USE_SIGNAL
	    monitorStack = xmalloc( 512 );   /* allocate 512 byte stack */
	    /* 0x1f = move, left and right button events */
	    msm_signal( 0x1f, MouseMonitor, (char*)monitorStack+512 );
	  #endif
	}
	else
	    mouseOkay = 0;
      #endif
    }

    return mouseOkay;
}


void MouseTerminate()
{
  #if USE_MOUSE
    if( mouseOkay )
	msm_term();
  #endif
}



#ifdef DOCUMENTATION
@Summary MouseGet
 #include <wk/mouse.h>

 unsigned MouseGet( int *x, int *y );
@Description
 Gibt die aktuelle Position der Mouse und den Button state zurck.
 Konstanten fr Button state: MOUSE_(LEFT|RIGHT)[DC]
 Ist keine Mouse vorhanden, so wird Button state 0 und position
 (0,0) zurckgegeben.
@Return Value
 Button State
#endif /*DOCUMENTATION*/

unsigned MouseGet( int *x, int *y )
{
    unsigned state;
  #if !USE_SIGNAL && USE_MOUSE
    int btn;
  #endif

    state = 0;
  #if USE_SIGNAL
    *x = (int)mouseX;
    *y = (int)mouseY;
    if( mouseOkay ) {
	if( leftPressed )
	    state |= MOUSE_LEFT;
	if( leftDblPressed )
	    state |= MOUSE_LEFTDBL;
	if( rightPressed )
	    state |= MOUSE_RIGHT;
	if( rightDblPressed )
	    state |= MOUSE_RIGHTDBL;
    }
  #else
   #if USE_MOUSE
    if( mouseOkay ) {
	btn = msm_getstatus( &mouseX, &mouseY );
	mouseX /= mstransX;
	mouseY /= mstransY;
	if( btn & 1 )
	    state |= MOUSE_LEFT;
	if( btn & 2 )
	    state |= MOUSE_RIGHT;
	*x = (int)mouseX;
	*y = (int)mouseY;
    }
   #endif
  #endif
    return state;
}

#ifdef DOCUMENTATION
@Summary MousePut
 #include <wk/mouse.h>

 void MousePut( int *x, int *y );
@Description
 Setzt die aktuelle Position der Mouse.
#endif /*DOCUMENTATION*/

void MousePut( int x, int y )
{
   #if USE_MOUSE
    if( mouseOkay ) {
	mouseX = x * mstransX;
	mouseY = y * mstransY;
	msm_setcurpos( mouseX, mouseY );
    }
  #endif
}


#ifdef DOCUMENTATION
@Summary MouseShow
 #include <wk/mouse.h>

 void MouseShow(void);
@Description
 Bringt den Mouse cursor zur Anzeige. Wird es mehrfach aufgerufen,
 so ist MouseHide() auch mehrfach aufzurufen, um den Cursor wieder
 unsichtbar zu machen
@See Also
 MouseHide
#endif /*DOCUMENTATION*/

void MouseShow()
{
   #ifdef USE_MOUSE
    if( mouseOkay )
	msm_showcursor();
   #endif
}

#ifdef DOCUMENTATION
@Summary MouseHide
 #include <wk/mouse.h>

 void MouseHide(void);
@Description
 Entfernt den Mousecursor vom Bildschirm. Wird es mehrfach aufgerufen,
 so ist MouseShow() auch mehrfach aufzurufen, um den Cursor wieder
 sichtbar zu machen.
@See Also
 MouseShow MouseHideCond
#endif /*DOCUMENTATION*/

void MouseHide()
{
   #ifdef USE_MOUSE
    if( mouseOkay )
	msm_hidecursor();
  #endif
}


#ifdef DOCUMENTATION
@Summary MouseHideCond
 #include <wk/mouse.h>

 void MouseHideCond(void);
@Description
 Wie MouseHide() aber der Cursor wird nur entfernt wenn er innerhalb
 der angegebenen Grenzen liegt.
 Ein entsprechendes ShowCursor ist in jedem Fall notwendig
@See Also
 MouseShow MouseHideCond
#endif /*DOCUMENTATION*/

void MouseHideCond( int x1, int y1, int x2, int y2 )
{
   #ifdef USE_MOUSE
    if( mouseOkay )
	msm_condoff( x1 * mstransX, y1 * mstransY,
		     x2 * mstransX, y2 * mstransY );
  #endif
}


#ifdef DOCUMENTATION
@Summary MouseSpeed
 #include <wk/mouse.h>

 long MouseSpeed( int mode, long value );
@Description
 Setzt verschieden Mouse Parameter:
 z.Z. ist definiert:
 mode 1: Setzt die Zeitspanne fr einen DoubleClick
	 value ist in Millisekunden anzugeben; fall value -1L ist
	 wird der Wert nicht gendert.
@Return Value
 Den entsprechenden ursprnglichen Wert.
#endif /*DOCUMENTATION*/

long MouseSpeed( int mode, long value )
{
    long old;

    if( mode == 1 ) {
	old = dblClickTime;
	if( value >= 0 ) {
	  #if USE_SIGNAL
	    int_off();
	  #endif
	    dblClickTime = value;
	  #if USE_SIGNAL
	    int_on();
	  #endif
	}
    }
    else
	old = 0;

    return old ;
}



/*
 * This function is called as an interrupthandler for Mouse events
 * REMEMBER: This is an interrupt Function within the MSDOS Environment
 * mask: Kind of event: Bit 0 = mouse moved
 *			    1 = left button pressed
 *			    2 = left button released
 *			    3 = right button pressed
 *			    4 = right button released
 * state: is button event: Bit 0 = left button changed
 *			       1 = right button changed
 * curposx, curposy: current mouse position
 */

#if USE_SIGNAL
static void MouseMonitor( unsigned mask, unsigned state,
			  unsigned curposx, unsigned curposy )
{
    static unsigned leftX, leftY;
    static unsigned rightX, rightY;

    mouseX = curposx / mstransX;
    mouseY = curposy / mstransY;

    /* process left button events */
    if( mask & M_LEFT_PRESSED ) {
	leftPressed = 1;
      #if USE_TIMERS
	if( leftClicked ) {
	    if( TimerQuery( dc1Timer ) && leftX == mouseX && leftY == mouseY )
		leftDblPressed = 1;
	    else
		TimerStart( dc1Timer, dblClickTime );
	}
	else
	    TimerStart( dc1Timer, dblClickTime );
      #endif
	leftX = mouseX;
	leftY = mouseY;
	leftClicked = 0;
    }
    if( mask & M_LEFT_RELEASED ) {
	if( leftPressed )
	    leftClicked = 1;
	leftPressed = leftDblPressed = 0;
    }

    /* process right button events */
    if( mask & M_RIGHT_PRESSED ) {
	rightPressed = 1;
      #if USE_TIMERS
	if( rightClicked ) {
	    if( TimerQuery( dc2Timer ) && rightX == mouseX && rightY == mouseY )
		rightDblPressed = 1;
	    else
		TimerStart( dc2Timer, dblClickTime );
	}
	else
	    TimerStart( dc2Timer, dblClickTime );
      #endif
	rightX = mouseX;
	rightY = mouseY;
	rightClicked = 0;
    }
    if( mask & M_RIGHT_RELEASED ) {
	if( rightPressed )
	    rightClicked = 1;
	rightPressed = rightDblPressed = 0;
    }

}
#endif

#if USE_MOUSE
static void CleanUp( void *data )
{
   #ifdef USE_MOUSE
    msm_term();
    mouseOkay=0;
  #if USE_SIGNAL
    FREE( monitorStack );
  #endif
  #endif
}
#endif

#ifdef TEST /****** test suite *********/

#include <wk/keys.h>

void main( int argc, char **argv )
{
    unsigned state;
    int x,y;

    MouseInitialize();
    MouseShow();
    while( Inkey() != K_ESCAPE ) {
	state = MouseGet( &x, &y );
        printf( "\r%3d %3d L=%c%c R=%c%c",
		 x, y,
                 state & MOUSE_LEFT   ? 'P':' ',
                 state & MOUSE_LEFTDBL? 'D':' ',
                 state & MOUSE_RIGHT  ? 'P':' ',
                 state & MOUSE_RIGHTDBL? 'D':' '
				  );

	fflush(stdout);
    }

    exit(0);
}
#endif /* TEST */
/*** bottom of File ***/
