/* [nmpipe.c wk 10.5.92] Named Pipes Interface Layer
 *	Copyright (c) 1992 by Werner Koch (dd9jn)
 *  This file is part of WkLib.
 *
 *  WkLib is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkLib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 * 17.12.92 wk	DOS Client-Support
 * 18.01.94 wk	Moved from COMLIB to WkLib; support for OS/2 2.0
 * 11.03.94 wk	Added NmPipeDrain()
 * 14.03.94 wk	Added NmPipeCall()
 */

#include <wk/tailor.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <wk/nmpipe.h>
#include <wk/string.h>

#if OS2
  #define INCL_ERRORS 1
  #define INCL_DOSNMPIPES 1
  #include <os2.h>
  #if OS20
    #define DosWaitNmPipe	DosWaitNPipe
    #define DosMakeNmPipe	DosCreateNPipe
    #define DosConnectNmPipe	DosConnectNPipe
    #define DosSetNmPHandState	DosSetNPHState
    #define DosDisConnectNmPipe DosDisConnectNPipe
  #endif
#elif MSDOS
  #include <dos.h>
  #include <io.h>
  #include <fcntl.h>
  #include <share.h>
  #include <errno.h>
#elif POSIX
  #include <unistd.h>
#endif

/***** constants *******/
#define MAX_HANDLES 10	 /* max. is 10 pipes */
/***** types ***********/
/**** global vars ******/
static struct { 	/* to keep track about all created pipes */
	char *name;	/* name of pipe ( as specified by NmPipeCreate ) */
			/* or NULL to indicate this as an empty slot */
	int   handle;	/* handle of the pipe */
	char *server;	/* NULL for server, or name of requested server */
			/* if used by a client */
    } hdTable[MAX_HANDLES];

/****** protos *********/
#if OS2 || MSDOS || NETWARE
static int DosErr( ushort n );
#endif
/***** functions *******/

#ifdef DOCUMENTATION
@Summary NmPipeCreate
 #include <wk/nmpipe.h>

 int NmPipeCreate( retHd, name, mode, outSize, inSize, noInst );
 int *retHd;		Returns: Handle of created pipe
 const char *name;	Name of pipe (without server, pipe prefix is optional)
 const char *mode;	Some flags
 ushort outSize;	# of bytes to reserve or output buffer
 ushort inSize; 	# of bytes to reserve or input buffer
 ushort noInst; 	# of instances for the pipe ( 0 = unlimited )
@Description
 This is a server call to create a pipe.
 The name is a standard filename, you may use forward slashes, of course
 there is no need to specify the server name, for we are the creator.
 You may use 0 for the buffersizes, to select the default values.
 Mode flags are:
 1. Character in the string must be:
    'd'  Create a duplex pipe
    'i'  Create an inbound pipe (from client to server)
    'o'  Create an outbound pipe (from server to client)
 The next characters may be given in any order:
    "bb" byte stream pipe / readmode byte stream ( default ) , or
    "bm" byte stream pipe / readmode message stream, or
    "mb" message stream pipe / readmode byte stream, or
    "mm" message stream pipe / readmode message stream
    'w'  Wait mode (reading and writing will wait until data is available)
	 (Connect also blocks)
@Return Value
 0 = Okay or Errorcode E_NMPIPE_xxxxx
@See Also
 NmPipeClose NmPipeConnect
@Notes
 Also OS/2 uses an ushort we will use an int for the handle, there will
 be no problems, because we do not mess with handles. The pipe name will be
 stored in an internal buffer.
#endif /*DOCUMENTATION*/

int
NmPipeCreate( int *retHd,
	      const char *name,
	      const char *mode,
	      ushort outSize,
	      ushort inSize,
	      ushort noInst )
{
  #if OS2
    char *dosName, *p;
    int err, i;
    USHORT fsOpenMode, fsPipeMode, doserr;
    HPIPE dosHd;

    err = 0;
    dosName = xmalloc( 6 + strlen(name) + 1 );
    /* Make an OS conforming name */
    if( strlen(name) > 6 && (*name == '/'|| *name=='\\') &&
			    (name[5]=='/'|| name[5]=='\\') )
	name += 6;

    strcpy( dosName, "\\pipe\\" );
    p = dosName + 6;
    for( i = *name == '/' || *name == '\\'? 1:0; name[i]; i++ )
	*p++ = name[i] == '/' ? '\\' : name[i];
    *p = '\0';
    /* analyse flags */
    switch( *mode ) {
      case 'd': fsOpenMode = NP_ACCESS_DUPLEX; break;
      case 'i': fsOpenMode = NP_ACCESS_INBOUND; break;
      case 'o': fsOpenMode = NP_ACCESS_OUTBOUND; break;
      default: err = E_NMPIPE_INVARG; goto retLabel;
    }
    fsOpenMode |= NP_INHERIT | NP_WRITEBEHIND;
    mode++;
    fsPipeMode = strchr(mode, 'w')? NP_WAIT : NP_NOWAIT;
    if( p = strchr(mode, 'b') )
	fsPipeMode |= NP_TYPE_BYTE;
    else if( p = strchr(mode, 'm') )
	fsPipeMode |= NP_TYPE_MESSAGE;
    if( p )
	fsPipeMode |= p[1]=='m'? NP_READMODE_MESSAGE: NP_READMODE_BYTE;
    if( !noInst )
	fsPipeMode |= NP_UNLIMITED_INSTANCES;
    else if( noInst < 255 )
	fsPipeMode |= noInst & 0xff;
    else {
	err = E_NMPIPE_INVARG;
	goto retLabel;
    }
    if( !inSize )
	inSize = 512;
    if( !outSize )
	outSize = 512;

    switch( doserr = DosMakeNmPipe( (PSZ)dosName, &dosHd,
				    fsOpenMode, fsPipeMode,
			   outSize, inSize, 1000 /* timeout = 1s */) ) {
      case 0: /* pipe is created */
	*retHd = dosHd;     /* return the handle */
	for(i=0; i < MAX_HANDLES; i++ )
	    if( !hdTable[i].name )
		break;
	if( i >= MAX_HANDLES )
	    Bug("NmPipeCreate(): Handle Table full");
	hdTable[i].name = xstrdup( name );
	hdTable[i].handle = *retHd;
	hdTable[i].server = NULL;
	break;
      case ERROR_INVALID_PARAMETER: err = E_NMPIPE_INVARG; break;
      case ERROR_NOT_ENOUGH_MEMORY: /* fall thru */
      case ERROR_OUT_OF_STRUCTURES: err = E_NMPIPE_NOMEM; break;
      case ERROR_PATH_NOT_FOUND:    err = E_NMPIPE_NOPATH; break;
      case ERROR_PIPE_BUSY:	    err = E_NMPIPE_BUSY; break;
      default: err = DosErr( doserr ); break;
    }

  retLabel:
    free( dosName );
    return err;
  #else
    return E_NMPIPE_NOTAVL;
  #endif
}


#ifdef DOCUMENTATION
@Summary NmPipeOpen
 #include <wk/nmpipe.h>

 int NmPipeOpen( retHd, server, name, timeout );
 int *retHd;		Returns: Handle of pipe
 const char *server;	Name of requested server or NULL to take it from
			the name.
 const char *name;	Name of pipe (without server, pipe prefix is optional)
 const char *mode;	character string:
			First character must be:
			'd'  duplex
			'i'  inbound (client receives data)
			'o'  outbound (client sends data)
			The next characters are optional:
			'b'  read in byte mode
			'm'  read in message mode
			'w'  Wait mode
			(reading and writing will wait until data is available)
 ulong timeout; 	timeout value in milliseconds
			( 0 = no wait, 1 = use a default value )
@Description
 This is a client call to open a pipe.
 The servername may start with the usual backslashes, but there is
 no need for them, the server name will stop at the first slash or backslash.
 If server is an empty string this is a local pipe; NULL is a reserved value,
 which will be used to find a default server.
 Default timeout value is 5 seconds
@Return Value
 0 = Okay or Errorcode E_NMPIPE_xxxxx
@See Also
 NmPipeClose
#endif /*DOCUMENTATION*/

int
NmPipeOpen( int *retHd,
	    const char *server,
	    const char *name,
	    const char *mode,
	    ulong timeout )
{
    char *dosName, *p;
    int err, i, noserver;
  #if OS2
    int rpt;
    ushort fsOpenMode, fsPipeMode;
    #if OS20
      ulong doserr;
      ulong dosHd,usAction;
    #else
      ushort doserr;
      ushort dosHd,usAction;
    #endif
  #elif MSDOS
    int rpt;
    int oflag, doserr, dosHd;
  #endif

  #if !(OS2 || MSDOS)
    return E_NMPIPE_NOTAVL;
  #endif

    err = 0;
    if( noserver = server == NULL )
	server = "";
    dosName = xmalloc(2 + strlen(server) + 6 + strlen(name) + 1 );
    /* Make an OS conforming name */
    p = dosName;
    if( !noserver ) {
	if( *server ) {
	    strcpy( dosName, "\\\\" );
	    p += 2;
	    for( i = 0; server[i]; i++ )
		if( server[i] == '\\' || server[i] == '/' ) {
		    if( i > 1 )
			break; /* end of servername */
		}
		else
		    *p++ = server[i];
	}
	if( strlen(name) > 6 && (*name == '/'|| *name=='\\') &&
				(name[5]=='/'|| name[5]=='\\') )
	    name += 6;
	strcpy( p, "\\pipe\\" );
	p += 6;
	for( i = *name == '/' || *name == '\\'? 1:0; name[i]; i++ )
	    *p++ = name[i] == '/' ? '\\' : name[i];
    }
    else {
	for( i = 0; name[i]; i++ )
	    *p++ = name[i] == '/' ? '\\' : name[i];
    }
    *p = '\0';

    if( timeout == 1 )
	timeout = 5000; /* 5 seconds is a good default value */
    switch( *mode ) {
      #if OS2
      case 'd': fsOpenMode = OPEN_ACCESS_READWRITE; break;
      case 'i': fsOpenMode = OPEN_ACCESS_READONLY; break;
      case 'o': fsOpenMode = OPEN_ACCESS_WRITEONLY; break;
      #elif MSDOS
      case 'd': oflag = O_RDWR; break;
      case 'i': oflag = O_RDONLY; break;
      case 'o': oflag = O_WRONLY; break;
      #endif
      default: err = E_NMPIPE_INVARG; goto retLabel;
    }
  #if OS2
    fsOpenMode |= OPEN_SHARE_DENYNONE;
    fsPipeMode = strchr(mode, 'w')? NP_WAIT : NP_NOWAIT;
    if( p = strchr(mode, 'b') )
	fsPipeMode |= NP_READMODE_BYTE;
    else if( p = strchr(mode, 'm') )
	fsPipeMode |= NP_READMODE_MESSAGE;

    rpt = 0;
    do {  /* loop will execute 1 or 2 times */
	doserr = DosOpen( (PSZ)dosName, &dosHd, &usAction, 0, 0,
				 FILE_OPEN, fsOpenMode, 0 );
	switch( doserr ) {
	  case 0:
	    *retHd = dosHd;	/* return the handle */
	    for(i=0; i < MAX_HANDLES; i++ )
		if( !hdTable[i].name )
		    break;
	    if( i >= MAX_HANDLES )
		Bug("NmPipeOpen(): Handle Table full");
	    hdTable[i].name = xstrdup( name );
	    hdTable[i].handle = *retHd;
	    if( noserver ) {
		int j;
		for(j=0; name[j]; j++ )
		    if( j > 1 && (name[j] == '/' || name[j] == '\\'))
			break;
		hdTable[i].server = xmalloc(j+1);
		mem2str(hdTable[i].server, name, j );
	    }
	    else
		hdTable[i].server = xstrdup(server);
	    break;
	  case ERROR_PIPE_BUSY:
	    if( !rpt && timeout ) {
	      #if 1
		switch( doserr = DosWaitNmPipe( (PSZ)dosName, timeout ) ) {
		  case 0: rpt++; break; /* try again */
		  case ERROR_BAD_PIPE:	err = E_NMPIPE_BADPIPE; break;
		  case ERROR_SEM_TIMEOUT:  err = E_NMPIPE_TIMEOUT; break;
		  default: err = DosErr( doserr ); break;
		}
	      #else /* falls es mal nicht funktioniert */
		if( timeout < 50 )
		    err = E_NMPIPE_TIMEOUT;
		else {
		    DosSleep( (ulong)30 );
		    rpt++;
		    timeout -= 30;
		}
	      #endif
	    }
	    else
		err = E_NMPIPE_BUSY; break;
	    break;
	  case ERROR_PATH_NOT_FOUND:	/* fall thru */
	  case ERROR_FILE_NOT_FOUND:	err = E_NMPIPE_NOPATH; break;
	  case ERROR_INVALID_PARAMETER: err = E_NMPIPE_INVARG; break;
	  case ERROR_ACCESS_DENIED:	err = E_NMPIPE_NOACC;  break;
	  case ERROR_PIPE_NOT_CONNECTED:err = E_NMPIPE_NOCON;  break;
	  case ERROR_INTERRUPT: 	err = E_NMPIPE_INTR;   break;
	  default: err = DosErr( doserr ); break;
	}
    } while( rpt );
    if( !err )
	if( doserr = DosSetNmPHandState( dosHd, fsPipeMode ) )
	    if( err = DosErr( doserr ) )
		NmPipeClose( *retHd );
  #elif MSDOS
    /*dosHd = sopen( dosName, oflag, SH_DENYNO );*/
    #if __WATCOMC__
    if( _dos_open( dosName, oflag|SH_DENYNO, &dosHd ) )
	dosHd = -1;
    #else
    dosHd = dos_open( dosName, oflag|SH_DENYNO );
    #endif
    if( dosHd == -1 ) {
	switch( errno ) {
	  case ENOENT:
	  case ENOTDIR: err = E_NMPIPE_NOPATH; break;
	  case EACCES:	err = E_NMPIPE_BUSY;   break;
	  default: err = DosErr( errno ); break;
	}
    }
    else {
	*retHd = dosHd;     /* return the handle */
	for(i=0; i < MAX_HANDLES; i++ )
	    if( !hdTable[i].name )
		break;
	if( i >= MAX_HANDLES )
	    Bug("NmPipeOpen(): Handle Table full");
	hdTable[i].name = xstrdup( name );
	hdTable[i].handle = *retHd;
	if( noserver ) {
	    int j;
	    for(j=0; name[j]; j++ )
		if( j > 1 && (name[j] == '/' || name[j] == '\\'))
		    break;
	    hdTable[i].server = xmalloc(j+1);
	    mem2str(hdTable[i].server, name, j );
	}
	else
	    hdTable[i].server = xstrdup(server);
    }
  #endif

  retLabel:
    free( dosName );
    return err;
}



#ifdef DOCUMENTATION
@Summary NmPipeClose
 #include <wk/nmpipe.h>

 int NmPipeClose( hd );
 int hd;    /* handle of pipe to close */
@Description
 This call is used to close (at the client side) or to destroy (at
 the server side) the pipe.
 Function does not return any error, but issues an Fatal Error Message
 instead. ( using function Fatal() )
@See Also
 NmPipeCreate
#endif /*DOCUMENTATION*/


int
NmPipeClose( int hd )
{
    int i;
  #if OS2 || MSDOS
    ushort ret;
  #endif

  #if !(OS2 || MSDOS)
    return E_NMPIPE_NOTAVL;
  #endif

    for(i=0; i < MAX_HANDLES; i++ ) {
	if( hdTable[i].name && hdTable[i].handle == hd ) {
	    FREE(hdTable[i].name);
	    FREE(hdTable[i].server);
	  #if OS2
	    if( ret = DosClose( hd ) )
		Fatal("NmPipeDestroy: Hd %#x DOS Error %#x", hd, ret);
	  #elif UNIX
	    /* nothing */
	  #else
	  #if NETWARE
	    if( close(hd) )
	  #elif __WATCOMC__
	    if( _dos_close(hd) )
	  #else
	    if( close(hd) )
	  #endif
		Fatal("NmPipeDestroy: Hd %#x DOS Error %#x", hd, errno);
	  #endif
	    return 0;
	}
    }
    Fatal("NmPipeClose: Hd %#x not in Handle Table", hd );
    return 0;
}


#ifdef DOCUMENTATION
@Summary NmPipeConnect
 #include <wk/nmpipe.h>

 int NmPipeConnect( hd );
 int hd;    handle of pipe to connect
@Description
 This is the server call to connect to a pipe.
 If a client end of the pipe is open, this function returns immediately.
 If there is no open client end, this function will block, if it was created
 with wait mode, or return immediately errorcode E_NMPIPE_NOCON.
 Errorcode ...BROKEN will occur if the client had closed the pipe, but
 the pipe has not been disconnected by the server.
@Return Value
 0 = Okay or Errorcode E_NMPIPE_xxxxx
@See Also
 NmPipeDisConnect
#endif /*DOCUMENTATION*/

int
NmPipeConnect( int hd )
{
  #if OS2
    USHORT doserr;

    switch( doserr = DosConnectNmPipe( (USHORT)hd ) ) {
      case 0: return 0;
      case ERROR_PIPE_NOT_CONNECTED: return E_NMPIPE_NOCON;
      case ERROR_BAD_PIPE:	     return E_NMPIPE_BADPIPE;
      case ERROR_BROKEN_PIPE:	     return E_NMPIPE_BROKEN;
      case ERROR_INVALID_FUNCTION:   return E_NMPIPE_INVFNC;
      case ERROR_INTERRUPT:	     return E_NMPIPE_INTR;
      default: return DosErr( doserr ); break;
    }
  #else
    return E_NMPIPE_NOTAVL;
  #endif
}


#ifdef DOCUMENTATION
@Summary NmPipeDisConnect
 #include <wk/nmpipe.h>

 int NmPipeDisConnect( hd );
 int hd;    handle of pipe to connect
@Description
 This is the server call to disconnect from a pipe.
 If a client end of the pipe is open, it is forced closed; But the client
 still has to close it at his side, all other operation at the client
 side will return an error. Pending data may be discarded.
@Return Value
 0 = Okay or Errorcode E_NMPIPE_xxxxx
@See Also
 NmPipeConnect
#endif /*DOCUMENTATION*/

int
NmPipeDisConnect( int hd )
{
  #if OS2
    USHORT doserr;
    switch( doserr = DosDisConnectNmPipe( (USHORT)hd ) ) {
      case 0: return 0;
      case ERROR_BAD_PIPE:	     return E_NMPIPE_BADPIPE;
      case ERROR_INVALID_FUNCTION:   return E_NMPIPE_INVFNC;
      case ERROR_PIPE_NOT_CONNECTED: return E_NMPIPE_NOCON;
      default: return DosErr( doserr ); break;
    }
  #else
    return E_NMPIPE_NOTAVL;
  #endif
}


#ifdef DOCUMENTATION
@Summary NmPipeRead
 #include <wk/nmpipe.h>

 int NmPipeRead( hd, buffer, size, nread );
 int hd;	    Handle of pipe
 void *buffer;	    Buffer to receive data
 ushort size;	    length of buffer
 ushort *nread;     returns: n of bytes read;
@Description
 This call reads from a pipe. When reading a message mode pipe in message
 mode, you may get the errorcode E_NMPIPE_MORE, indicating that there
 is more data of this message available.
@Return Value
 0 = Okay or Errorcode E_NMPIPE_xxxxx
@See Also
 NmPipeWrite
#endif /*DOCUMENTATION*/

int
NmPipeRead( int hd, void *buffer, ushort size, ushort *nread )
{
  #if OS2
    unsigned doserr, dosSize;
    #if OS20	/* C Set/2 complains about: ulong != unsigned int, ;-) */
      ulong dosNread;
    #else
      ushort dosNread;
    #endif
    dosSize = size;
    doserr = DosRead( hd, buffer, dosSize, &dosNread );
    *nread = dosNread;
    switch( doserr ) {
      case 0: return 0;
      case ERROR_NO_DATA:  *nread = 0; return 0;
      case ERROR_MORE_DATA:	     return E_NMPIPE_MORE;
      case ERROR_BAD_PIPE:	     return E_NMPIPE_BADPIPE;
      case ERROR_BROKEN_PIPE:	     return E_NMPIPE_BROKEN;
      case ERROR_ACCESS_DENIED:      return E_NMPIPE_NOACC;
      case ERROR_PIPE_NOT_CONNECTED: return E_NMPIPE_NOCON;
      case ERROR_INTERRUPT:	     return E_NMPIPE_INTR;
      default: return DosErr( doserr ); break;
    }
  #elif MSDOS
  #if __WATCOMC__
    unsigned nbytes;
    if( _dos_read( hd, buffer, size, &nbytes) ) {
	switch( errno ) {
	  case EACCES:	return E_NMPIPE_NOACC;
	  default:	return DosErr( errno );
	}
    }
  #else
    int nbytes;
    nbytes = read( hd, buffer, size );
    if( nbytes < 0 ) {
	switch( errno ) {
	  case EACCES:	return E_NMPIPE_NOACC;
	  default:	return DosErr( errno );
	}
    }
  #endif
    else {
	*nread = nbytes;
	return 0;
    }
  #else
    return E_NMPIPE_NOTAVL;
  #endif
}


#ifdef DOCUMENTATION
@Summary NmPipeWrite
 #include <wk/nmpipe.h>

 int NmPipeWrite( hd, buffer, nbytes );
 int hd;	    Handle of pipe
 void *buffer;	    Buffer with data
 ushort nbytes	    # of bytes to write
@Description
 This call writes to a pipe.
@Return Value
 0 = Okay or Errorcode E_NMPIPE_xxxxx
@See Also
 NmPipeDrain NmPipeRead
#endif /*DOCUMENTATION*/

int
NmPipeWrite( int hd, void *buffer, ushort nbytes )
{
  #if OS2
    #if OS20	/* C Set/2 complains about: ulong != unsigned int, ;-) */
      ulong nwritten, doserr;
      ulong dosnbytes;
    #else
      ushort nwritten, doserr;
      ushort dosnbytes;
    #endif
    dosnbytes = nbytes;
    doserr = DosWrite( hd, buffer, dosnbytes, &nwritten );
    switch( doserr ) {
      case 0: return  nwritten == dosnbytes? 0 : E_NMPIPE_WFAULT;
      case ERROR_WRITE_FAULT:	     return E_NMPIPE_WFAULT;
      case ERROR_BAD_PIPE:	     return E_NMPIPE_BADPIPE;
      case ERROR_BROKEN_PIPE:	     return E_NMPIPE_BROKEN;
      case ERROR_ACCESS_DENIED:      return E_NMPIPE_NOACC;
      case ERROR_PIPE_NOT_CONNECTED: return E_NMPIPE_NOCON;
      case ERROR_INTERRUPT:	     return E_NMPIPE_INTR;
      default: return DosErr( doserr ); break;
    }
  #elif MSDOS
  #if __WATCOMC__
    unsigned nwritten;
    if( _dos_write( hd, buffer, nbytes, &nwritten) || nbytes != nwritten ) {
	switch( errno ) {
	  case EACCES:	return E_NMPIPE_NOACC;
	  default:	return DosErr( errno );
	}
    }
  #else
    int nwritten;
    nwritten = write( hd, buffer, nbytes );
    if( nwritten != nbytes ) {
	switch( errno ) {
	  case EACCES:	return E_NMPIPE_NOACC;
	  default:	return DosErr( errno );
	}
    }
  #endif
    else {
	return 0;
    }
  #else
    return E_NMPIPE_NOTAVL;
  #endif
}


#ifdef DOCUMENTATION
@Summary NmPipeDrain
 #include <wk/nmpipe.h>

 int NmPipeDrain( hd );
 int hd;	    Handle of pipe
@Description
 This call waits until all written data has been sent to the other
 end (read by the consumer).
@Return Value
 0 = Okay or Errorcode E_NMPIPE_xxxxx
@See Also
 NmPipeWrite
#endif /*DOCUMENTATION*/

int
NmPipeDrain( int hd )
{
  #if OS2
    #if OS20	/* C Set/2 complains about: ulong != unsigned int, ;-) */
      ulong doserr;
      doserr = DosResetBuffer( hd );
    #else
      ushort doserr;
      doserr = DosBufReset( hd );
    #endif
    switch( doserr ) {
      case 0: return 0;
      case ERROR_WRITE_FAULT:	     return E_NMPIPE_WFAULT;
      case ERROR_BAD_PIPE:	     return E_NMPIPE_BADPIPE;
	/* we will return no error when the pipe is broken, assuming
	 * that this function will be used prior to an DisConnect to
	 * assure that all bytes have been sent to the client; in other
	 * cases the next Write() will give the broken-pipe-error
	 */
      case ERROR_BROKEN_PIPE:	     return 0;
      case ERROR_ACCESS_DENIED:      return E_NMPIPE_NOACC;
      case ERROR_PIPE_NOT_CONNECTED: return E_NMPIPE_NOCON;
      case ERROR_INTERRUPT:	     return E_NMPIPE_INTR;
      default: return DosErr( doserr ); break;
    }
  #elif MSDOS
    return 0;	/* what is the name of the DosCommit call in the C-library ?*/
  #else
    return E_NMPIPE_NOTAVL;
  #endif
}


#ifdef DOCUMENTATION
@Summary NmPipeCall
 #include <wk/nmpipe.h>

 int NmPipeCall( const char *server, const char *pipename,
		 void *requestBuffer, ushort requestBufferLen,
		 void *replyBuffer, ushort replyBufferLen,
		 ushort *bytesReplied, ulong timeout );
@Description
 This call performs a call on a named pipe; that is a sequence of
 Open, Write, Read and Close and is intended to make the live of a client
 more easy.
 timeout 0 means: do not wait, timeout 1 means: use a default timeout value
 all other values are timeout values in milliseconds.
@Return Value
 0 = Okay or Errorcode E_NMPIPE_xxxxx
@See Also
 NmPipeOpen NmPipeWrite NmPipeRead
#endif /*DOCUMENTATION*/


int
NmPipeCall( const char *server, const char *pipename,
	    void *requestBuffer, ushort requestBufferLen,
	    void *replyBuffer, ushort replyBufferLen,
	    ushort *bytesReplied, ulong timeout )
{
    int hd, err;

    if( !(err = NmPipeOpen( &hd, server, pipename, "dmw", timeout )) ) {
	if( !(err = NmPipeWrite(hd, requestBuffer, requestBufferLen )) )
	    err=NmPipeRead(hd,replyBuffer,replyBufferLen,bytesReplied);
	NmPipeClose(hd);
    }
    return err;
}



const char *
NmPipeStrError( int n )
{
    const char *p;
    static char buffer[16];

    switch(n) {
      case E_NMPIPE_GENERAL : p = "general error";          break;
      case E_NMPIPE_NOMEM   : p = "out of memory";          break;
      case E_NMPIPE_INVARG  : p = "invalid argument";       break;
      case E_NMPIPE_BUSY    : p = "pipe is busy";           break;
      case E_NMPIPE_BROKEN  : p = "pipe is broken";         break;
      case E_NMPIPE_NOPATH  : p = "pipe not found";         break;
      case E_NMPIPE_NOCON   : p = "not connected";          break;
      case E_NMPIPE_BADPIPE : p = "pipe is bad";            break;
      case E_NMPIPE_INVFNC  : p = "invalid function";       break;
      case E_NMPIPE_TIMEOUT : p = "timeout";                break;
      case E_NMPIPE_NOACC   : p = "access denied";          break;
      case E_NMPIPE_MORE    : p = "more data available";    break;
      case E_NMPIPE_WFAULT  : p = "write fault";            break;
      case E_NMPIPE_NOTAVL  : p = "pipe support not available"; break;
      case E_NMPIPE_INTR    : p = "operation was interrupted"; break;
      default: sprintf( buffer, "Error %d", n ); p = buffer; break;
    }
    return p;
}

#if OS2 || MSDOS || NETWARE
static int DosErr( ushort n )
{
    Error(0,"DOS returns errorcode %u", n);
    return E_NMPIPE_GENERAL;
}
#endif

/**** bottom of file ****/
