/* [strg10.c wk 16.01.95] Base 64 conversions
 *	Copyright (c) 1988-94 by Werner Koch (dd9jn)
 *  This file is part of WkLib.
 *
 *  WkLib is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkLib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 * 24.01.95 wk	changed base-64 coding to reflect RFC-1113 coding
 *		(I'm lucky that (nearly) no software, using the old
 *		 coding, is out of the door)
 */

#include <wk/tailor.h>
RCSID("$Id: strg10.c,v 1.6 1996/01/10 19:06:03 wernerk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <wk/string.h>


static byte bintoasc[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
			 "abcdefghijklmnopqrstuvwxyz"
			 "0123456789+/";
static byte  isInitialized = 0;
static void  Init(void);
static byte  asctobin[128]; /* runtime initialized */


#ifdef DOCUMENTATION
@Summary ConvertToBase64
 #include <wk/string.h>

 size_t ConvertToBase64( buffer, source, sourcelength );
 char *buffer;		  Buffer to receive the base-64 string
 void *source;		  Binary data to be converted
 size_t sourcelength;	  Size of the binary data (bytes)
@Description
 This function converts binary data to a string, using a base-64
 representation. The used characterset consists of all uppercase
 characters, the dot, all lowercase characters, the underscore and
 all digits. If NULL is specified for buffer, the function will only
 calculate the needed size for the buffer.
 The coding rules are interchangeable with those set up in RFC-1113,
 but there is no need to use a multiple of 3 bytes.
@Return Value
 Length of the returned string.
@See Also
 ConvertFromBase64 CheckBase64String
#endif /*DOCUMENTATION*/


size_t
ConvertToBase64( char *buffer, const void *src, size_t srclen )
{
    byte *d;
    const byte *s;
    size_t n;

    if( !buffer ) {
	n = srclen / 3 * 4;
	srclen %= 3;
	if( srclen )
	    n += srclen + 1;
	return n + 1;
    }

    d = (byte*)buffer;
    s = src;
    for(; srclen > 2; srclen -= 3, s += 3 ) {
	*d++ = bintoasc[(*s >> 2) & 077];
	*d++ = bintoasc[(((*s << 4) & 060) | ((s[1] >> 4) & 017)) & 077];
	*d++ = bintoasc[(((s[1] << 2) & 074) | ((s[2] >> 6) & 03)) & 077];
	*d++ = bintoasc[s[2] & 077];
    }
    if( srclen ) {
	*d++ = bintoasc[(*s >> 2) & 077];
	if( !--srclen )
	    *d++ = bintoasc[((*s << 4) & 060) & 077];
	else {
	    *d++ = bintoasc[(((*s << 4) & 060) | ((s[1] >> 4) & 017)) & 077];
	    *d++ = bintoasc[((s[1] << 2) & 074) & 077];
	}
    }
    *d = 0;
    return d - (byte*)buffer;
}



#ifdef DOCUMENTATION
@Summary ConvertFromBase64
 #include <wk/string.h>

 size_t ConvertFromBase64( buffer, source);
 char *buffer;		  Buffer to receive the binary data
 void *source;		  String with the base-64 representation
@Description
 This function converts a base-64 string back to its binary representation.
 Invalid characters (e.g. spaces) are simply skipped. You may use NULL for
 buffer to get the required length of buffer.
 The coding rules are interchangeable with those set up in RFC-1113;
 the padding character ('=') is valid, but has no effect.
@Return Value
 Length of the buffer.
@See Also
 ConvertToBase64 CheckBase64String
#endif /*DOCUMENTATION*/


size_t
ConvertFromBase64( void *buffer, const char *string )
{
    byte *d;
    const byte *s;
    int i;
    unsigned c, reg;

    if( !isInitialized )
	Init();

    d = buffer;
    i = 0;
    if( !d ) { /* count only */
	for(s = (const byte*)string; *s; s++ ) {
	    if( *s >= 128 || (c = asctobin[*s]) >= 64 )
		continue; /* skip invalid characters */
	    switch(i) {
	      case 0:	   i = 1; break;
	      case 1: d++; i = 2; break;
	      case 2: d++; i = 3; break;
	      case 3: d++; i = 0; break;
	    }
	}
	if( i==1 )
	    d++;
    }
    else {
	reg = 0; /* avoid compiler warning */
	for(s = (const byte*)string; *s; s++ ) {
	    if( *s >= 128 || (c = asctobin[*s]) >= 64 )
		continue; /* skip invalid characters */
	    switch(i) {
	      case 0:
		reg = c << 2;
		i = 1; break;
	      case 1:
		reg |= (c >> 4) & 0x03; *d++ = reg; reg = (c << 4) & 0xf0;
		i = 2; break;
	      case 2:
		reg |= (c >> 2) & 0x0f; *d++ = reg; reg = (c << 6) & 0xc0;
		i = 3; break;
	      case 3:
		reg |= c & 0x3f; *d++ = reg;
		i = 0; break;
	    }
	}
	if( i==1 )
	    *d++ = reg;
    }

    return d - (byte*)buffer;
}


#ifdef DOCUMENTATION
@Summary FWriteFromBase64
 #include <stdio.h>
 #include <wk/string.h>

 size_t FWriteFromBase64( fp, source);
 FILE *fp;		  Output stream
 const char *source;	  String with the base-64 representation
@Description
 This function converts a base-64 string back to its binary representation
 by writing it to FP.
 Invalid characters (e.g. spaces) are simply skipped.
 The coding rules are interchangeable with those set up in RFC-1113;
 the padding character ('=') is valid, but has no effect.
@Return Value
 Number of bytes written.
@See Also
 ConvertFromBase64 CheckBase64String
#endif /*DOCUMENTATION*/

size_t
FWriteFromBase64( FILE *fp, const char *string )
{
    size_t n;
    const byte *s;
    int i;
    unsigned c, reg;

    if( !isInitialized )
	Init();

    i = 0;
    reg = 0; /* avoid compiler warning */
    n = 0;
    for(s = (const byte*)string; *s; s++ ) {
	if( *s >= 128 || (c = asctobin[*s]) >= 64 )
	    continue; /* skip invalid characters */
	switch(i) {
	  case 0:
	    reg = c << 2;
	    i = 1; break;
	  case 1:
	    reg |= (c >> 4) & 0x03;
	    putc(reg, fp); n++;
	    reg = (c << 4) & 0xf0;
	    i = 2; break;
	  case 2:
	    reg |= (c >> 2) & 0x0f;
	    putc(reg, fp); n++;
	    reg = (c << 6) & 0xc0;
	    i = 3; break;
	  case 3:
	    reg |= c & 0x3f;
	    putc(reg, fp); n++,
	    i = 0; break;
	}
    }
    if( i==1 ) {
	putc(reg, fp);
	n++;
    }

    return n;
}



#ifdef DOCUMENTATION
@Summary CheckBase64String
 #include <wk/string.h>

 int CheckBase64String( string );
 void *string;		  String with the base-64 representation
@Description
 This function validates a string; that it, it checks wether the string
 contains only base-64 characters or not.
 Note, that '=' is a valid character, but '*' is not valid.
@Return Value
 0 := string does only contain base-64 characters
 1 := string includes additional white spaces
 2 := string contains other charaters
@See Also
 ConvertToBase64 ConvertFromBase64
#endif /*DOCUMENTATION*/


int
CheckBase64String( const char *string )
{
    const byte *s;
    int flag=0;

    if( !isInitialized )
	Init();
    for(s = (const byte*)string; *s; s++ ) {
	if( *s >= 128 )
	    return 2; /* always invalid */
	else if( asctobin[*s] >= 64 ) {
	    if( *s == '=' )
		; /* filler character used by rfc1113 */
	    else if( isspace( *s ) )
		flag = 1;
	    else
		return 2; /* invalid */
	}
    }
    return flag;
}



/****************
 * runtime initialization
 */

static void
Init()
{
    const byte *s;
    int i;

    for(i=0; i < 128; i++ )
	asctobin[i] = 255;
    for(s=bintoasc,i=0; *s; s++,i++ )
	if( *s < 128 ) /* yes, I'm a little bit paranoid */
	    asctobin[*s] = i;
    isInitialized = 1;
}

/*** bottom of file ***/
