/*
ppm2pbm.c Version 0.2.0 - Convert PPM to PBM
Copyright (C) 2004-2010  dondalah721@yahoo.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* Example of a PPM header */
/* 000000  50362031 36383020  32313238 20323535  {P6 1680 2128 255}  */
/* 000010  0affffff ffffffff  ffffffff ffffffff  {                }  */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#define PBMBIT(bit) \
      { \
      if (*p       == clrmap->bgred \
         && *(p+1) == clrmap->bggreen \
         && *(p+2) == clrmap->bgblue) \
	 bgfound = 1; \
      else \
	 { \
	 *r |= bit; \
         if (clrmap->fgred > 255) \
	    { \
	    clrmap->fgred   = *p; \
	    clrmap->fggreen = *(p+1); \
	    clrmap->fgblue  = *(p+2); \
	    } /* if first foreground color */ \
         else if (*p  == clrmap->fgred   \
            && *(p+1) == clrmap->fggreen \
            && *(p+2) == clrmap->fgblue); \
         else \
	    { \
	    badppm("More than 2 colors."); \
	    } /* more than 2 colors */ \
	 } /* else foreground */ \
      rowsz++; \
      if (rowsz >= clrmap->wdth) break; \
      p += 3; \
      }

typedef struct clrfmt {
   int hndl;
   int wdth;
   int hght;
   int maxclr;
   int pbmlen;
   int fgred;
   int fggreen;
   int fgblue;
   int bgred;
   int bggreen;
   int bgblue;
   unsigned char *buf;
   unsigned char *pbmbuf;
   char bgname[128];
   char map[256];
   } clrfmt;

void putstx(pgm)
char *pgm;
   {
   fprintf(stderr,"Usage: %s [-b color] "
      "[-m filename]\n", pgm);
   fprintf(stderr,"Where -b color is the background color\n");
   fprintf(stderr,"Color is a name in rgb.txt\n");
   fprintf(stderr,"   or your color map file\n");
   fprintf(stderr,"Filename is the name "
      "of your color map file\n");
   fprintf(stderr,"Input is read from standard input.\n");
   fprintf(stderr,"Output is written to standard output.\n");
   fprintf(stderr,"Example 1: %s -b blue\n", pgm);
   fprintf(stderr,"Example 2: %s -b blue "
      "-m tstrgb.txt\n", pgm);
   exit(1);
   } /* putstx */

void badparm(msg,pgm)
char *msg;
char *pgm;
   {
   fprintf(stderr,"%s\n", msg);
   putstx(pgm);
   exit(1);
   } /* badparm */

void badrgb(msg)
char *msg;
   {
   fprintf(stderr,"ppm2pbm: color map file error\n");
   fprintf(stderr,"%s\n", msg);
   exit(1);
   } /* badrgb */

void badppm(msg)
char *msg;
   {
   fprintf(stderr,"ppm2pbm: incorrect ppm file\n");
   fprintf(stderr,"%s\n", msg);
   exit(1);
   } /* badppm */

int getbyte()
   {
   int rdlen;
   unsigned char buf[8];
   rdlen = read(0,buf,1);
   if (rdlen == 1) return(buf[0]);
   if (!rdlen) return(EOF);
   perror("ppm2pbm: PPM read error");
   exit(1);
   } /* getbyte */

int getraster(len,clrmap)
int len;
clrfmt *clrmap;
   {
   int totlen;
   int rdlen;
   totlen = len;
   while (totlen > 0)
      {
      rdlen = read(0,clrmap->buf,totlen);
      totlen -= rdlen;
      if (!rdlen) return(len - totlen);
      else if (rdlen < 0)
	 {
         perror("ppm2pbm: PPM raster read error");
         exit(1);
	 } /* read error */
      } /* read loop */
   return(len);
   } /* getraster */

void putbyte(ch)
int ch;
   {
   int wrtlen;
   unsigned char buf[8];
   buf[0] = ch;
   wrtlen = write(1,buf,1);
   if (wrtlen == 1) return;
   perror("ppm2pbm: PPM write error");
   exit(1);
   } /* putbyte */

void putraster(clrmap)
clrfmt *clrmap;
   {
   int totlen;
   int wrtlen;
   totlen = clrmap->pbmlen;
   while (totlen > 0)
      {
      wrtlen = write(1,clrmap->pbmbuf,totlen);
      totlen -= wrtlen;
      if (!totlen) return;
      else if (totlen < 0)
	 {
         fprintf(stderr,"ppm2pbm: write len %d\n",
	    clrmap->pbmlen - totlen);
         fprintf(stderr,"len should be %d\n",
	    clrmap->pbmlen);
         perror("PPM raster write error");
         exit(1);
	 } /* write error */
      else if (wrtlen < 0)
	 {
         perror("ppm2pbm: PBM raster write error");
         exit(1);
	 } /* write error */
      } /* write loop */
   } /* putraster */

int getrgb(hndl)
int hndl;
   {
   int rdlen;
   unsigned char buf[8];
   rdlen = read(hndl,buf,1);
   if (rdlen == 1) return(buf[0]);
   if (!rdlen) return(EOF);
   perror("ppm2pbm: rgb.txt read error");
   exit(1);
   } /* getbyte */

int opn(clrmap)
clrfmt *clrmap;
   {
   int hndl;
   hndl = open(clrmap->map,O_RDONLY);
   if (hndl < 0)
      {
      fprintf(stderr,"ppm2pbm: error "
	 "opening %s\n",
	 clrmap->map);
      perror("Error opening color map file");
      exit(1);
      } /* open err */
   return(hndl);
   } /* opn */

void cls(hndl)
int hndl;
   {
   int rslt;
   rslt = close(hndl);
   if (rslt < 0)
      {
      perror("ppm2pbm: Error closing rgb.txt");
      exit(1);
      } /* close err */
   } /* cls */

void flush(hndl)
int hndl;
   {
   int ch;
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF);
   if (ch == EOF) badrgb("During flush");
   } /* flush */

int getcolor(hndl,red,green,blue,name)
int hndl;
int *red;
int *green;
int *blue;
char *name;
   {
   int ch;
   int col;
   char *p;
   ch = 999999;
   p = (char *) name;
   *p = '\0';
   *red = *green = *blue = col = 0;
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF)
      {
      if (!col && ch == '!')
	 {
	 col = 1;
	 flush(hndl);
         continue;
	 }
      if (ch >= '0' && ch <= '9') break;
      } /* left justify red */
   if (ch == EOF) return(1);
   if (ch < '0' || ch > '9')
      badrgb("Before red");
   *red = (*red * 10) + (ch - '0');
   while ((ch = getrgb(hndl)) != ' '
      && ch != '\t'
      && ch != '\n'
      && ch != EOF)
      {
      if (ch >= '0' && ch <= '9')
         *red = (*red * 10) + (ch - '0');
      } /* red */
   if (ch == ' ' || ch == '\t');
   else badrgb("After red");
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF)
      {
      if (ch >= '0' && ch <= '9') break;
      } /* left justify green */
   if (ch < '0' || ch > '9')
      badrgb("Before green");
   *green = (*green * 10) + (ch - '0');
   while ((ch = getrgb(hndl)) != ' '
      && ch != '\t'
      && ch != '\n'
      && ch != EOF)
      {
      if (ch >= '0' && ch <= '9')
         *green = (*green * 10) + (ch - '0');
      } /* green */
   if (ch == ' ' || ch == '\t');
   else badrgb("After green");
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF)
      {
      if (ch >= '0' && ch <= '9') break;
      } /* left justify blue */
   if (ch < '0' || ch > '9')
      badrgb("Before blue");
   *blue = (*blue * 10) + (ch - '0');
   while ((ch = getrgb(hndl)) != ' '
      && ch != '\t'
      && ch != '\n'
      && ch != EOF)
      {
      if (ch >= '0' && ch <= '9')
         *blue = (*blue * 10) + (ch - '0');
      } /* blue */
   if (ch == ' ' || ch == '\t');
   else badrgb("After blue");
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF)
      {
      if (ch == ' ') continue;
      if (ch == '\t') continue;
      break;
      } /* left justify name */
   if (ch <= ' ' || ch > '~')
      badrgb("Before name");
   /* p points to name */
   *p++ = ch;
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF)
      {
      if (ch >= ' ' && ch <= '~')
         *p++ = ch;
      } /* name */
   if (ch == '\n');
   else badrgb("After name");
   *p = '\0';
   return(0);
   } /* getcolor */

void getmap(clrmap)
clrfmt *clrmap;
   {
   int rslt;
   int eofsw;
   int red;
   int green;
   int blue;
   char name[128];
   eofsw = 0;
   clrmap->bgred   = 999999;
   clrmap->bgblue  = 999999;
   clrmap->bggreen = 999999;
   while (!eofsw)
      {
      eofsw = getcolor(clrmap->hndl,
	 &red,&green,&blue,name);
      if (!eofsw)
	 {
	 rslt = strcmp(clrmap->bgname,name);
	 if (!rslt)
	    {
	    clrmap->bgred   = red;
	    clrmap->bgblue  = blue;
	    clrmap->bggreen = green;
	    break;
	    } /* if background */
	 } /* if not eof */
      } /* for each color */
   } /* getmap */

void puthdr(clrmap)
clrfmt *clrmap;
   {
   int ch;
   ch = getbyte();
   if (ch != 'P')
      badppm("Byte 1 is not 'P'");
   putbyte(ch);
   ch = getbyte();
   if (ch != '6')
      badppm("Byte 2 is not '6'");
   /* convert P6 to P4 */
   putbyte('4');
   clrmap->wdth = clrmap->hght =
      clrmap->maxclr = 0;
   /* left justify width */
   while ((ch = getbyte()) == ' '
      || ch == '\t'
      || ch == '\r'
      || ch == '\n'
      || ch == '#')
      {
      if (ch == '#')
	 {
	 putbyte(ch);
         while ((ch = getbyte()) != '\n'
	    && ch != EOF)
	    putbyte(ch);
	 if (ch == EOF)
            badppm("End of file reading header.");
	 else putbyte(ch);
	 continue;
	 } /* comment */
      else putbyte(ch);
      } /* white space */
   if (ch >= '0' && ch <= '9')
      {
      clrmap->wdth = (clrmap->wdth * 10) + (ch - '0');
      putbyte(ch);
      } /* if valid first digit */
   else badppm("Invalid width");
   /* width */
   while ((ch = getbyte()) >= '0'
      && ch <= '9')
      {
      clrmap->wdth = (clrmap->wdth * 10) + (ch - '0');
      putbyte(ch);
      } /* for each digit */
   if (ch != ' '
      && ch != '\t'
      && ch != '\r'
      && ch != '\n')
      badppm("Invalid width");
   else
      putbyte(ch);
   /* left justify height */
   while ((ch = getbyte()) == ' '
      || ch == '\t'
      || ch == '\r'
      || ch == '\n'
      || ch == '#')
      {
      if (ch == '#')
	 {
	 putbyte(ch);
         while ((ch = getbyte()) != '\n'
	    && ch != EOF)
	    putbyte(ch);
	 if (ch == EOF)
            badppm("End of file reading header.");
	 else putbyte(ch);
	 continue;
	 } /* comment */
      else putbyte(ch);
      } /* white space */
   if (ch >= '0' && ch <= '9')
      {
      clrmap->hght = (clrmap->hght * 10) + (ch - '0');
      putbyte(ch);
      } /* if valid first digit */
   else badppm("Invalid height");
   /* height */
   while ((ch = getbyte()) >= '0'
      && ch <= '9')
      {
      clrmap->hght = (clrmap->hght * 10) + (ch - '0');
      putbyte(ch);
      } /* for each digit */
   if (ch != ' '
      && ch != '\t'
      && ch != '\r'
      && ch != '\n')
      badppm("Invalid height");
   else
      putbyte('\n');
   /* left justify max color */
   while ((ch = getbyte()) == ' '
      || ch == '\t'
      || ch == '\r'
      || ch == '\n'
      || ch == '#')
      {
      if (ch == '#')
	 {
         while ((ch = getbyte()) != '\n'
	    && ch != EOF);
	 if (ch == EOF)
            badppm("End of file reading header.");
	 continue;
	 } /* comment */
      else putbyte(ch);
      } /* white space */
   if (ch >= '0' && ch <= '9')
      {
      clrmap->maxclr = (clrmap->maxclr * 10) + (ch - '0');
      } /* if valid first digit */
   else badppm("Invalid maximum color");
   /* max color */
   while ((ch = getbyte()) >= '0'
      && ch <= '9')
      {
      clrmap->maxclr = (clrmap->maxclr * 10) + (ch - '0');
      } /* for each digit */
   if (ch != ' '
      && ch != '\t'
      && ch != '\r'
      && ch != '\n')
      badppm("Invalid maximum color");
   if (clrmap->maxclr > 255)
      badppm("Invalid maximum color");
   if (clrmap->maxclr < 1)
      badppm("Invalid maximum color");
   } /* puthdr */

int cvt2pbm(len,clrmap)
int len;
clrfmt *clrmap;
   {
   int rowsz;
   int bgfound;
   unsigned char *p,*q,*r;
   rowsz = bgfound = 0;
   p = (unsigned char *) clrmap->buf;
   q = (unsigned char *) p + len;
   r = (unsigned char *) clrmap->pbmbuf;
   while (p < q)
      {
      *r = '\0';  /* initialize pbm background */
      PBMBIT(0x80)
      PBMBIT(0x40)
      PBMBIT(0x20)
      PBMBIT(0x10)
      PBMBIT(0x08)
      PBMBIT(0x04)
      PBMBIT(0x02)
      PBMBIT(0x01)
      r++;
      } /* for each 8 pixels in row */
   putraster(clrmap);
   return(bgfound);
   } /* cvt2pbm */

void cvt(clrmap)
clrfmt *clrmap;
   {
   int len;
   int rownum;
   int rowsz;
   int bgfound;
   rowsz = clrmap->wdth * 3;
   clrmap->buf = (unsigned char *) malloc(rowsz + 16);
   if (clrmap->buf == NULL)
      {
      fprintf(stderr,"ppm2pbm: out of memory "
	 "allocating PPM buffer\n");
      exit(1);
      } /* out of mem */
   clrmap->pbmlen = (clrmap->wdth + 7) >> 3;
   clrmap->pbmbuf = (unsigned char *)
      malloc(clrmap->pbmlen + 16);
   if (clrmap->pbmbuf == NULL)
      {
      fprintf(stderr,"ppm2pbm: out of memory "
	 "allocating PBM buffer\n");
      exit(1);
      } /* out of mem */
   bgfound = rownum = 0;
   len = 999999;
   while (len)
      {
      len = getraster(rowsz,clrmap);
      if (!len)
	 {
	 if (rownum != clrmap->hght)
	    {
	    fprintf(stderr,"ppm2pbm: rows read = %d\n",
	       rownum);
	    fprintf(stderr,"Should have read %d rows.\n",
	       clrmap->hght);
	    exit(1);
	    } /* invalid # rows */
	 break;
	 } /* if eof */
      rownum++;
      if (len != rowsz)
	 {
	 fprintf(stderr,"ppm2pbm: row %d has %d bytes\n",
	    rownum, len);
	 fprintf(stderr,"Should read %d bytes "
	    "in each row.\n",
	    rowsz);
	 exit(1);
	 } /* invalid row len */
      bgfound |= (int) cvt2pbm(len,clrmap);
      } /* for each row */
   if (!bgfound)
      {
      fprintf(stderr,"Background %s not found\n",
	 clrmap->bgname);
      } /* background not found */
   free(clrmap->buf);
   free(clrmap->pbmbuf);
   } /* cvt */

int main(argc,argv)
int argc;
char **argv;
   {
   int i;
   int rslt;
   int rslt2;
   clrfmt clrmap;
   if (argc == 1 || argc == 3 || argc == 5);
   else
      {
      if (argc < 5)
	 {
         badparm("Too few parms",
	    *argv);
	 } /* too few parms */
      else if (argc > 5)
	 {
         badparm("Too many parms",
	    *argv);
	 } /* too few parms */
      } /* incorrect parm */

   /* in Debian rgb.txt is at */
   /* /usr/X11/lib/X11/rgb.txt */

   /* in Ubuntu rgb.txt is at */
   /* /usr/share/X11/rgb.txt */

   /* default values */
   strcpy(clrmap.bgname,"white");
   strcpy(clrmap.map,"/usr/share/X11/rgb.txt");
   i = 1;
   while (i < argc)
      {
      rslt  = strcmp(*(argv+i),"-b");
      rslt2 = strcmp(*(argv+i),"-m");
      if (!rslt)
	 {
	 strcpy(clrmap.bgname,*(argv+i+1));
	 } /* background */
      else if (!rslt2)
	 {
	 strcpy(clrmap.map,*(argv+i+1));
	 } /* color map file */
      else
	 {
	 fprintf(stderr,"Invalid switch %s\n",
	    *(argv+i));
	 putstx(*argv);
	 } /* invalid switch */
      i += 2;
      } /* for each switch/value pair */

   clrmap.hndl = opn(&clrmap);
   getmap(&clrmap);
   cls(clrmap.hndl);
   if (clrmap.bgred > 255
      || clrmap.bgblue > 255
      || clrmap.bggreen > 255)
      {
      fprintf(stderr,"ppm2pbm: background color "
	 "%s\n   not found in %s\n",
	 clrmap.bgname, clrmap.map);
      exit(1);
      } /* background not found */
   /* to count the # of foreground colors */
   clrmap.fgred   = 999999;
   clrmap.fggreen = 999999;
   clrmap.fgblue  = 999999;
   puthdr(&clrmap);
   cvt(&clrmap);
   return(0);
   } /* main */
