/*
ppmcolor.c Version 0.2.0 - Convert PPM Colors
Copyright (C) 2004-2010  dondalah721@yahoo.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* Example of a PPM header */
/* 000000  50362031 36383020  32313238 20323535  {P6 1680 2128 255}  */
/* 000010  0affffff ffffffff  ffffffff ffffffff  {                }  */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

typedef struct clrfmt {
   int hndl;
   int wdth;
   int hght;
   int maxclr;
   int buflen;
   int fgred;
   int fggreen;
   int fgblue;
   int bgred;
   int bggreen;
   int bgblue;
   int curr_fgred;
   int curr_fggreen;
   int curr_fgblue;
   int curr_bgred;
   int curr_bggreen;
   int curr_bgblue;
   unsigned char *buf;
   char fgname[128];
   char bgname[128];
   char curr_fg[128];
   char curr_bg[128];
   char map[256];
   } clrfmt;

void putstx(pgm)
char *pgm;
   {
   fprintf(stderr,"Usage: %s <options>\n", pgm);
   fprintf(stderr,"Options:\n");
   fprintf(stderr,"   -f color    (foreground color)\n");
   fprintf(stderr,"   -F color    (current foreground color)\n");
   fprintf(stderr,"   -b color    (background color)\n");
   fprintf(stderr,"   -B color    (current background color)\n");
   fprintf(stderr,"   -m filename (color map file)\n");
   fprintf(stderr,"Color is a name in rgb.txt\n");
   fprintf(stderr,"   or your own color map file\n");
   fprintf(stderr,"Input is read from standard input.\n");
   fprintf(stderr,"Output is written to standard output.\n");
   fprintf(stderr,"Example 1: %s -f red -b blue\n", pgm);
   fprintf(stderr,"Example 2: %s -f red -b blue "
      "-F yellow -B \"dark green\"\n", pgm);
   fprintf(stderr,"Example 3: %s -f red -b blue "
      "-m tstrgb.txt\n", pgm);
   exit(1);
   } /* putstx */

void badswtch(msg,pgm)
char *msg;
char *pgm;
   {
   fprintf(stderr,"%s\n", msg);
   putstx(pgm);
   exit(1);
   } /* badswtch */

void badrgb(msg)
char *msg;
   {
   fprintf(stderr,"ppmcolor: color map error\n");
   fprintf(stderr,"%s\n", msg);
   exit(1);
   } /* badrgb */

void badppm(msg)
char *msg;
   {
   fprintf(stderr,"ppmcolor: incorrect PPM file\n");
   fprintf(stderr,"%s\n", msg);
   exit(1);
   } /* badppm */

int getbyte()
   {
   int rdlen;
   unsigned char buf[8];
   rdlen = read(0,buf,1);
   if (rdlen == 1) return(buf[0]);
   if (!rdlen) return(EOF);
   perror("ppmcolor: PPM read error");
   exit(1);
   } /* getbyte */

int getraster(len,clrmap)
int len;
clrfmt *clrmap;
   {
   int totlen;
   int rdlen;
   totlen = len;
   while (totlen > 0)
      {
      rdlen = read(0,clrmap->buf,totlen);
      totlen -= rdlen;
      if (!rdlen) return(len - totlen);
      else if (rdlen < 0)
	 {
         perror("ppmcolor: PPM raster read error");
         exit(1);
	 } /* read error */
      } /* read loop */
   return(len);
   } /* getraster */

void putbyte(ch)
int ch;
   {
   int wrtlen;
   unsigned char buf[8];
   buf[0] = ch;
   wrtlen = write(1,buf,1);
   if (wrtlen == 1) return;
   perror("ppmcolor: PPM write error");
   exit(1);
   } /* putbyte */

void putraster(len,clrmap)
int len;
clrfmt *clrmap;
   {
   int totlen;
   int wrtlen;
   totlen = len;
   while (totlen > 0)
      {
      wrtlen = write(1,clrmap->buf,totlen);
      totlen -= wrtlen;
      if (!totlen) return;
      else if (totlen < 0)
	 {
         fprintf(stderr,"ppmcolor: write len %d\n",
	    len - totlen);
         fprintf(stderr,"len should be %d\n",
	    len);
         perror("PPM raster write error");
         exit(1);
	 } /* write error */
      else if (wrtlen < 0)
	 {
         perror("ppmcolor: PPM raster write error");
         exit(1);
	 } /* write error */
      } /* write loop */
   } /* putraster */

int getrgb(hndl)
int hndl;
   {
   int rdlen;
   unsigned char buf[8];
   rdlen = read(hndl,buf,1);
   if (rdlen == 1) return(buf[0]);
   if (!rdlen) return(EOF);
   perror("ppmcolor: rgb.txt read error");
   exit(1);
   } /* getbyte */

int opn(clrmap)
clrfmt *clrmap;
   {
   int hndl;
   hndl = open(clrmap->map,O_RDONLY);
   if (hndl < 0)
      {
      fprintf(stderr,"ppmcolor: error opening %s\n",
	 clrmap->map);
      perror("Error opening color map");
      exit(1);
      } /* open err */
   return(hndl);
   } /* opn */

void cls(hndl)
int hndl;
   {
   int rslt;
   rslt = close(hndl);
   if (rslt < 0)
      {
      perror("ppmcolor: error closing "
	 "color map file");
      exit(1);
      } /* close err */
   } /* cls */

void flush(hndl)
int hndl;
   {
   int ch;
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF);
   if (ch == EOF) badrgb("During flush");
   } /* flush */

int getcolor(hndl,red,green,blue,name)
int hndl;
int *red;
int *green;
int *blue;
char *name;
   {
   int ch;
   int col;
   char *p;
   ch = 999999;
   p = (char *) name;
   *p = '\0';
   *red = *green = *blue = col = 0;
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF)
      {
      if (!col && ch == '!')
	 {
	 col = 1;
	 flush(hndl);
         continue;
	 }
      if (ch >= '0' && ch <= '9') break;
      } /* left justify red */
   if (ch == EOF) return(1);
   if (ch < '0' || ch > '9')
      badrgb("Before red");
   *red = (*red * 10) + (ch - '0');
   while ((ch = getrgb(hndl)) != ' '
      && ch != '\t'
      && ch != '\n'
      && ch != EOF)
      {
      if (ch >= '0' && ch <= '9')
         *red = (*red * 10) + (ch - '0');
      } /* red */
   if (ch == ' ' || ch == '\t');
   else badrgb("After red");
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF)
      {
      if (ch >= '0' && ch <= '9') break;
      } /* left justify green */
   if (ch < '0' || ch > '9')
      badrgb("Before green");
   *green = (*green * 10) + (ch - '0');
   while ((ch = getrgb(hndl)) != ' '
      && ch != '\t'
      && ch != '\n'
      && ch != EOF)
      {
      if (ch >= '0' && ch <= '9')
         *green = (*green * 10) + (ch - '0');
      } /* green */
   if (ch == ' ' || ch == '\t');
   else badrgb("After green");
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF)
      {
      if (ch >= '0' && ch <= '9') break;
      } /* left justify blue */
   if (ch < '0' || ch > '9')
      badrgb("Before blue");
   *blue = (*blue * 10) + (ch - '0');
   while ((ch = getrgb(hndl)) != ' '
      && ch != '\t'
      && ch != '\n'
      && ch != EOF)
      {
      if (ch >= '0' && ch <= '9')
         *blue = (*blue * 10) + (ch - '0');
      } /* blue */
   if (ch == ' ' || ch == '\t');
   else badrgb("After blue");
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF)
      {
      if (ch == ' ') continue;
      if (ch == '\t') continue;
      break;
      } /* left justify name */
   if (ch <= ' ' || ch > '~')
      badrgb("Before name");
   /* p points to name */
   *p++ = ch;
   while ((ch = getrgb(hndl)) != '\n'
      && ch != EOF)
      {
      if (ch >= ' ' && ch <= '~')
         *p++ = ch;
      } /* name */
   if (ch == '\n');
   else badrgb("After name");
   *p = '\0';
   return(0);
   } /* getcolor */

void getmap(clrmap)
clrfmt *clrmap;
   {
   int rslt;
   int rslt2;
   int rslt3;
   int rslt4;
   int eofsw;
   int red;
   int green;
   int blue;
   char name[128];
   eofsw = 0;
   clrmap->fgred   = 999999;
   clrmap->fgblue  = 999999;
   clrmap->fggreen = 999999;
   clrmap->bgred   = 999999;
   clrmap->bgblue  = 999999;
   clrmap->bggreen = 999999;
   clrmap->curr_fgred   = 999999;
   clrmap->curr_fgblue  = 999999;
   clrmap->curr_fggreen = 999999;
   clrmap->curr_bgred   = 999999;
   clrmap->curr_bgblue  = 999999;
   clrmap->curr_bggreen = 999999;
   while (!eofsw)
      {
      eofsw = getcolor(clrmap->hndl,
	 &red,&green,&blue,name);
      if (!eofsw)
	 {
	 rslt  = strcmp(clrmap->fgname,name);
	 rslt2 = strcmp(clrmap->bgname,name);
	 rslt3 = strcmp(clrmap->curr_fg,name);
	 rslt4 = strcmp(clrmap->curr_bg,name);
	 if (!rslt)
	    {
	    clrmap->fgred   = red;
	    clrmap->fgblue  = blue;
	    clrmap->fggreen = green;
	    } /* if foreground */
	 if (!rslt2)
	    {
	    clrmap->bgred   = red;
	    clrmap->bgblue  = blue;
	    clrmap->bggreen = green;
	    } /* if background */
	 if (!rslt3)
	    {
	    clrmap->curr_fgred   = red;
	    clrmap->curr_fgblue  = blue;
	    clrmap->curr_fggreen = green;
	    } /* if current foreground */
	 if (!rslt4)
	    {
	    clrmap->curr_bgred   = red;
	    clrmap->curr_bgblue  = blue;
	    clrmap->curr_bggreen = green;
	    } /* if current background */
	 } /* if not eof */
      if (clrmap->fgred < 256
         && clrmap->bgred < 256
         && clrmap->curr_fgred < 256
         && clrmap->curr_bgred < 256)
         break;
      } /* for each color */
   } /* getmap */

void puthdr(clrmap)
clrfmt *clrmap;
   {
   int ch;
   ch = getbyte();
   if (ch == EOF) badppm("Empty PPM file");
   if (ch != 'P')
      badppm("Byte 1 is not 'P'");
   putbyte(ch);
   ch = getbyte();
   if (ch == EOF) badppm("Empty PPM file");
   if (ch != '6')
      badppm("Byte 2 is not '6'");
   putbyte(ch);
   clrmap->wdth = clrmap->hght =
      clrmap->maxclr = 0;
   /* left justify width */
   while ((ch = getbyte()) == ' '
      || ch == '\t'
      || ch == '\r'
      || ch == '\n'
      || ch == '#')
      {
      if (ch == '#')
	 {
	 putbyte(ch);
         while ((ch = getbyte()) != '\n'
	    && ch != EOF)
	    putbyte(ch);
	 if (ch == EOF)
	    badppm("End of file reading header");
	 else putbyte(ch);
	 continue;
	 } /* if comment */
      else putbyte(ch);
      } /* white space */
   if (ch >= '0' && ch <= '9')
      {
      clrmap->wdth = (clrmap->wdth * 10) + (ch - '0');
      putbyte(ch);
      } /* if valid first digit */
   else badppm("Invalid width");
   /* width */
   while ((ch = getbyte()) >= '0'
      && ch <= '9')
      {
      clrmap->wdth = (clrmap->wdth * 10) + (ch - '0');
      putbyte(ch);
      } /* for each digit */
   if (ch != ' '
      && ch != '\t'
      && ch != '\r'
      && ch != '\n')
      badppm("Invalid width");
   else
      putbyte(ch);
   /* left justify height */
   while ((ch = getbyte()) == ' '
      || ch == '\t'
      || ch == '\r'
      || ch == '\n'
      || ch == '#')
      {
      if (ch == '#')
	 {
	 putbyte(ch);
         while ((ch = getbyte()) != '\n'
	    && ch != EOF)
	    putbyte(ch);
	 if (ch == EOF)
	    badppm("End of file reading header");
	 else putbyte(ch);
	 continue;
	 } /* if comment */
      else putbyte(ch);
      } /* white space */
   if (ch >= '0' && ch <= '9')
      {
      clrmap->hght = (clrmap->hght * 10) + (ch - '0');
      putbyte(ch);
      } /* if valid first digit */
   else badppm("Invalid height");
   /* height */
   while ((ch = getbyte()) >= '0'
      && ch <= '9')
      {
      clrmap->hght = (clrmap->hght * 10) + (ch - '0');
      putbyte(ch);
      } /* for each digit */
   if (ch != ' '
      && ch != '\t'
      && ch != '\r'
      && ch != '\n')
      badppm("Invalid height");
   else
      putbyte(ch);
   /* left justify max color */
   while ((ch = getbyte()) == ' '
      || ch == '\t'
      || ch == '\r'
      || ch == '\n'
      || ch == '#')
      {
      if (ch == '#')
	 {
	 putbyte(ch);
         while ((ch = getbyte()) != '\n'
	    && ch != EOF)
	    putbyte(ch);
	 if (ch == EOF)
	    badppm("End of file reading header");
	 else putbyte(ch);
	 continue;
	 } /* if comment */
      else putbyte(ch);
      } /* white space */
   if (ch >= '0' && ch <= '9')
      {
      clrmap->maxclr = (clrmap->maxclr * 10) + (ch - '0');
      putbyte(ch);
      } /* if valid first digit */
   else badppm("Invalid maximum color");
   /* max color */
   while ((ch = getbyte()) >= '0'
      && ch <= '9')
      {
      clrmap->maxclr = (clrmap->maxclr * 10) + (ch - '0');
      putbyte(ch);
      } /* for each digit */
   if (ch != ' '
      && ch != '\t'
      && ch != '\r'
      && ch != '\n')
      badppm("Invalid maximum color");
   else
      putbyte(ch);
   if (clrmap->maxclr > 255)
      badppm("Invalid maximum color");
   if (clrmap->maxclr < 1)
      badppm("Invalid maximum color");
   } /* puthdr */

void cvt(len,clrmap)
int len;
clrfmt *clrmap;
   {
   unsigned char *p,*q;
   p = (unsigned char *) clrmap->buf;
   q = (unsigned char *) p + len;
   while (p < q)
      {
      if (*p == clrmap->curr_fgred
         && *(p+1) == clrmap->curr_fggreen
         && *(p+2) == clrmap->curr_fgblue)
	 {
	 *p     = (unsigned char) clrmap->fgred   & 255;
	 *(p+1) = (unsigned char) clrmap->fggreen & 255;
	 *(p+2) = (unsigned char) clrmap->fgblue  & 255;
	 } /* if foreground */
      else if (*p  == clrmap->curr_bgred
         && *(p+1) == clrmap->curr_bggreen
         && *(p+2) == clrmap->curr_bgblue)
	 {
	 *p     = (unsigned char) clrmap->bgred   & 255;
	 *(p+1) = (unsigned char) clrmap->bggreen & 255;
	 *(p+2) = (unsigned char) clrmap->bgblue  & 255;
	 } /* else if background */
      else
         {
	 fprintf(stderr,"ppmcolor: color does not "
	    "match image file.\n");
	 fprintf(stderr,"Foreground color is not %s or\n",
	    clrmap->curr_fg);
	 fprintf(stderr,"background color is not %s\n",
	    clrmap->curr_bg);
	 exit(1);
	 } /* not current fg or current bg */
      p += 3;
      } /* for each pixel in row */
   putraster(len,clrmap);
   } /* cvt */

void chgclr(clrmap)
clrfmt *clrmap;
   {
   int len;
   int rownum;
   int rowsz;
   rowsz = clrmap->wdth * 3;
   clrmap->buf = (unsigned char *) malloc(rowsz + 16);
   if (clrmap->buf == NULL)
      {
      fprintf(stderr,"ppmcolor: out of memory "
	 "allocating PPM buffer\n");
      exit(1);
      } /* out of mem */
   rownum = 0;
   len = 999999;
   while (len)
      {
      len = getraster(rowsz,clrmap);
      if (!len)
	 {
	 if (rownum != clrmap->hght)
	    {
	    fprintf(stderr,"ppmcolor: rows read = %d\n",
	       rownum);
	    fprintf(stderr,"Should have read %d rows.\n",
	       clrmap->hght);
	    exit(1);
	    } /* invalid # rows */
	 break;
	 } /* if eof */
      rownum++;
      if (len != rowsz)
	 {
	 fprintf(stderr,"ppmcolor: row %d has %d bytes\n",
	    rownum, len);
	 fprintf(stderr,"Should read %d bytes "
	    "in each row.\n",
	    rowsz);
	 exit(1);
	 } /* invalid row len */
      cvt(len,clrmap);
      } /* for each block */
   free(clrmap->buf);
   } /* chgclr */

int main(argc,argv)
int argc;
char **argv;
   {
   int i;
   int rslt;
   int rslt2;
   int rslt3;
   int rslt4;
   int rslt5;
   clrfmt clrmap;
   if (argc == 3 || argc == 5 || argc == 7
      || argc == 9 || argc == 11);
   else
      {
      if (argc < 11)
	 {
         badswtch("Too few parms",
	    *argv);
	 } /* too few parms */
      else if (argc > 11)
	 {
         badswtch("Too many parms",
	    *argv);
	 } /* too many parms */
      } /* incorrect argc */

   /* default values */
   strcpy(clrmap.fgname,"black");
   strcpy(clrmap.bgname,"white");
   strcpy(clrmap.curr_fg,"black");
   strcpy(clrmap.curr_bg,"white");

   /* in Debian rgb.txt is at */
   /* /usr/X11/lib/X11/rgb.txt */

   /* in Ubuntu rgb.txt is at */
   /* /usr/share/X11/rgb.txt */

   strcpy(clrmap.map,"/usr/share/X11/rgb.txt");
   i = 1;
   while (i < argc)
      {
      rslt  = strcmp(*(argv+i),"-f");
      rslt2 = strcmp(*(argv+i),"-b");
      rslt3 = strcmp(*(argv+i),"-F");
      rslt4 = strcmp(*(argv+i),"-B");
      rslt5 = strcmp(*(argv+i),"-m");
      if (!rslt)
	 {
	 strcpy(clrmap.fgname,*(argv+i+1));
	 } /* foreground */
      else if (!rslt2)
	 {
	 strcpy(clrmap.bgname,*(argv+i+1));
	 } /* background */
      else if (!rslt3)
	 {
	 strcpy(clrmap.curr_fg,*(argv+i+1));
	 } /* current foreground */
      else if (!rslt4)
	 {
	 strcpy(clrmap.curr_bg,*(argv+i+1));
	 } /* current background */
      else if (!rslt5)
	 {
	 strcpy(clrmap.map,*(argv+i+1));
	 } /* color map */
      else
	 {
	 fprintf(stderr,"Invalid switch %s\n",
	    *(argv+i));
	 putstx(*argv);
	 } /* invalid parm */
      i += 2;
      } /* for each switch and value */

   clrmap.hndl = opn(&clrmap);
   getmap(&clrmap);
   cls(clrmap.hndl);
   if (clrmap.fgred > 255
      || clrmap.fgblue > 255
      || clrmap.fggreen > 255)
      {
      fprintf(stderr,"ppmcolor: foreground color "
	 "%s\n   not found in %s\n",
	 clrmap.fgname, clrmap.map);
      exit(1);
      } /* foreground not found */
   if (clrmap.bgred > 255
      || clrmap.bgblue > 255
      || clrmap.bggreen > 255)
      {
      fprintf(stderr,"ppmcolor: background color "
	 "%s\n   not found in %s\n",
	 clrmap.bgname, clrmap.map);
      exit(1);
      } /* background not found */
   if (clrmap.curr_fgred > 255
      || clrmap.curr_fgblue > 255
      || clrmap.curr_fggreen > 255)
      {
      fprintf(stderr,"ppmcolor: current foreground "
	 "color %s\n   not found in %s\n",
	 clrmap.curr_fg, clrmap.map);
      exit(1);
      } /* current foreground not found */
   if (clrmap.curr_bgred > 255
      || clrmap.curr_bgblue > 255
      || clrmap.curr_bggreen > 255)
      {
      fprintf(stderr,"ppmcolor: current background "
	 "color %s\n   not found in %s\n",
	 clrmap.curr_bg, clrmap.map);
      exit(1);
      } /* current background not found */
   puthdr(&clrmap);
   chgclr(&clrmap);
   return(0);
   } /* main */
