
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: browser.c,v 1.16 2000/04/28 20:46:33 dr Exp $ 
*/


#include <axv_config.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include "browser.h"
#include "fs_browser.h"
#include "zalbum.h"
#include "../lib/dcache.h"
#include "../lib/paramui.h"

/* 10 updates per second should be enough */
#define PROGRESS_BAR_UPDATE 50
#define PROGRESS_BAR_WIDTH  150
#define PROGRESS_BAR_INC    (1.0 / PROGRESS_BAR_WIDTH)

extern void        exit_application ();

static void        browser_init (Browser *browser);
static void        browser_select_file (GtkWidget *widget, char *path, gpointer data);
static gint        browser_select_image (GtkWidget *widget, GdkEventButton *event, Browser *browser);
static gint        browser_progress_monitor (Browser *browser);
static void        browser_destroy (GtkWidget *widget, Browser *browser);
static void        browser_viewer_destroyed (GtkWidget *widget, Browser *browser);

static void        browser_menu_quit (GtkWidget *widget, Browser *browser);
static void        browser_menu_preferences (GtkWidget *widget, Browser *browser);
static void        browser_menu_view_list (GtkWidget *widget, Browser *browser);
static void        browser_menu_view_thumbnails (GtkWidget *widget, Browser *browser);
static void        browser_menu_sort_by_name (GtkWidget *widget, Browser *browser);
static void        browser_menu_sort_by_size (GtkWidget *widget, Browser *browser);
static void        browser_menu_sort_by_date (GtkWidget *widget, Browser *browser);
static void        browser_menu_about (GtkWidget *widget, Browser *browser);


static GtkItemFactoryEntry browser_menu_factory [] = {
  { "/_File"               , NULL, NULL                        , 0, "<Branch>"},
  { "/File/_Open ..."      , NULL, NULL                        , 0, NULL},
  { "/File/sep1"           , NULL, NULL                        , 0, "<Separator>" },
  { "/File/_Quit"          , NULL, browser_menu_quit           , 0, NULL },
  { "/_Edit"               , NULL, NULL                        , 0, "<Branch>"},
  { "/Edit/Open in _Gimp"  , NULL, NULL                        , 0, NULL},
  { "/Edit/Open in _XV"    , NULL, NULL                        , 0, NULL},
  { "/Edit/sep1"           , NULL, NULL                        , 0, "<Separator>" },
  { "/Edit/Cut"            , NULL, NULL                        , 0, NULL},
  { "/Edit/sep2"           , NULL, NULL                        , 0, "<Separator>" },
  { "/Edit/_Preferences"   , NULL, browser_menu_preferences    , 0, NULL},
  { "/_View"               , NULL, NULL                        , 0, "<Branch>"},
  { "/View/_List"          , NULL, browser_menu_view_list      , 0, NULL },
  { "/View/_Thumbnails"    , NULL, browser_menu_view_thumbnails, 0, NULL },
  { "/View/sep1"           , NULL, NULL                        , 0, "<Separator>" },
  { "/View/Sort by name"   , NULL, browser_menu_sort_by_name   , 0, NULL },
  { "/View/Sort by size"   , NULL, browser_menu_sort_by_size   , 0, NULL },
  { "/View/Sort by date"   , NULL, browser_menu_sort_by_date   , 0, NULL },
  { "/_Help"               , NULL, NULL                        , 0, "<LastBranch>" },
  { "/Help/_About"         , NULL, browser_menu_about          , 0, NULL }
};

static int browser_menu_factory_count = sizeof (browser_menu_factory) / sizeof (GtkItemFactoryEntry);
 
Browser *
browser_new (char *start_path) {
  Browser *browser;

  browser = g_new (Browser, 1);
  browser->images      = image_set_new ();
  browser->dir_tree    = fs_browser_new (start_path);
  browser->album       = zalbum_new (ZALBUM_MODE_PREVIEW);
  browser->viewer      = NULL;

  browser_init (browser);

  return browser;
}

static void
browser_init (Browser *browser) {
  GtkWidget *window;
  GtkWidget *vbox, *hbox, *shbox;
  GtkWidget *frame;
  /*  GtkWidget *menubar; */
  GtkWidget *scrolled;
  GtkWidget *paned;
  GtkAccelGroup *accel;
  GtkItemFactory *item_factory;

  gtk_signal_connect (GTK_OBJECT (browser->album), "button_press_event",
		      GTK_SIGNAL_FUNC (browser_select_image), browser);

  accel = gtk_accel_group_new ();
  
  item_factory = gtk_item_factory_new (GTK_TYPE_MENU, "<browser>", accel);
  gtk_item_factory_create_items_ac (item_factory, 
				    browser_menu_factory_count, browser_menu_factory, 
				    browser, 2);

  browser->menu = gtk_item_factory_get_widget (item_factory, "<browser>");

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (browser_destroy), browser);
  gtk_window_set_default_size (GTK_WINDOW (window), 640, 480);
  gtk_accel_group_attach (accel, GTK_OBJECT (window));

  vbox = gtk_vbox_new (0, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_widget_show (vbox);

  /*
  menubar = gtk_item_factory_get_widget (item_factory, "<browser>");
  gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, TRUE, 0);
  gtk_widget_show (menubar);
  */

  paned = gtk_hpaned_new ();
  gtk_paned_set_gutter_size (GTK_PANED (paned), 10);
  gtk_container_add (GTK_CONTAINER (vbox), paned);
  gtk_widget_show (paned);
  
  scrolled = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scrolled), browser->dir_tree);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled), 
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (browser->dir_tree, 210, -1);
  gtk_widget_show (browser->dir_tree);

  gtk_paned_add1 (GTK_PANED (paned), scrolled);
  gtk_signal_connect (GTK_OBJECT (browser->dir_tree), "select_file",
		      GTK_SIGNAL_FUNC (browser_select_file), browser);
  gtk_widget_show (scrolled);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame),
                             GTK_SHADOW_IN);
  gtk_paned_add2 (GTK_PANED (paned), frame);
  gtk_widget_show (frame);

  scrolled = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scrolled), browser->album);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
			  	  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_widget_show (browser->album); 
  gtk_container_add (GTK_CONTAINER (frame), scrolled);
  gtk_widget_show (scrolled);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, TRUE, 0);
  gtk_widget_show (hbox);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame),
                             GTK_SHADOW_IN);
  gtk_box_pack_end (GTK_BOX (hbox), frame, FALSE, TRUE, 0);
  gtk_widget_show (frame);
  
  browser->status_progress = gtk_progress_bar_new ();
  gtk_container_add (GTK_CONTAINER (frame), browser->status_progress);
  gtk_widget_set_usize (browser->status_progress, PROGRESS_BAR_WIDTH, 10);
  gtk_widget_show (browser->status_progress);

  shbox = gtk_hbox_new (TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), shbox, TRUE, TRUE, 0);
  gtk_widget_show (shbox);
  
  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame),
			     GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX (shbox), frame, TRUE, TRUE, 0);
  gtk_widget_show (frame);

  browser->status_directory = gtk_label_new ("");
  gtk_container_add (GTK_CONTAINER (frame), browser->status_directory);
  gtk_widget_show (browser->status_directory);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame),
                             GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX (shbox), frame, TRUE, TRUE, 0);
  gtk_widget_show (frame);

  browser->status_selection = gtk_label_new ("");
  gtk_container_add (GTK_CONTAINER (frame), browser->status_selection);
  gtk_widget_show (browser->status_selection);

  gtk_widget_show (window);
}


void                      
browser_set_directory (Browser *browser, char *path) {
  Image *image;
  DIR *dir;
  struct dirent *entry;
  struct stat   statbuf;
  GList *dirs;
  char buf [PATH_MAX];
  int i, recursive = 0;

  g_return_if_fail (browser);

  if (browser->progress_id)
    gtk_timeout_remove (browser->progress_id);

  zlist_clear (ZLIST (browser->album));
  image_set_clear (browser->images);

  if (!path)
    return;

  dirs = g_list_prepend (NULL, g_strdup (path));

  while (dirs) {
    path = dirs->data;
    dirs = g_list_remove_link (dirs, dirs);

    dcache_update_directory (path);
    
    dir = opendir (path);
    if (!dir)
      continue;
    
    while ((entry = readdir (dir))) {

      /* skip '.' && '..' */
      if ((entry->d_name [0] == '.' && entry->d_name [1] == 0) ||
	  (entry->d_name [0] == '.' && entry->d_name [1] == '.' && entry->d_name [2] == 0))
	  continue;

      if (is_readable_image (entry->d_name, path)) {

	image = image_new (entry->d_name, path);
	image_set_add (browser->images, image);
	image_unref (image);

      } else if (recursive) { /* only stat when needed as it could be quite slow */
	snprintf (buf, PATH_MAX, "%s/%s", path, entry->d_name);
	
	if (!lstat (buf, &statbuf) && S_ISDIR (statbuf.st_mode))
	  dirs = g_list_prepend (dirs, g_strdup (buf));
      }
      
    } /* while entry */
    
    closedir (dir);
    g_free (path);

  } /* while dirs */

  if (image_set_size (browser->images)) {
    image_set_sort (browser->images, IMAGE_SET_SORT_BY_NAME);
    
    zalbum_freeze (browser->album);
    for (i = 0; i < image_set_size (browser->images); i++)
      zalbum_add (ZALBUM (browser->album), 
		  image_set_get_image (browser->images, i));

    zalbum_thawn (browser->album);
  }

  snprintf (buf, PATH_MAX, "%d files", 
	    image_set_size (browser->images));
  gtk_label_set (GTK_LABEL (browser->status_directory), buf);

  if (browser->viewer)
    viewer_set_images (browser->viewer, browser->images, image_set_get_image (browser->images, 0)); 

  browser->progress    = 0.0;
  browser->progress_id = gtk_timeout_add (PROGRESS_BAR_UPDATE, (GtkFunction) browser_progress_monitor, browser);
}

static void
browser_select_file (GtkWidget *widget, char *path, gpointer data) {
  Browser *browser;
  
  g_return_if_fail (data && path);

  browser = data;
  browser_set_directory (browser, path);
}

static gint        
browser_select_image (GtkWidget *widget, GdkEventButton *event, Browser *browser) {
  ZAlbumCell *cell;
  char buf [PATH_MAX];

  g_return_val_if_fail (widget && event && browser, FALSE);

  if (event->type == GDK_BUTTON_PRESS && event->button == 3) {
    gtk_menu_popup (GTK_MENU (browser->menu), 
		    NULL, NULL, NULL, NULL,
		    event->button, event->time);
    return TRUE;
  }

  cell = zlist_cell_from_xy (ZLIST (browser->album), event->x, event->y);
  if (!cell)
    return FALSE;

  snprintf (buf, PATH_MAX, "%s (%dx%dx%d)", 
	    cell->image->file_name,
	    cell->image->width,
	    cell->image->height,
	    cell->image->bpp * 8);
  gtk_label_set (GTK_LABEL (browser->status_selection), buf);

  if (event->type != GDK_2BUTTON_PRESS) 
    return FALSE;
  
  if (!browser->viewer) {
    browser->viewer = viewer_new ();
    viewer_set_images (browser->viewer, browser->images, cell->image);
    gtk_signal_connect (GTK_OBJECT (browser->viewer->zimage), "destroy",
			GTK_SIGNAL_FUNC (browser_viewer_destroyed), browser);
  } else
    viewer_set_image (browser->viewer, cell->image);
  
  return FALSE;
}

static gint
browser_progress_monitor (Browser *browser) {
  gfloat progress;

  g_return_val_if_fail (browser, 0);

  progress = zalbum_get_progress (ZALBUM (browser->album));
  if (progress >= 1.0) {
    gtk_progress_bar_update (GTK_PROGRESS_BAR (browser->status_progress),
			     0.0);
    browser->progress_id = 0;
    return 0;
  }

  if (progress - browser->progress > PROGRESS_BAR_INC) {
    browser->progress = progress;
    gtk_progress_bar_update (GTK_PROGRESS_BAR (browser->status_progress),
			     progress);
  }

  return 1;
}

static void
browser_destroy (GtkWidget *widget, Browser *browser) {
  g_return_if_fail (browser);
  
  browser_set_directory (browser, NULL);

  exit_application ();
}

static void
browser_viewer_destroyed (GtkWidget *widget, Browser *browser) {
  g_return_if_fail (browser);

  browser->viewer = NULL;
}

static void        
browser_menu_quit (GtkWidget *widget, Browser *browser) {
  g_return_if_fail (browser);

  gtk_exit (0);
}

static void        
browser_menu_preferences (GtkWidget *widget, Browser *browser) {
  
  param_ui_new (NULL);
}

static void        
browser_menu_view_list (GtkWidget *widget, Browser *browser) {
  g_return_if_fail (browser);

  zalbum_set_mode (ZALBUM (browser->album), ZALBUM_MODE_LIST);
}

static void        
browser_menu_view_thumbnails (GtkWidget *widget, Browser *browser) {
  g_return_if_fail (browser);

  zalbum_set_mode (ZALBUM (browser->album), ZALBUM_MODE_PREVIEW);
}

static void        
browser_menu_sort_by_name (GtkWidget *widget, Browser *browser) {
  g_return_if_fail (browser);

  image_set_sort (browser->images, IMAGE_SET_SORT_BY_NAME);
}

static void        
browser_menu_sort_by_size (GtkWidget *widget, Browser *browser) {
  g_return_if_fail (browser);

  image_set_sort (browser->images, IMAGE_SET_SORT_BY_SIZE);
}

static void        
browser_menu_sort_by_date (GtkWidget *widget, Browser *browser) {
  g_return_if_fail (browser);

  image_set_sort (browser->images, IMAGE_SET_SORT_BY_MTIME);
}

static void        
browser_menu_about (GtkWidget *widget, Browser *browser) {
  GtkWidget *window;
  GtkWidget *label, *ev;

  g_return_if_fail (browser);

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  ev = gtk_event_box_new ();
  gtk_container_add (GTK_CONTAINER (window), ev);
  label = gtk_label_new ("Another X Viewer\n\n version " VERSION "\n\n"
			 "Copyright (C) 2000 by david Ramboz");
  
  gtk_container_add (GTK_CONTAINER (ev), label);
 
  gtk_signal_connect_object (GTK_OBJECT (ev), "button_press_event", 
			     (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (window));
 
  gtk_widget_show_all (window);
}









