
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 
s
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: fs_browser.c,v 1.3 2000/04/28 20:46:33 dr Exp $ 
*/


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <time.h>
#include <pwd.h>
#include <grp.h>

#include "fs_browser.h"
#include "dnd.h"

#define CTREE_SPACING 5
#define NODE_TEXT(tree, node) (GTK_CELL_PIXTEXT(GTK_CTREE_ROW(node)->row.cell[(tree)->tree_column])->text)

static void          fs_browser_class_init (FSBrowserClass *kclass);
static void          fs_browser_init (FSBrowser *fb);


static void          fs_browser_tree_select_row (GtkCTree *tree, GtkCTreeNode *node, gint column);
static void          fs_browser_tree_expand (GtkCTree *tree, GtkCTreeNode *node);
static void          fs_browser_tree_collapse (GtkCTree *tree, GtkCTreeNode *node);


static void          tree_get_path (FSBrowser *fb, GtkCTreeNode *node);
static void          _tree_get_path (FSBrowser *fb, GtkCTreeNode *node, char **buffer, int *buffer_size);
static void          tree_expand_node (FSBrowser *fb, GtkCTreeNode *node);
static void          tree_collapse (GtkCTree *tree, GtkCTreeNode *node);
static GtkCTreeNode* tree_find_file (FSBrowser *fb, GtkCTreeNode *parent, char *file_name);

static void          fb_init_dnd (FSBrowser *fb);

enum {
    SELECT_FILE,
    LAST_SIGNAL
};

static GtkCTreeClass                *parent_class = NULL;

static guint                        fs_browser_signals [LAST_SIGNAL] = {0};

GtkType
fs_browser_get_type (void) {
  static GtkType type = 0;
  
  if ( !type) {
    GtkTypeInfo fs_browser_info = {
      "FSBrowser",
      sizeof( FSBrowser),
      sizeof( FSBrowserClass),
      (GtkClassInitFunc) fs_browser_class_init,
      (GtkObjectInitFunc) fs_browser_init,
      /* reserved_1 */ NULL,
      /* reserved_2 */ NULL,
      (GtkClassInitFunc) NULL,
    };
    
    type = gtk_type_unique (GTK_TYPE_CTREE, &fs_browser_info);
  }
  
  return type;
}


static void        
fs_browser_class_init (FSBrowserClass *klass) {
  GtkObjectClass *object_class;
  GtkCTreeClass  *ctree_class;
  
  object_class = (GtkObjectClass *) klass;
  ctree_class  = (GtkCTreeClass*) klass;
  
  parent_class = gtk_type_class (gtk_ctree_get_type());
  
  fs_browser_signals [SELECT_FILE] = 
    gtk_signal_new ("select_file",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET ( FSBrowserClass, select_file),
		    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  
  gtk_object_class_add_signals (object_class, fs_browser_signals, LAST_SIGNAL);
  
  ctree_class->tree_expand         = fs_browser_tree_expand;
  ctree_class->tree_collapse       = fs_browser_tree_collapse;
  ctree_class->tree_select_row     = fs_browser_tree_select_row;
  
  klass->select_file               = NULL;
  
}


static void        
fs_browser_init (FSBrowser *fb) {
  
  *fb->path_buffer = 0;

}

GtkWidget*
fs_browser_new (char *start_path) {
  GtkWidget *widget;
  FSBrowser *fb;
  GtkCTreeNode *node = NULL;
  char *root = "/", *rp, *tmp;
  
  g_return_val_if_fail (start_path != NULL, NULL);
  
  widget = gtk_type_new (fs_browser_get_type());
  gtk_ctree_construct (GTK_CTREE(widget), 1, 0, NULL);
  
  fb = FS_BROWSER(widget);
  
  node = gtk_ctree_insert_node (GTK_CTREE(fb),
				NULL, NULL,
				&root, CTREE_SPACING,
				NULL, NULL, NULL, NULL,
				FALSE, TRUE);
  tree_expand_node(fb, node);
   
  rp = g_strdup (start_path);
  tmp = strtok (rp, "/");
  while (tmp) {
    node = tree_find_file (fb, node, tmp);
    if (!node)
      break;
    gtk_ctree_expand (GTK_CTREE( fb), node);
    tmp = strtok (NULL, "/");
  }
    
  g_free (rp);
  
  fb_init_dnd (fb);

  return widget;
}

static void
fs_browser_tree_select_row (GtkCTree *tree, GtkCTreeNode *node, gint column) {
  FSBrowser *fb;
  
  g_return_if_fail (tree && node);
  
  if (parent_class->tree_select_row)
    (*parent_class->tree_select_row) (tree, node, column);
  
  fb = FS_BROWSER(tree);
  
  tree_get_path (fb, node);

  gtk_signal_emit (GTK_OBJECT(fb), 
		   fs_browser_signals [SELECT_FILE],
		   fb->path_buffer);
}

static void
fs_browser_tree_expand (GtkCTree *tree, GtkCTreeNode *node) {
  GtkAdjustment *h_adjustment;
  gfloat row_align;

  g_return_if_fail (tree && node);
    
  tree_expand_node (FS_BROWSER(tree), node);
  
  if (parent_class->tree_expand)
    (*parent_class->tree_expand) (tree, node);

  /* FIXME : buggy */
  h_adjustment = gtk_clist_get_hadjustment (GTK_CLIST(tree));
  if (h_adjustment && h_adjustment->upper != 0.0) {
    row_align    = (float) (tree->tree_indent * GTK_CTREE_ROW(node)->level) / h_adjustment->upper;
    gtk_ctree_node_moveto (tree, node, tree->tree_column, 0.5, row_align);
  }
}

static void          
fs_browser_tree_collapse (GtkCTree *tree, GtkCTreeNode *node) {

  if (parent_class->tree_collapse)
    (*parent_class->tree_collapse) (tree, node);

}

static void
tree_collapse (GtkCTree *tree, GtkCTreeNode *node) {
  GtkCTreeNode *child;

  gtk_clist_freeze (GTK_CLIST(tree));
  child = GTK_CTREE_ROW(node)->children;

  while (child) {
    gtk_ctree_remove_node (GTK_CTREE(tree), child);
    child = GTK_CTREE_ROW(node)->children;
  } 

  gtk_clist_thaw (GTK_CLIST(tree));

}

static void
tree_expand_node (FSBrowser *fb, GtkCTreeNode *node) {
    GtkCTreeNode *child;
    DIR *dir;
    struct stat stat_buff;
    struct dirent *entry;
    gint  has_subdirs;
    char *old_path;
    char *subdir = "?";
    char *file_name;

    old_path = getcwd (NULL, 0);
    if (!old_path) 
      return;

    tree_get_path (fb, node);
    if (chdir (fb->path_buffer)) {
      g_free (old_path);
      return;
    }

    dir = opendir (".");
    if (!dir) {
      chdir (old_path);
      g_free (old_path);
      return;
    }

    gtk_clist_freeze (GTK_CLIST(fb));

    tree_collapse (GTK_CTREE(fb), node);

    child = NULL;
    while ( (entry =  readdir (dir))) {
      if (strcmp (entry->d_name, ".") && strcmp (entry->d_name, "..") &&
	  !stat (entry->d_name, &stat_buff) &&
	  S_ISDIR(stat_buff.st_mode)) {
	
	has_subdirs = stat_buff.st_nlink > 2;

	file_name = entry->d_name;
	child = gtk_ctree_insert_node (GTK_CTREE(fb),
				       node, child,
				       &file_name,
				       CTREE_SPACING,
				       NULL, NULL, NULL, NULL,
				       !has_subdirs, 0);

	if (has_subdirs) 
	  gtk_ctree_insert_node (GTK_CTREE(fb),
				 child, NULL,
				 &subdir,
				 CTREE_SPACING,
				 NULL, NULL, NULL, NULL,
				 0, 0);
      }
    }

    closedir (dir);
    chdir (old_path);
    g_free (old_path);

    gtk_ctree_sort_node (GTK_CTREE(fb), node);

    gtk_clist_thaw (GTK_CLIST(fb));

}
    

static GtkCTreeNode* 
tree_find_file (FSBrowser *fb, GtkCTreeNode *parent, char *file_name) {
  GtkCTreeNode *child;
  
  g_return_val_if_fail (fb && parent && file_name, NULL);
  
  child = GTK_CTREE_ROW(parent)->children;
  while (child) {
    if (!strcmp (NODE_TEXT(GTK_CTREE(fb), child), file_name))
      return child;
    
    child = GTK_CTREE_ROW(child)->sibling;
  }
  
  return NULL;
}


static void
tree_get_path (FSBrowser *fb, GtkCTreeNode *node) {
  int buffer_size;
  char *buffer;

  buffer_size = MAX_PATH_LENGTH;
  buffer = fb->path_buffer;

  _tree_get_path (fb, node, &buffer, &buffer_size);
}



static void
_tree_get_path (FSBrowser *fb, GtkCTreeNode *node, char **buffer, int *buffer_size) {
  char *dir_name;
  int l;

  if (!node)
    return;

  _tree_get_path (fb, GTK_CTREE_ROW(node)->parent, buffer, buffer_size);

  dir_name = NODE_TEXT(GTK_CTREE(fb), node);
  if (dir_name) {
    snprintf (*buffer, *buffer_size, "/%s", dir_name);
    l = strlen (dir_name) + 1;
    *buffer += l;
    *buffer_size -= l;
  }
}


static GtkTargetEntry fb_target_table [] = {
  { "STRING"         , 0  , DND_TARGET_STRING },
  { "text/plain"     , 0  , DND_TARGET_STRING }, 
  { "text/uri-list"  , 0  , DND_TARGET_URL }
};

static int fb_targets = sizeof (fb_target_table) / sizeof (GtkTargetEntry);

static void
fb_init_dnd (FSBrowser *fb) {
  g_return_if_fail (fb);

  gtk_drag_source_set (GTK_WIDGET (fb),
		       GDK_BUTTON1_MASK,
		       fb_target_table, fb_targets,
		       GDK_ACTION_COPY /*| GDK_ACTION_MOVE | GDK_ACTION_LINK */);

  gtk_drag_dest_set (GTK_WIDGET (fb),
		     GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
		     fb_target_table, fb_targets,
		     GDK_ACTION_COPY /*| GDK_ACTION_MOVE | GDK_ACTION_LINK */);
}

