
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: zalbum.c,v 1.6 2000/04/28 20:46:34 dr Exp $ 
*/


#include <stdio.h>
#include <string.h>
#include "zalbum.h"
#include "dnd.h"
#include "../lib/codec.h"
#include "../lib/mcache.h"
#include "../lib/draw.h"
#include "../lib/scale.h"

#define ZALBUM_CELL(ptr) ((ZAlbumCell *) (ptr))

#define bw(widget)   GTK_CONTAINER(widget)->border_width

#define CELL_PADDING 4
#define LINE_PADDING 2
#define LINE_HEIGHT(font) ((font)->ascent + (font)->descent + LINE_PADDING)
#define STRING_BUFFER_SIZE 1024

#define CELL_STATE(cell)  (ZALBUM_CELL (cell)->flags & ZALBUM_CELL_SELECTED ? GTK_STATE_SELECTED : GTK_STATE_NORMAL)

typedef void (*DrawFunc) (ZAlbum *album, ZAlbumCell *cell, GdkRectangle *cell_area, GdkRectangle *area);
 
static void               zalbum_class_init (ZAlbumClass *klass);
static void               zalbum_init (ZAlbum *album);

static void               zalbum_finalize (GtkObject *object);

static void               zalbum_clear (ZList *list);
static void               zalbum_size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static void               zalbum_cell_size_request (ZList *list, gpointer cell, GtkRequisition *requisition);
static void               zalbum_draw_cell (ZList *list, gpointer cell, GdkRectangle *cell_area, GdkRectangle *area);
static void               zalbum_draw_list (ZAlbum *album, ZAlbumCell *cell, GdkRectangle *cell_area, GdkRectangle *area);
static void               zalbum_draw_preview (ZAlbum *album, ZAlbumCell *cell, GdkRectangle *cell_area, GdkRectangle *area);
static void               zalbum_prepare_cell (ZAlbum *album, ZAlbumCell *cell, GdkRectangle *cell_area, GdkRectangle *area);
static void               zalbum_cell_draw_focus (ZList *list, gpointer cell, GdkRectangle *cell_area);
static void               zalbum_cell_draw_default (ZList *list, gpointer cell, GdkRectangle *cell_area);
static void               zalbum_cell_select (ZList *list, gpointer cell);
static void               zalbum_cell_unselect (ZList *list, gpointer cell);
static void               make_string (GdkFont *font, char *string, int max_width, char *buffer, int buffer_size);
static void               zalbum_draw_string (GtkWidget *widget, ZAlbumCell *cell, char *string, 
					      int x, int y, int max_width, int center);

static void               zalbum_build_gpengine (ZAlbum *album);
static int                zalbum_next_cell_to_render (ZAlbum *album);
static void               zalbum_render_next_cell (ZAlbum *album);
static void               zalbum_render_callback (GPEngine *gpe, 
						  GPEngineState state, 
						  int scanlines, 
						  ZAlbum *album);

static void               zalbum_init_dnd (ZAlbum *album);
static void               zalbum_drag_data_get (GtkWidget        *widget,
						GdkDragContext   *context,
						GtkSelectionData *selection_data,
						guint             info,
						guint             time);
static void               zalbum_drag_data_received  (GtkWidget          *widget,
						      GdkDragContext     *context,
						      gint                x,
						      gint                y,
						      GtkSelectionData   *data,
						      guint               info,
						      guint               time);

static GtkWidgetClass *   parent_class = NULL;

static ZAlbumColumnInfo   default_cinfo [ZALBUM_COLUMNS] = 
{ 
  { 200, 10, 0 },
  { 80,  10, 0 },
  { 80,  10, 0 }
};

static DrawFunc           
zalbum_draw_funcs [] = {
  zalbum_draw_list,
  zalbum_draw_preview,
  NULL
};

GtkType 
zalbum_get_type () {
  static GtkType type = 0;

  if (!type) {
    static GtkTypeInfo info = {
      "ZAlbum",
      sizeof (ZAlbum),
      sizeof (ZAlbumClass),
      (GtkClassInitFunc) zalbum_class_init,
      (GtkObjectInitFunc) zalbum_init,
      /* reserved 1 */ NULL,
      /* reserved 2 */ NULL,
      (GtkClassInitFunc) NULL
    };

    type = gtk_type_unique (zlist_get_type (), &info);
  }

  return type;
}

static void
zalbum_class_init (ZAlbumClass *klass) {
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  ZListClass *zlist_class;

  parent_class = gtk_type_class (zlist_get_type ());

  object_class = (GtkObjectClass *) klass;
  widget_class = (GtkWidgetClass *) klass;
  zlist_class  = (ZListClass *) klass;

  object_class->finalize           = zalbum_finalize;

  widget_class->size_allocate      = zalbum_size_allocate;
  widget_class->drag_data_get      = zalbum_drag_data_get;
  widget_class->drag_data_received = zalbum_drag_data_received;

  zlist_class->clear               = zalbum_clear;
  zlist_class->cell_draw           = zalbum_draw_cell;
  zlist_class->cell_size_request   = zalbum_cell_size_request;
  zlist_class->cell_draw_focus     = zalbum_cell_draw_focus;
  zlist_class->cell_draw_default   = zalbum_cell_draw_default;
  zlist_class->cell_select         = zalbum_cell_select;
  zlist_class->cell_unselect       = zalbum_cell_unselect;
}

static void
zalbum_init (ZAlbum *zalbum) {
  zalbum->mode            = ZALBUM_MODE_LIST;
  zalbum->gpe             = NULL;
  zalbum->render_cell     = 0; 
  zalbum->max_pix_width   = 100;
  zalbum->max_pix_height  = 100;
}

GtkWidget *
zalbum_new (ZAlbumMode mode) {
  ZAlbum *album;

  g_return_val_if_fail (mode < ZALBUM_MODES, NULL);

  album = (ZAlbum *) gtk_type_new (zalbum_get_type ());
  g_return_val_if_fail (album != NULL, NULL);
  
  memcpy (&album->cinfo, &default_cinfo, sizeof (default_cinfo));

  zlist_construct (ZLIST(album), 0); 
  zlist_set_selection_mode (ZLIST(album), GTK_SELECTION_EXTENDED);
  zalbum_set_mode (album, mode);
  zalbum_init_dnd (album);

  return (GtkWidget *) album;
}

static void
zalbum_finalize (GtkObject *object) {

  /* 
     couldn't be done by the parent
     because when the finalize signal
     is received the object's refcount
     is 0 and so calling gtk_signal_emit
     is excluded
  */
  zalbum_clear (ZLIST (object));
}

void
zalbum_add (ZAlbum *album, Image *image) {
  ZAlbumCell *cell;
  g_return_if_fail (album && image);

  cell            = g_new (ZAlbumCell, 1);
  cell->image     = image_ref (image);
  cell->ipix      = NULL;
  cell->flags     = 0;

  zlist_add (ZLIST(album), cell);

  if (!album->gpe || !gpengine_is_running (album->gpe))
    zalbum_render_next_cell (album);
}

void
zalbum_set_mode (ZAlbum *album, ZAlbumMode mode) {
  GtkRequisition requisition;
  g_return_if_fail (album != NULL && mode >= 0 && mode < ZALBUM_MODES);

  if (album->mode == mode)
    return;

  album->mode = mode;
  zalbum_cell_size_request (ZLIST(album), NULL, &requisition);

  switch (mode) {
  case ZALBUM_MODE_LIST:
    zlist_set_cell_padding (ZLIST(album), 0, 0);
    zlist_set_cell_size (ZLIST(album), -1, requisition.height);
    zlist_set_1 (ZLIST(album), 1);
    break;
  case ZALBUM_MODE_PREVIEW:
    zlist_set_cell_padding (ZLIST(album), 4, 4);
    zlist_set_1 (ZLIST(album), 0);
    zlist_set_cell_size (ZLIST(album), requisition.width, requisition.height);
    break;
  default:
    break;
  }
}

gfloat
zalbum_get_progress (ZAlbum *album) {
  
  if (!ZLIST (album)->cell_count)
    return 1.0;

  return ((gfloat) album->rendered_cells +
	  gpengine_get_progress (album->gpe)) / ZLIST (album)->cell_count;
}

static void
zalbum_clear (ZList *list) {
  ZAlbumCell *cell;
  int i;

  g_return_if_fail (list);

  if (ZALBUM(list)->gpe) {
    gpengine_abort (ZALBUM(list)->gpe);
    /*
    module_free (ZALBUM(list)->gpe);
    ZALBUM(list)->gpe = NULL;
    */
  }
  
  ZALBUM (list)->rendered_cells = 0;
  ZALBUM (list)->render_cell    = 0;

  for (i = 0; i < list->cell_count; i++) {
    cell = ZLIST_CELL_FROM_INDEX (list, i);
    if (cell->ipix) image_pixmap_unref (cell->image, cell->ipix);
    image_unref (cell->image);
    g_free (cell);
  }
}

static void
zalbum_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
  ZAlbum *album;
  int i, w = 0;

  album = ZALBUM (widget);

  /* expand the first column */
  for (i = 1; i < ZALBUM_COLUMNS; i++) {
    w += album->cinfo [i].pad;
    w += album->cinfo [i].width;
  }
    
  album->cinfo [0].width = allocation->width - w - album->cinfo [0].pad;

  if (parent_class && parent_class->size_allocate)
    (* parent_class->size_allocate) (widget, allocation);
  
}

static void
zalbum_cell_size_request (ZList *list, gpointer cell, GtkRequisition *requisition) {
  ZAlbum *album;
  int i, w = 0;

  g_return_if_fail (list && requisition);

  album = ZALBUM (list);

  switch (album->mode) {
  case ZALBUM_MODE_LIST:
    for (i = 0; i < ZALBUM_COLUMNS; i++) {
      w += album->cinfo [i].pad;
      w += album->cinfo [i].width;
    }
    requisition->width  = w + 2 * CELL_PADDING;
    requisition->height = LINE_HEIGHT (GTK_WIDGET (list)->style->font) + 2 * CELL_PADDING;
    break;

  case ZALBUM_MODE_PREVIEW:
    
    requisition->width  = album->max_pix_width + 2 * CELL_PADDING;
    requisition->height = album->max_pix_height + 
      LINE_HEIGHT(GTK_WIDGET(list)->style->font) + 2 * CELL_PADDING; 
    break;

  default:
  }
}
 
static void
zalbum_draw_cell (ZList *list, gpointer cell, GdkRectangle *cell_area, GdkRectangle *area) {
  g_return_if_fail (list && cell);

  (* zalbum_draw_funcs [ZALBUM(list)->mode]) ((ZAlbum *) list, (ZAlbumCell *) cell, cell_area, area);
}

static void
zalbum_draw_list (ZAlbum *album, ZAlbumCell *cell, GdkRectangle *cell_area, GdkRectangle *area) {
  GtkWidget *widget;
  char buff [64];
  int xdest, ydest, size;
  
  widget = GTK_WIDGET(album);

  zalbum_prepare_cell (album, cell, cell_area, area);

  ydest = cell_area->y + widget->style->font->ascent + LINE_PADDING / 2;
  xdest = cell_area->x + album->cinfo [0].pad;

  zalbum_draw_string (widget, cell, cell->image->file_name, 
		      xdest, ydest, album->cinfo [0].width, 0);

  xdest += album->cinfo [0].width + album->cinfo [1].pad;

  snprintf (buff, 64, "%dx%dx%d", cell->image->width, cell->image->height, cell->image->bpp * 8);
  zalbum_draw_string (widget, cell, buff, 
		      xdest, ydest, album->cinfo [1].width, 0);

  xdest += album->cinfo [1].width + album->cinfo [2].pad;
  
  size = cell->image->file_size;
  if (size >= 0x100000)
    snprintf (buff, 64, "%d M", cell->image->file_size >> 20);
  else if (size >= 0x400)
    snprintf (buff, 64, "%d K", cell->image->file_size >> 10);
  else
    snprintf (buff, 64, "%d b", cell->image->file_size);

  zalbum_draw_string (widget, cell, buff, 
		      xdest, ydest, album->cinfo [2].width, 0);

}

static void
zalbum_draw_preview (ZAlbum *album, ZAlbumCell *cell, GdkRectangle *cell_area, GdkRectangle *area) {
  GtkWidget *widget;
  GdkRectangle intersect_area;
  int text_area_height, ydest;
  int xpad, ypad;

  g_return_if_fail (album && cell && cell_area);

  widget = GTK_WIDGET (album);
  if (!GTK_WIDGET_DRAWABLE (widget))
    return;

  if (!area)
    area = cell_area;

  zalbum_prepare_cell (album, cell, cell_area, area);

  text_area_height = LINE_HEIGHT(widget->style->font);

  if (cell->ipix) {
    GdkRectangle pixmap_area;
    int w, h;

    w    = cell->ipix->width;
    h    = cell->ipix->height;
    xpad = (cell_area->width  - w)  / 2;
    ypad = (cell_area->height - text_area_height - h) / 2;

    if (xpad < 0) {
      w += xpad;
      xpad = 0;
    }

    if (ypad < 0) {
      h += ypad;
      ypad = 0;
    }

    pixmap_area.x = cell_area->x + xpad;
    pixmap_area.y = cell_area->y + ypad;
    pixmap_area.width  = w;
    pixmap_area.height = h;
    if (gdk_rectangle_intersect (area, &pixmap_area, &intersect_area))
     gdk_draw_pixmap (widget->window,
		      widget->style->fg_gc [GTK_STATE_NORMAL],
		      cell->ipix->pixmap,
		      intersect_area.x - pixmap_area.x,
		      intersect_area.y - pixmap_area.y,
		      intersect_area.x, intersect_area.y,
		      intersect_area.width, intersect_area.height);
  }

  ydest = (cell_area->y + cell_area->height - text_area_height) + widget->style->font->ascent + LINE_PADDING;
  zalbum_draw_string (widget, cell, cell->image->file_name, 
		      cell_area->x, ydest, cell_area->width, 1);

  /*
  ydest += LINE_HEIGHT(widget->style->font);

  snprintf (buff, 64, "%dx%dx%d", cell->image->width, cell->image->height, cell->image->bpp * 8);

  zalbum_draw_string (widget, cell, buff, 
		      cell_area->x, ydest, cell_area->width, 1);
  */
}

static void
make_string (GdkFont *font, char *string, int max_width, char *buffer, int buffer_size) {
  int s, e, l, m = 0;

  s = 0;
  e = l = strlen (string);
  while (e - s > 1) {
    m = s + (e - s) / 2;
    if (gdk_text_width (font, string, m) > max_width) 
      e = m;
    else
      s = m;
  }

  if (m + 4 > buffer_size)
    m = buffer_size - 4;
 
  if (m < l) {
    if (m < 4) { *buffer = 0; return;};
    memcpy (buffer, string, m - 3);
    buffer [m-1] = '.'; buffer [m-2] = '.'; buffer [m-3] = '.';
  } else
    memcpy (buffer, string, m);

  buffer [m] = 0;
}

static void
zalbum_draw_string (GtkWidget *widget, ZAlbumCell *cell, char *string, 
		    int x, int y, int max_width, int center) {
  char buffer [STRING_BUFFER_SIZE];
  int x_pad;

  x_pad = (max_width - gdk_string_width (widget->style->font, string)) / 2;
  if (x_pad < 0) {
    make_string (widget->style->font, string, max_width, buffer, STRING_BUFFER_SIZE);
    string = buffer;
    x_pad = 0;
  }

  if (!center)
    x_pad = 0;

  gdk_draw_string (widget->window,
		   widget->style->font,
		   widget->style->fg_gc [CELL_STATE(cell)],
		   x + x_pad, y, string);
}


static void
zalbum_prepare_cell (ZAlbum *album, ZAlbumCell *cell, GdkRectangle *cell_area, GdkRectangle *area) {
  GtkWidget *widget;
  GdkRectangle intersect_area;

  widget = GTK_WIDGET(album);
  /*
  gdk_draw_rectangle (widget->window,
		      widget->style->dark_gc [GTK_STATE_NORMAL], 
		      FALSE, cell_area->x, cell_area->y,
		      cell_area->width - 1, cell_area->height - 1);

  cell_area->x += 1; cell_area->y += 1; cell_area->width -= 2; cell_area->height -= 2;

  gdk_draw_rectangle (widget->window,
		      widget->style->light_gc [GTK_STATE_NORMAL],
		      FALSE, cell_area->x, cell_area->y,
		      cell_area->width, cell_area->height);
    
  cell_area->x += CELL_PADDING - 1; cell_area->y += CELL_PADDING - 1; 
  cell_area->width -= 2 * CELL_PADDING - 2; cell_area->height -= 2 * CELL_PADDING - 2;
  */

  if (gdk_rectangle_intersect (area, cell_area, &intersect_area))
    gdk_draw_rectangle (widget->window,
			widget->style->bg_gc [CELL_STATE(cell)], 
			TRUE, intersect_area.x, intersect_area.y,
			intersect_area.width, intersect_area.height);

  
  gtk_draw_shadow (widget->style, widget->window, 
		   CELL_STATE (cell),
		   GTK_SHADOW_OUT, 
		   cell_area->x, cell_area->y,
		   cell_area->width, cell_area->height);
  
  cell_area->x += CELL_PADDING; cell_area->y += CELL_PADDING;
  cell_area->width -= 2 * CELL_PADDING; cell_area->height -= 2 * CELL_PADDING;
}

static void               
zalbum_cell_draw_focus (ZList *list, gpointer cell, GdkRectangle *cell_area) {

  gtk_draw_shadow (GTK_WIDGET (list)->style, GTK_WIDGET (list)->window, 
		   CELL_STATE (cell),
		   GTK_SHADOW_IN, 
		   cell_area->x, cell_area->y,
		   cell_area->width, cell_area->height);
  /*
  gdk_draw_rectangle (GTK_WIDGET(list)->window,
		      GTK_WIDGET(list)->style->black_gc, 
		      FALSE, cell_area->x, cell_area->y, 
		      cell_area->width - 1, cell_area->height - 1); */
}

static void
zalbum_cell_draw_default (ZList *list, gpointer cell, GdkRectangle *cell_area) {
  gtk_draw_shadow (GTK_WIDGET (list)->style, GTK_WIDGET (list)->window, 
		   CELL_STATE (cell),
		   GTK_SHADOW_OUT, 
		   cell_area->x, cell_area->y,
		   cell_area->width, cell_area->height);
  /*
  gdk_draw_rectangle (GTK_WIDGET(list)->window,
		      GTK_WIDGET(list)->style->white_gc, //dark_gc [GTK_STATE_NORMAL], 
		      FALSE, cell_area->x, cell_area->y,
		      cell_area->width - 1, cell_area->height - 1);
  */
}

static void               
zalbum_cell_select (ZList *list, gpointer cell) {
  ZALBUM_CELL (cell)->flags |= ZALBUM_CELL_SELECTED;
}

static void               
zalbum_cell_unselect (ZList *list, gpointer cell) {
  ZALBUM_CELL (cell)->flags &= ~ZALBUM_CELL_SELECTED;
}

/* 
 * 
 *
 */

static void
zalbum_build_gpengine (ZAlbum *album) {
  Module *module;

  g_return_if_fail (album);

  album->gpe = gpengine_new ();

  module     = codec_new (CODEC_READ_IMAGE, 100);
  gpengine_add_filter (album->gpe, FILTER (module));

  module     = scale_new (SCALE_DONT_MAGNIFY, 100, 100, 
			  &GTK_WIDGET (album)->style->bg [GTK_STATE_NORMAL]);
  gpengine_add_filter (album->gpe, FILTER (module));
  
  module     = mcache_new (1);
  gpengine_add_filter (album->gpe, FILTER (module));
  
  module     = draw_new (NULL, GTK_WIDGET (album)->style->black_gc, 1, 
			 gdk_visual_get_best_depth ());
  gpengine_add_filter (album->gpe, FILTER (module));
}

static int
zalbum_next_cell_to_render (ZAlbum *album) {
  ZAlbumCell *cell;
  int i, first;

  first = zlist_cell_index_from_xy (ZLIST (album), 
				    ZLIST (album)->x_pad, 
				    ZLIST (album)->y_pad);
  if (first < 0)
    first = album->render_cell;
  if (first >= ZLIST(album)->cell_count)
    first = 0;

  /* XXX quite inefficient */
  for (i = first; i < ZLIST(album)->cell_count; i++) {
    cell = ZLIST_CELL_FROM_INDEX(album, i);
    if (!(cell->flags & ZALBUM_CELL_RENDERED))
      return i;
  }

  for (i = 0; i < first; i ++) {
    cell = ZLIST_CELL_FROM_INDEX(album, i);
    if(!(cell->flags & ZALBUM_CELL_RENDERED))
      return i;
  }

  return -1;
}

static void
zalbum_render_next_cell (ZAlbum *album) {
  ZAlbumCell *cell;
  int index;

  index = zalbum_next_cell_to_render (album);
  if (index < 0)
    return;

  album->render_cell = index;
  cell = ZLIST_CELL_FROM_INDEX (album, index);

  if (!album->gpe)
    zalbum_build_gpengine (album);

  gpengine_start (album->gpe, 0, cell->image, NULL, 
		  (GPEngineCallback) zalbum_render_callback, album);
}

static void
zalbum_render_callback (GPEngine *gpe, GPEngineState state, int scanlines, ZAlbum *album) {
  ParamValue *value;
  ZAlbumCell *cell;

  if (state == GPENGINE_INITIALIZED) {

    cell = ZLIST_CELL_FROM_INDEX (album, album->render_cell);
    value = gpengine_get_param_value (gpe, -1, "pixmap", PARAM_TYPE_POINTER);

    if (value) {
      cell->ipix = value->v_pointer;
      image_pixmap_ref (cell->image, cell->ipix);
    }

  } else if (state == GPENGINE_FINISHED) {

    album->rendered_cells ++;

    cell = ZLIST_CELL_FROM_INDEX (album, album->render_cell);
    cell->flags |= ZALBUM_CELL_RENDERED;

    if (cell->ipix) {
      album->max_pix_width  = MAX (album->max_pix_width, cell->ipix->width);
      album->max_pix_height = MAX (album->max_pix_height, cell->ipix->height);
      
      if (!zlist_update_cell_size (ZLIST(album), (gpointer) cell))
	zlist_draw_cell (ZLIST(album), album->render_cell);
    }

    zalbum_render_next_cell (album);
  } else if (state == GPENGINE_ABORTED)
    zalbum_render_next_cell (album);
}

/*
 *
 *
 */

static GtkTargetEntry zalbum_target_table [] = {
  { "STRING"         , 0  , DND_TARGET_STRING },
  { "text/plain"     , 0  , DND_TARGET_STRING }, 
  { "text/uri-list"  , 0  , DND_TARGET_URL }
};

static int zalbum_targets = sizeof (zalbum_target_table) / sizeof (GtkTargetEntry);

static void
zalbum_init_dnd (ZAlbum *album) {
  g_return_if_fail (album);

  ZLIST (album)->flags |= ZLIST_USES_DND;

  gtk_drag_source_set (GTK_WIDGET (album),
		       GDK_BUTTON1_MASK,
		       zalbum_target_table, zalbum_targets,
		       GDK_ACTION_COPY /*| GDK_ACTION_MOVE | GDK_ACTION_LINK */);

  gtk_drag_dest_set (GTK_WIDGET (album),
		     GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
		     zalbum_target_table, zalbum_targets,
		     GDK_ACTION_COPY /*| GDK_ACTION_MOVE | GDK_ACTION_LINK */);
}

static void
zalbum_drag_data_get (GtkWidget        *widget,
		      GdkDragContext   *context,
		      GtkSelectionData *selection_data,
		      guint             info,
		      guint             time) {

  ZAlbum *album;
  ZAlbumCell *cell;
  GList *list, *images = NULL;

  g_return_if_fail (widget && IS_ZALBUM (widget));

  album = (ZAlbum *) widget;

  list = ZLIST (album)->selection;
  while (list) {
    cell = (ZAlbumCell *) ZLIST_CELL_FROM_INDEX (album, GPOINTER_TO_UINT (list->data));
    images = g_list_append (images, cell->image);
    
    list = list->next;
  }
  
  if (images)
    dnd_set_from_images (selection_data, info, images);

  g_list_free (images);  
}

static void  
zalbum_drag_data_received  (GtkWidget          *widget,
			    GdkDragContext     *context,
			    gint                x,
			    gint                y,
			    GtkSelectionData   *data,
			    guint               info,
			    guint               time) {
  ZAlbum *album;
  Image *image;
  GList *images, *list;

  g_return_if_fail (widget && IS_ZALBUM (widget));

  album = (ZAlbum *) widget;

  images = list = dnd_get_images (data, info);
  
  while (list) {
    image = list->data;
    list = list->next;

    zalbum_add (album, image);
  }

  if (images) {
    g_list_foreach (images, (GFunc) image_unref, NULL);
    g_list_free (images);
  }
}
