
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: zimage.c,v 1.8 2000/04/28 20:46:34 dr Exp $ 
*/

#include <pthread.h>
#include "zimage.h"
#include "../lib/codec.h"
#include "../lib/gpengine.h"
#include "../lib/mcache.h"
#include "../lib/scale.h"

#define AUTO_SCROLL_TIMEOUT 20

static void        zimage_class_init (ZImageClass *class);
static void        zimage_init (ZImage *zimage);
static void        zimage_finalize (GtkObject *object);
static void        zimage_map (GtkWidget *widget);
static void        zimage_unmap (GtkWidget *widget);
static void        zimage_realize (GtkWidget *widget);
static void        zimage_unrealize (GtkWidget *widget);
static void        zimage_size_request (GtkWidget *widget, GtkRequisition *requisition);
static void        zimage_size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static gint        zimage_expose (GtkWidget *widget, GdkEventExpose *event);
static void        zimage_draw (GtkWidget *widget, GdkRectangle *area);
static gint        zimage_button_press (GtkWidget *widget, GdkEventButton *event);
static gint        zimage_motion_notify (GtkWidget *widget, GdkEventMotion *event);
static gint        zimage_do_motion (ZImage *zimage);
static gint        zimage_button_release (GtkWidget *widget, GdkEventButton *event);
static void        zimage_adjust_adjustments (Scrolled *scrolled);
static void        zimage_update_geom (ZImage *zimage);
static void        zimage_load_func (GPEngine *gpe, 
				     GPEngineState state, 
				     int scanlines,
				     ZImage *zimage);

static GtkWidgetClass *parent_class = NULL;

GtkType 
zimage_get_type () {
  static GtkType type = 0;

  if (!type) {
    static GtkTypeInfo zimage_info = {
      "ZImage",
      sizeof (ZImage),
      sizeof (ZImageClass),
      (GtkClassInitFunc) zimage_class_init,
      (GtkObjectInitFunc) zimage_init,
      /* reserved_1 */ NULL,
      /* reserved_2 */ NULL,
      (GtkClassInitFunc) NULL
    };
    
    type = gtk_type_unique (SCROLLED_TYPE, &zimage_info);
  }

  return type;
}

static void
zimage_class_init (ZImageClass *class) {
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  ScrolledClass  *scrolled_class;

  parent_class     = gtk_type_class (SCROLLED_TYPE);
  object_class     = (GtkObjectClass *) class;
  widget_class     = (GtkWidgetClass *) class;
  scrolled_class   = (ScrolledClass *) class;

  object_class->finalize                = zimage_finalize;

  widget_class->map                     = zimage_map;
  widget_class->unmap                   = zimage_unmap;
  widget_class->realize                 = zimage_realize;
  widget_class->unrealize               = zimage_unrealize;
  widget_class->expose_event            = zimage_expose;
  widget_class->size_request            = zimage_size_request;
  widget_class->size_allocate           = zimage_size_allocate;
  widget_class->draw                    = zimage_draw;
  widget_class->button_press_event      = zimage_button_press;
  widget_class->motion_notify_event     = zimage_motion_notify;
  widget_class->button_release_event    = zimage_button_release;

  scrolled_class->adjust_adjustments    = zimage_adjust_adjustments;
}

static void
zimage_init (ZImage *zimage) {

}

GtkWidget *
zimage_new (Image *image) {
  ZImage *zimage;
  Module *module;

  zimage = (ZImage *) gtk_type_new (zimage_get_type ());

  zimage->image       = NULL;
  zimage->next_image  = NULL;
  zimage->da          = NULL;
  zimage_set_image (zimage, image);

  zimage->gpe_load = gpengine_new ();

  module = codec_new (CODEC_READ_IMAGE, -1);
  gpengine_add_filter (zimage->gpe_load, FILTER (module));

  module = mcache_new (0);
  gpengine_add_filter (zimage->gpe_load, FILTER (module));

  return (GtkWidget *) zimage;
}

void
zimage_set_image (ZImage *zimage, Image *image) {
  ImageCache *cache;
  GtkAdjustment *adj;

  g_return_if_fail (zimage);

  if (zimage->image == image)
    return;

  if (gpengine_is_running (zimage->gpe_load)) {
    if (zimage->next_image)
      image_unref (zimage->next_image);
    
    zimage->next_image = image ? image_ref (image) : NULL;
    gpengine_abort (zimage->gpe_load);
    
    /* 
       wait until the GPENGINE_ABORTED status is
       signaled
    */
    return; 
  }

  if (zimage->image) {
    image_unref (zimage->image);
    zimage->image = NULL;
  }

  zimage->image       = image ? image_ref (image) : NULL;
  
  if (image) {
    cache = image_cache_get (image, -1, -1);
   
    if (cache) {
      if (zimage->da)
	draw_area_set_image (zimage->da, zimage->image);

      zimage_update_geom (zimage);

      gtk_widget_draw (GTK_WIDGET (zimage), NULL);

    } else if (zimage->da) /* ie widget is realized */
      gpengine_start (zimage->gpe_load, GPENGINE_RUN_INTERACTIVE, image, NULL,
		      (GPEngineCallback) zimage_load_func, zimage);
  }

  if (GTK_WIDGET_DRAWABLE (zimage)) {
    scrolled_freeze (SCROLLED (zimage));
    zimage_adjust_adjustments (SCROLLED (zimage));
    
    adj = SCROLLED (zimage)->h_adjustment;
    if (adj) {
      adj->value = 0;
      gtk_signal_emit_by_name (GTK_OBJECT (adj), "value_changed");
    }
    
    adj = SCROLLED (zimage)->v_adjustment;
    if (adj) {
      adj->value = 0;
      gtk_signal_emit_by_name (GTK_OBJECT (adj), "value_changed");
    }
    
    scrolled_thawn (SCROLLED (zimage));
  }
}

int
zimage_loaded (ZImage *zimage) {
  g_return_val_if_fail (zimage, 0);

  return !gpengine_is_running (zimage->gpe_load);
}

/* BUG: zimage_load_func may be called after a call
   to this function. This bug is a consequence of
   the gpengine_free bug and both of this bugs
   result from the fact that after a call
   to gpengine_abort, the thread is may still be alive
*/
static void
zimage_finalize (GtkObject *object) {
  ZImage *zimage;

  zimage = (ZImage *) object;

  if (zimage->gpe_load) 
    module_free (MODULE (zimage->gpe_load));

  if (zimage->da)
    module_free ((Module *) zimage->da);

  if (zimage->image)
    image_unref (zimage->image);
}

static void        
zimage_map (GtkWidget *widget) {
  GTK_WIDGET_SET_FLAGS (widget, GTK_MAPPED);
  gdk_window_show (widget->window);
}

static void        
zimage_unmap (GtkWidget *widget) {
  GTK_WIDGET_UNSET_FLAGS (widget, GTK_MAPPED);
  gdk_window_hide (widget->window);
}

static void        
zimage_realize (GtkWidget *widget) {
  ZImage *zimage;
  GdkWindowAttr attributes;
  gint attributes_mask;

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  attributes.window_type   = GDK_WINDOW_CHILD;
  attributes.x             = widget->allocation.x;
  attributes.y             = widget->allocation.y;
  attributes.width         = widget->allocation.width;
  attributes.height        = widget->allocation.height;
  attributes.wclass        = GDK_INPUT_OUTPUT;
  attributes.visual        = gtk_widget_get_visual (widget);
  attributes.colormap      = gtk_widget_get_colormap (widget);
  attributes.event_mask    = gtk_widget_get_events (widget);
  attributes.event_mask    |= (GDK_EXPOSURE_MASK |
			       GDK_BUTTON_PRESS_MASK |
			       GDK_BUTTON_RELEASE_MASK |
			       GDK_POINTER_MOTION_MASK |
			       GDK_KEY_PRESS_MASK);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
    
  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
				   &attributes, attributes_mask);
  gdk_window_set_user_data (widget->window, widget);
  widget->style = gtk_style_attach (widget->style, widget->window);

  gdk_window_set_background (widget->window,
			     &widget->style->black);

  scrolled_realize (SCROLLED (widget));


  zimage = (ZImage *) widget;
  zimage->da = (DrawArea *) draw_area_new (widget->window, 
					   widget->style->black_gc,
					   &widget->style->black);

  if (zimage->image)
    gpengine_start (zimage->gpe_load, GPENGINE_RUN_INTERACTIVE,
		    zimage->image, NULL,
		    (GPEngineCallback) zimage_load_func,
		    zimage);
}

static void
zimage_unrealize (GtkWidget *widget) {

  scrolled_unrealize (SCROLLED (widget));

  if (parent_class->unrealize)
    (* parent_class->unrealize) (widget);
}

static void        
zimage_size_request (GtkWidget *widget, GtkRequisition *requisition) {
  ZImage *zimage;
  
  zimage = (ZImage *) widget;

  if (zimage->image) {
    if (zimage->da) {
      requisition->width       = DRAW_AREA_APPLY_ZOOM (zimage->da, zimage->image->width);
      requisition->height      = DRAW_AREA_APPLY_ZOOM (zimage->da, zimage->image->height);
    } else {
      requisition->width       = zimage->image->width;
      requisition->height      = zimage->image->height;
    }

  } else {
    requisition->width = requisition->height = 100;
  }

}

static void        
zimage_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
  ZImage *zimage;
  
  zimage = (ZImage *) widget;

  widget->allocation = *allocation;

  if (GTK_WIDGET_REALIZED (widget))
    gdk_window_move_resize (widget->window,
			    allocation->x, allocation->y,
			    allocation->width, allocation->height);


  zimage_update_geom (zimage);

  zimage_adjust_adjustments (SCROLLED (zimage));
}

static gint        
zimage_expose (GtkWidget *widget, GdkEventExpose *event) {
  
  if (GTK_WIDGET_DRAWABLE (widget) && event->window == widget->window)
    zimage_draw (widget, &event->area);

  return FALSE;
}

static void        
zimage_draw (GtkWidget *widget, GdkRectangle *area) {
  ZImage *zimage;
  GdkRectangle widget_area, intersect_area;
  int c;

  zimage = (ZImage *) widget;

  if (!GTK_WIDGET_DRAWABLE (widget))
    return;

  widget_area.x      = widget_area.y = 0;
  widget_area.width  = widget->allocation.width;
  widget_area.height = widget->allocation.height;

  if (!area)
    area = &widget_area;

  if (!zimage->image) {
    gdk_window_clear_area (widget->window, 
			   area->x, area->y, 
			   area->width, area->height);
    return;
  }

  if (!gdk_rectangle_intersect (&widget_area, area, &intersect_area))
    return;

  area = &intersect_area;

  /* clear widget padding */
  c = zimage->x_pad - area->x;
  if (c > 0)
    gdk_window_clear_area (widget->window, area->x, area->y, c, area->height);

  c = area->x + area->width - SCROLLED_X (widget, zimage->width + zimage->x_pad);
  if (c > 0)
    gdk_window_clear_area (widget->window, 
			   SCROLLED_X (widget, zimage->width + zimage->x_pad), 
			   area->y, c, area->height);

  c = zimage->y_pad - area->y;
  if (c > 0)
    gdk_window_clear_area (widget->window, area->x, area->y, area->width, c);
  
  c = area->y + area->height - SCROLLED_Y (widget, zimage->height + zimage->y_pad);
  if (c > 0)
    gdk_window_clear_area (widget->window, area->x, 
			   SCROLLED_Y (widget, zimage->height + zimage->y_pad), 
			   area->width, c);


  draw_area_draw (zimage->da,
		  SCROLLED_VX (widget, area->x - zimage->x_pad), 
		  SCROLLED_VY (widget, area->y - zimage->y_pad),
		  area);
}

static gint        
zimage_button_press (GtkWidget *widget, GdkEventButton *event) {
  ZImage *zimage;
  GtkAdjustment *adj;
  double scale_factor;
  int x, y, max;

  zimage = (ZImage *) widget;

  if (!zimage->image || event->button == 3)
    return FALSE;

  if (event->button == 2) {
    
    gdk_window_get_root_origin (widget->window, &x, &y);

    gdk_window_get_pointer (widget->window,
			    &zimage->lx, &zimage->ly, NULL);

    zimage->lx += x;
    zimage->ly += y;

    if (!GTK_WIDGET_HAS_FOCUS (widget))
      gtk_widget_grab_focus (widget);

    if (gdk_pointer_grab (widget->window, TRUE,
			  GDK_BUTTON2_MOTION_MASK |
			  GDK_BUTTON_RELEASE_MASK,
			  NULL, NULL, event->time))
      return FALSE;
  
    gtk_grab_add (widget);

    return FALSE;
  }

  gdk_window_get_pointer (widget->window, &x, &y, NULL);
  x += SCROLLED_VX (widget, 0);
  y += SCROLLED_VY (widget, 0);
  
  x = DRAW_AREA_APPLY_IZOOM (zimage->da, x);
  y = DRAW_AREA_APPLY_IZOOM (zimage->da, y);

  if (event->state & GDK_SHIFT_MASK)
    draw_area_zoom_out (zimage->da);
  else 
    draw_area_zoom_in (zimage->da);
  

  scale_factor = DRAW_AREA_APPLY_ZOOM (zimage->da, 1);
  zimage_update_geom (zimage);

  zimage_adjust_adjustments (SCROLLED (widget));

  scrolled_freeze (SCROLLED (zimage));
  adj = SCROLLED (widget)->h_adjustment;
  if (adj) {
    x = DRAW_AREA_APPLY_ZOOM (zimage->da, x);
    x -= adj->page_size / 2;

    max = adj->upper - adj->page_size;
    if (max < 0) max = 0;
    x = x < 0 ? 0 : (x > max ? max : x);

    adj->value = x;
    gtk_signal_emit_by_name (GTK_OBJECT (adj), "value_changed");
  }

  adj = SCROLLED (widget)->v_adjustment;
  if (adj) {
    y = DRAW_AREA_APPLY_ZOOM (zimage->da, y);
    y -= adj->page_size / 2;

    max = adj->upper - adj->page_size;
    if (max < 0) max = 0;
    y = y < 0 ? 0 : (y > max ? max : y);

    adj->value = y;
    gtk_signal_emit_by_name (GTK_OBJECT (adj), "value_changed");
  }

  scrolled_thawn (SCROLLED (zimage));

  return FALSE;
}

static gint        
zimage_motion_notify (GtkWidget *widget, GdkEventMotion *event) {

  if (gdk_pointer_is_grabbed () && GTK_WIDGET_HAS_GRAB (widget)) {
    ZIMAGE (widget)->mx = event->x_root;
    ZIMAGE (widget)->my = event->y_root;

    gtk_timeout_add (AUTO_SCROLL_TIMEOUT, (GtkFunction) zimage_do_motion, widget);
  }

  return FALSE;
}

static gint
zimage_do_motion (ZImage *zimage) {
  GtkAdjustment *adj;
  int d;
  
  /* XXX asssumes widget is not destroyed */

  d   = zimage->mx - zimage->lx;
  adj = SCROLLED (zimage)->h_adjustment;
  if (d && adj) {
    zimage->lx = zimage->mx;
    
    if (adj->value + d >= 0 && adj->value + d <= adj->upper - adj->page_size) {
      adj->value += d;
      gtk_signal_emit_by_name (GTK_OBJECT (adj), "value_changed");
    }
  }
  
  d   = zimage->my - zimage->ly;
  adj = SCROLLED (zimage)->v_adjustment; 
  if (d && adj) {
    zimage->ly = zimage->my;
    
    if (adj->value + d >= 0 && adj->value + d <= adj->upper - adj->page_size) {
      adj->value += d;
      gtk_signal_emit_by_name (GTK_OBJECT (adj), "value_changed");
    }
    
  }

  return FALSE;
}

static gint
zimage_button_release (GtkWidget *widget, GdkEventButton *event) {
  
  if (GTK_WIDGET_HAS_GRAB(widget))
    gtk_grab_remove (widget);

  if (gdk_pointer_is_grabbed ())
    gdk_pointer_ungrab (event->time);
  
  return FALSE;
}

static void
zimage_adjust_adjustments (Scrolled *scrolled) {
  ZImage *zimage;
  GtkWidget *widget;
  GtkAdjustment *adj;

  widget = (GtkWidget *) scrolled;
  zimage = (ZImage *) widget;

  if (!GTK_WIDGET_DRAWABLE (widget))
    return;

  adj = scrolled->h_adjustment;
  if (adj) {
    adj->page_size            = widget->allocation.width;
    adj->page_increment       = widget->allocation.width;
    adj->step_increment       = 5;
    adj->lower                = 0;
    adj->upper                = zimage->image ? zimage->width : 0;
    
    gtk_signal_emit_by_name (GTK_OBJECT (adj), "changed");
  }

  adj = scrolled->v_adjustment;
  if (adj) {
    adj->page_size            = widget->allocation.height;
    adj->page_increment       = widget->allocation.height;
    adj->step_increment       = 5;
    adj->lower                = 0;
    adj->upper                = zimage->image ? zimage->height : 0;
    
    gtk_signal_emit_by_name (GTK_OBJECT (adj), "changed");
  }
}

static void
zimage_update_geom (ZImage *zimage) {
  GtkAllocation *alloc;
  
  g_return_if_fail (zimage);

  if (!zimage->image || !zimage->da)
    return;

  alloc = &GTK_WIDGET (zimage)->allocation;

  zimage->width  = DRAW_AREA_APPLY_ZOOM (zimage->da, zimage->image->width);
  zimage->height = DRAW_AREA_APPLY_ZOOM (zimage->da, zimage->image->height);
  
  zimage->x_pad = alloc->width > zimage->width ? 
    (alloc->width - zimage->width) >> 1 : 0;
  zimage->y_pad = alloc->height > zimage->height ? 
    (alloc->height - zimage->height) >> 1 : 0;

}
  
static void         
zimage_load_func (GPEngine *gpe, GPEngineState state, int scanlines, ZImage *zimage) {
  GdkRectangle area;

  if (state == GPENGINE_PROCESSING) {

    area.x = SCROLLED_X (zimage, zimage->x_pad);
    area.y = SCROLLED_Y (zimage, zimage->y_pad + 
			 (int) DRAW_AREA_APPLY_ZOOM (zimage->da, zimage->load_pos));
    area.width  = zimage->width;
    area.height = zimage->da->zoom < 0 ? scanlines : DRAW_AREA_APPLY_ZOOM (zimage->da, scanlines);

    zimage_draw (GTK_WIDGET (zimage), &area);

    zimage->load_pos += scanlines;
  } else if (state == GPENGINE_INITIALIZED) {

    if (!image_cache_get (zimage->image, -1, -1))
      g_warning ("initialized but no cache");

    draw_area_set_image (zimage->da, zimage->image);

    zimage_update_geom (zimage);
    zimage_adjust_adjustments (SCROLLED (zimage));

    zimage->load_pos = 0;
    
    gdk_window_clear_area (GTK_WIDGET (zimage)->window,
			   0, 0,
			   GTK_WIDGET (zimage)->allocation.width,
			   GTK_WIDGET (zimage)->allocation.height);

    /* clear padding 
    area.x = 0;
    area.y = 0;
    area.width  = zimage->x_pad;
    area.height = GTK_WIDGET (zimage)->allocation.height;
    gdk_window_clear_area (GTK_WIDGET (zimage)->window,
			   area.x, area.y, area.width,  area.height);

    area.x = zimage->x_pad + zimage->width;
    gdk_window_clear_area (GTK_WIDGET (zimage)->window,
			   area.x, area.y, area.width,  area.height);

    area.x = area.y = 0;
    area.width = GTK_WIDGET (zimage)->allocation.width;
    area.height = zimage->y_pad;
    gdk_window_clear_area (GTK_WIDGET (zimage)->window,
			   area.x, area.y, area.width,  area.height);

    area.y = zimage->y_pad + zimage->height;
    gdk_window_clear_area (GTK_WIDGET (zimage)->window,
			   area.x, area.y, area.width,  area.height);
    */

  } else if (state == GPENGINE_FINISHED) {
    gdk_flush ();

  } else if (state == GPENGINE_ABORTED) {
    
    if (zimage->next_image) {
      zimage_set_image (zimage, zimage->next_image);
      image_unref (zimage->next_image);
      zimage->next_image = NULL;
    } 
  }
}
