
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: codec.c,v 1.6 2000/04/28 20:46:34 dr Exp $ 
*/


#include <unistd.h>
#include <string.h>
#include <sys/stat.h>

#include "codec.h"
#include "dcache.h"
#include "gpengine.h"

/*
  Improvement ? grouping tiles to make a bigger tile
  might make things faster by reducing the number of
  times we synchronize, and the number of function calls 
  (There could be several thousands calls for a single picture).

  eg cache tiles received by codec_process_tile
  (which are 1 row wide) to build bigger tiles
  (says 5 rows wide).

  note : synchronization is done 1 time per tile
  in GPENGINE_RUN_INTERACTIVE mode and 2 times
  per image otherwise.
*/

static void              _codec_class_init (ModuleClass *klass);
static void              _codec_init (Module *module);
static int               _codec_run (Filter *filter);

static int               initialized = 0;
/* list of codec classes */
static GList *           codec_list = NULL;

static Param  *          p_bounding_box_size = NULL;

ModuleType 
codec_get_type () {
  static ModuleType type = 0;

  if (!type) {
    static ModuleTypeInfo codec_info = {
      "Codec",
      NULL,
      NULL,
      sizeof (Codec),
      sizeof (CodecClass),
      _codec_init,
      _codec_class_init,
      NULL,
      NULL
    };

    type = module_type_unique (filter_get_type (), &codec_info);
  }

  return type;
}

static void
_codec_class_init (ModuleClass *klass) {
  FilterClass *filter_class;

  filter_class            = (FilterClass *) klass;

  filter_class->flags     |= FILTER_NEED_BUFFER;
  filter_class->run       = _codec_run;

  if (!g_list_find (codec_list, klass))
    codec_list = g_list_prepend (codec_list, klass);

  if (!initialized) {
    initialized = 1;
    p_bounding_box_size = 
      param_new ("bounding_box_size",
		 PARAM_INPUT,
		 "Bounding box size",
		 "This parameter is used to speedup the thumbnail creation. "
		 "Some codecs can create thumbnails much faster than they will "
		 "read the whole image\n"
		 "It's value must be:\n"
		 "\t- MAX (thumbnail_width, thumbnail_height) for creating thumbnails\n"
		 "\t- a negative value for reading the whole image.",
		 PARAM_TYPE_LONG,
		 (ParamValue) (gulong) 100);
    module_class_add_param (klass, p_bounding_box_size);	
  }
}

static void
_codec_init (Module *module) {

}

Module *
codec_new (CodecOperation operation, int bounding_box_size) {
  Codec *codec;

  codec = (Codec *) module_type_new (codec_get_type ());

  codec->operation = operation;
  codec_set_bounding_box_size (codec, bounding_box_size);

  return (Module *) codec;
}

void
codec_set_bounding_box_size (Codec *codec, int size) {
  g_return_if_fail (codec);

  module_set_param_value (MODULE (codec),
			  p_bounding_box_size->name,
			  p_bounding_box_size->type,
			  (ParamValue) (gulong) size);
}

static int
_codec_run (Filter *filter) {
  Codec *codec;
  Codec *real_codec;
  Image *image, *cimage = NULL;
  ModuleType type;
  CodecFunc func;
  ParamValue *bbs;
  struct stat statbuf;
  char *path;
  int retval;

  codec = CODEC (filter);
  g_return_val_if_fail (codec, 0);

  image = codec_get_image (codec);

  bbs = module_get_param_value (MODULE (codec),
				p_bounding_box_size->name,
				PARAM_TYPE_LONG);
  g_return_val_if_fail (bbs, 0);
  codec->bounding_box_size = bbs->v_long;

  /* check if this image has been cached to the HD */
  cimage = dcache_lookup (image, codec->bounding_box_size); 
  if (cimage && codec->operation != CODEC_WRITE_IMAGE) 
    image = cimage;

  path = image_get_path (image);
  
  /*
  if (codec->operation == CODEC_WRITE_IMAGE &&
      lstat (path, &statbuf) != ENOENT) {
    g_free (path);
    if (cimage) image_unref (cimage);
    return 0;
  }
  */

  codec->file = fopen (path, codec->operation == CODEC_WRITE_IMAGE ? "w+b" : "rb");
  if (!codec->file) {
    module_error (MODULE (filter), "Can't open file %s", path);
    if (cimage) image_unref (cimage);
    return 0;
  }

  if (!image->file_size &&
      !fstat (fileno (codec->file), &statbuf)) {
    image->file_size = statbuf.st_size;
    image->file_mtime = statbuf.st_mtime;
  }

  type = codec_find_image_type (image->file_name, 
				image->directory);
  real_codec = (Codec *) module_type_new (type);
  
  FILTER (real_codec)->image    = image;
  FILTER (real_codec)->gpe      = FILTER (codec)->gpe;
  real_codec->file              = codec->file;
  real_codec->bounding_box_size = codec->bounding_box_size;
  
  func = CODEC_CLASS (real_codec)->operations [codec->operation];
  retval = func ? (* func) (codec) : 0;  
  module_free (MODULE (real_codec));
  fclose (codec->file);

  if (!retval && codec->operation == CODEC_WRITE_IMAGE)
    unlink (path);
  g_free (path);

  if (cimage)
    image_unref (cimage);

  return retval;
}

ModuleType
codec_find_image_type (char *file_name, char *directory) {
  CodecClass *codec;
  GList *list;
  char *extension , buff [16];

  g_return_val_if_fail (file_name && directory, 0);

  extension = strrchr (file_name, '.');
  if (!extension || !extension [1]) /* should check the file itself */
    return 0;

  snprintf (buff, 16, "%s.", extension); 
  list = codec_list;

  while (list) {
    codec = list->data;
    list = list->next;

    if (codec->extensions && strstr (codec->extensions, extension) != NULL)
      return ((ModuleClass *) codec)->type;
  }

  return 0;
}

int 
codec_can_read_image (char *file_name, char *directory) {
  ModuleType type;

  g_return_val_if_fail (file_name && directory, 0);

  type = codec_find_image_type (file_name, directory);

  return type != 0;
}

