
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: dcache.c,v 1.5 2000/04/28 20:46:34 dr Exp $ 
*/



#include <stdlib.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <utime.h>

#include "dcache.h"
#include "codec.h"
#include "gpengine.h"
#include "scale.h"

#define DEBUG(str)    fprintf (stderr, str)

static void           dcache_write_callback (GPEngine *gpe, GPEngineState state, 
					     int scanlines, gpointer nada);

static int            dcache_initialized    = 0;
static char *         dcache_directory      = ".axvpixs";
static GPEngine *     gpe_write             = NULL;

Image *
dcache_lookup (Image *image, int bounding_box_size) {
  Image      *tmp_image;
  char       buf [PATH_MAX];
  
  g_return_val_if_fail (image, NULL);
  g_return_val_if_fail (dcache_initialized, NULL);

  if (bounding_box_size <= 0)
    return NULL;

  snprintf (buf, PATH_MAX, "%s/%s/%s", 
	    image->directory, 
	    dcache_directory,
	    image->file_name);

  if (access (buf, R_OK))
    return NULL;

  snprintf (buf, PATH_MAX, "%s/%s", 
	    image->directory, 
	    dcache_directory);
  tmp_image = image_new (image->file_name, buf);

  return tmp_image;
}

void
dcache_put (Image *image, ImageCache *cache) {
  Image *cimage;
  struct stat sbi, sbc;
  char  buf [PATH_MAX];
  
  g_return_if_fail (image && cache);
  g_return_if_fail (dcache_initialized);

  /* 
     all images should'nt be cached:
     small images & big caches should
     be ignored 
  */

  if (cache->width > (image->width >> 1))
    return;

  snprintf (buf, PATH_MAX, "%s/%s/%s", 
	    image->directory, 
	    dcache_directory,
	    image->file_name);

  if (!stat (buf, &sbc)) {
    /*    
      image has been cached.
      check if the cache needs to be updated 
    */

    snprintf (buf, PATH_MAX, "%s/%s",
	      image->directory,
	      image->file_name);
    
    if (stat (buf, &sbi))
      return;

    if (sbi.st_mtime <= sbc.st_mtime) 
      return;

    snprintf (buf, PATH_MAX, "%s/%s",
              image->directory,
              dcache_directory);
  } else { 
    /* cache doesn't exist:
       check if the cache directory exists
       and if not make it 
    */
    snprintf (buf, PATH_MAX, "%s/%s",
	      image->directory,
	      dcache_directory);
    
    if (access (buf, R_OK | W_OK) &&
	mkdir (buf, S_IRWXU | S_IRWXG | S_IRWXO))
      return;
  }

  cimage = image_new (image->file_name, buf);

  image_cache_add (cimage, cache);
  gpengine_start (gpe_write, GPENGINE_NO_THREAD, cimage, cache, 
		  (GPEngineCallback) dcache_write_callback, NULL);
  image_cache_unref (cimage, cache);
  image_unref (cimage);
}

void
dcache_initialize () {
  Module *module;

  g_return_if_fail (!dcache_initialized);

  gpe_write = gpengine_new ();
  /* XXX 
  module    = scale_new (0, 100, 100, NULL);
  gpengine_add_filter (gpe_write, FILTER (module));
  */

  module    = codec_new (CODEC_WRITE_IMAGE, 0);
  gpengine_add_filter (gpe_write, FILTER (module));

  dcache_initialized = 1;
}

void
dcache_finalize () {

  g_return_if_fail (dcache_initialized);

  module_free ((Module *) gpe_write);
  gpe_write = NULL;

  dcache_initialized = 0;
}

static void           
dcache_write_callback (GPEngine *gpe, GPEngineState state, 
		       int scanlines, gpointer nada) {

}

/*
  check to see if each cache correspond to an image.
  if the image has been unlinked then unlink the cache
*/

void
dcache_update_directory (char *path) {
  DIR            *dir;
  struct stat    sbi, sbc;
  struct dirent  *entry;
  char           buf [PATH_MAX];
  int            caches = 0;

  snprintf (buf, PATH_MAX, "%s/%s", 
	    path, dcache_directory);

  if (stat (path, &sbi) ||
      stat (buf, &sbc))
    return;

  /* only check if the image directory
     is more recent then the cache directory
  */
  if (sbi.st_mtime <= sbc.st_mtime)
    return;

  DEBUG ("updating DCache directory\n");

  /* open the cache directory */
  dir = opendir (buf);
  if (!dir)
    return;

  while ((entry = readdir (dir))) {
    snprintf (buf, PATH_MAX, "%s/%s",
	      path, entry->d_name);

    if (stat (buf, &sbi)) {      
      snprintf (buf, PATH_MAX,
		"%s/%s/%s",
		path, dcache_directory,
		entry->d_name);
      
      /*----------------------------------------------------- 
	 BUG: Any file under the dcache_directory could be
	 unlinked even if it isn't a cached image generated 
	 by this program
        ----------------------------------------------------*/
      if (unlink (buf))
	g_warning ("Can't unlink %s: %s", buf, strerror (errno));
      
    } else if (S_ISREG (sbi.st_mode))
      caches ++;
      
  }

  snprintf (buf, PATH_MAX, "%s/%s",
            path, dcache_directory);

  /*
    if the cache directory is empty then
    unlink it
  */

  if (caches == 0)
    rmdir (buf);

  closedir (dir);
  utime (buf, NULL);
}



