
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: directory.c,v 1.3 2000/04/28 20:46:34 dr Exp $ 
*/


#include <pthread.h>

#include "directory.h"

#define LOCK()            pthread_mutex_lock (&_directory_mutex_);
#define UNLOCK()          pthread_mutex_unlock (&_directory_mutex_);

static pthread_mutex_t    _directory_mutex_ = PTHREAD_MUTEX_INITIALIZER;
static void               directory_init ();
static GHashTable *       directory_hash_table = NULL;

static void
directory_init () {
  g_return_if_fail (!directory_hash_table);

  directory_hash_table = g_hash_table_new (g_str_hash,
					   g_str_equal);
}

Directory *
directory_new (char *path) {
  Directory *dir;
  struct stat statbuf;

  g_return_val_if_fail (path, NULL);

  dir = directory_find (path);
  if (dir)
    return directory_ref (dir);

  dir = g_new (Directory, 1);
  
  dir->refcount = 1;
  dir->name     = g_strdup (path);
  dir->mtime    = stat (path, &statbuf) ? 0 : statbuf.st_mtime;

  LOCK ();
  g_hash_table_insert (directory_hash_table, dir->name, dir);
  UNLOCK ();

  return dir;
}

Directory *
directory_ref (Directory *dir) {
  g_return_val_if_fail (dir, NULL);
  g_return_val_if_fail (dir->refcount < UINT_MAX, NULL);

  dir->refcount ++;  

  return dir;
}

void
directory_unref (Directory *dir) {
  g_return_if_fail (dir);
  g_return_if_fail (dir->refcount);
  
  dir->refcount --;
  if (dir->refcount == 0) {
    LOCK ();
    g_hash_table_remove (directory_hash_table, 
			 dir->name);
    UNLOCK ();
    g_free (dir->name);
    g_free (dir);
  }
}

Directory *
directory_find (char *path) {
  Directory *dir;

  if (!directory_hash_table)
    directory_init ();
  
  LOCK ();
  dir = g_hash_table_lookup (directory_hash_table, path);
  UNLOCK ();

  return dir;
}

int
directory_compare_mtime (char *d1, char *d2) {
  Directory *dir1, *dir2;

  dir1 = directory_find (d1);
  dir2 = directory_find (d2);

  if (dir1 == NULL && dir2 == NULL)
    return 0;
  else if (dir1 == NULL)
    return -1;
  else if (dir2 == NULL)
    return 1;

  return (int) (dir1->mtime - dir2->mtime);
}

char *
directory_ref_by_name (char *path) {
  Directory *dir;

  if (!path)
    return NULL;

  dir = directory_find (path);
  if (dir) 
    directory_ref (dir);
  else
    dir = directory_new (path);

  return dir->name;
}

void
directory_unref_by_name (char *path) {
  Directory *dir;

  dir = directory_find (path);
  if (!dir)
    return;

  directory_unref (dir);
}

