
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: draw.c,v 1.5 2000/04/28 20:46:34 dr Exp $ 
*/


#include "draw.h"

static void             draw_class_init (ModuleClass *class);
static void             draw_init (Module *module);
static void             draw_free (Module *module);
static int              draw_filter_init (Filter *filter);
static int              draw_filter_process_tile (Filter *filter, FilterTile *src, FilterTile *out);
static void             draw_free_outputs (Filter *filter);

static int              initialized = 0;
static Param *          p_output_pixmap = NULL;

ModuleType
draw_get_type () {
  static ModuleType type = 0;

  if (!type) {
    static ModuleTypeInfo draw_info = {
      "Draw",
      NULL,
      NULL,
      sizeof (Draw),
      sizeof (DrawClass),
      draw_init,
      draw_class_init,
      /* reserved1 */ NULL,
      /* reserved2 */ NULL,
    };

    type = module_type_unique (filter_get_type (), &draw_info);
  }

  return type;
}

static void
draw_class_init (ModuleClass *class) {
  FilterClass    *filter_class;
  DrawClass      *draw_class;

  filter_class = (FilterClass *) class;
  draw_class   = (DrawClass *) class;

  class->free                         = draw_free;

  filter_class->flags                 |= FILTER_USE_X;
  filter_class->filter_init           = draw_filter_init;
  filter_class->filter_process_tile   = draw_filter_process_tile;
  filter_class->free_outputs          = draw_free_outputs;

  if (!initialized) {
    initialized = 1;
    p_output_pixmap =
      param_new ("pixmap",
		 PARAM_OUTPUT,
		 NULL,
		 "The rendered pixmap.",
		 PARAM_TYPE_POINTER,
		 (ParamValue) (gpointer) NULL);
    module_class_add_param (class, p_output_pixmap);
  }
}

static void
draw_init (Module *module) {

}

Module *
draw_new (GdkDrawable *drawable, GdkGC *gc, int build_pixmap, int pixmap_depth) {
  Draw *draw;
  
  g_return_val_if_fail (gc, NULL);
  g_return_val_if_fail ((build_pixmap && pixmap_depth && !drawable) || drawable, NULL);

  draw = (Draw *) module_type_new (draw_get_type ());

  draw->drawable     = build_pixmap ? NULL : drawable;
  draw->gc           = gdk_gc_ref (gc);
  draw->build_pixmap = build_pixmap;
  draw->pixmap_depth = pixmap_depth;
  draw->ipix         = NULL;

  return (Module *) draw;
}

static void
draw_free (Module *module) {
  g_return_if_fail (module);

  if (DRAW (module)->gc)
    gdk_gc_unref (DRAW (module)->gc);
}

static int
draw_filter_init (Filter *filter) {
  Draw *draw;
  
  g_return_val_if_fail (filter, 0);

  draw = (Draw *) filter;

  draw->out_x     = 0;
  draw->out_y     = 0;

  if (draw->build_pixmap) {
    draw->ipix = image_pixmap_new (filter->image, filter->src_width, filter->src_height);
    draw->drawable = draw->ipix->pixmap;

    module_set_param_value (MODULE (filter),
			    p_output_pixmap->name,
			    p_output_pixmap->type,
			    (ParamValue) (gpointer) draw->ipix);
  }

  return 1;
}

static int
draw_filter_process_tile (Filter *filter, FilterTile *src, FilterTile *out) {

  if (filter->src_bpp == 1) 
    gdk_draw_gray_image (DRAW (filter)->drawable,
			 DRAW (filter)->gc,
			 DRAW (filter)->out_x, DRAW (filter)->out_y,
			 src->width, src->height,
			 GDK_RGB_DITHER_NORMAL,
			 src->data,
			 src->width);
  else
    gdk_draw_rgb_image_dithalign (DRAW (filter)->drawable,
				  DRAW (filter)->gc,
				  DRAW (filter)->out_x, DRAW (filter)->out_y,
				  src->width, src->height,
				  GDK_RGB_DITHER_NORMAL,
				  src->data,
				  src->width * 3,
				  DRAW (filter)->out_x, 
				  DRAW (filter)->out_y);

  DRAW (filter)->out_y += src->height; 

  return 1;
}

static void
draw_free_outputs (Filter *filter) {
  Draw *draw;

  draw = (Draw *) filter;
  
  g_return_if_fail (draw && draw->ipix);

  if (draw->ipix) {
    module_clear_param_value (MODULE (filter),
			      p_output_pixmap->name);
    image_pixmap_unref (filter->image, draw->ipix);
    draw->ipix = NULL;
  }
}

void                       
draw_set_output_pos (Draw *draw, int out_x, int out_y) {
  g_return_if_fail (draw);

  draw->out_x = out_x;
  draw->out_y = out_y;
}

