
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: drawarea.c,v 1.4 2000/04/28 20:46:34 dr Exp $ 
*/


#include "drawarea.h"

static void             draw_area_class_init (ModuleClass *klass);
static void             draw_area_init (Module *module);
static void             draw_area_free (Module *module);
static int              draw_area_draw_init (DrawArea *da);
static void             draw_area_draw_end (DrawArea *da);

ModuleType                 
draw_area_get_type () {
  static ModuleType type = 0;

  if (!type) {
    static ModuleTypeInfo info = {
      "DrawArea",
      NULL,
      NULL,
      sizeof (DrawArea),
      sizeof (DrawAreaClass),
      draw_area_init,
      draw_area_class_init,
      NULL,
      NULL
    };

    type = module_type_unique (filter_get_type (), &info);
  }

  return type;
}

static void             
draw_area_class_init (ModuleClass *klass) {
  FilterClass * filter_class;

  filter_class = (FilterClass *) klass;

  klass->free              = draw_area_free;
  filter_class->flags     |= FILTER_USE_X;
}

static void             
draw_area_init (Module *module) {
  
}

static void
draw_area_free (Module *module) {
  DrawArea *da;
  g_return_if_fail (module);

  da = (DrawArea *) module;

  if (da->initialized)
    draw_area_draw_end (da);

  draw_area_set_image (da, NULL);

  module_free ((Module *) da->scale);
  module_free ((Module *) da->draw);
}

Module *                   
draw_area_new (GdkDrawable *drawable, 
	       GdkGC *gc,
	       GdkColor *background) {
  DrawArea *da;

  g_return_val_if_fail (drawable && gc, NULL);

  da = (DrawArea *) module_type_new (draw_area_get_type ());
  da->zoom  = 1;
  da->scale = (Scale *) scale_new (0, -1, -1, background);
  da->draw  = (Draw *) draw_new (drawable, gc, 0, 0);
  
  return (Module *) da;
}

int
draw_area_set_zoom (DrawArea *da, int zoom) {
  g_return_val_if_fail (da, 0);
  g_return_val_if_fail (zoom != 0, 0);

  if (da->zoom == zoom)
    return 1;

  da->zoom = zoom;

  draw_area_draw_end (da);
  return draw_area_draw_init (da);
}

int
draw_area_zoom_in (DrawArea *da) {

  g_return_val_if_fail (da, 0);

  return draw_area_set_zoom (da, DRAW_AREA_ZOOM_IN (da->zoom));
}

int
draw_area_zoom_out (DrawArea *da) {
  g_return_val_if_fail (da, 0);

  return draw_area_set_zoom (da, DRAW_AREA_ZOOM_OUT (da->zoom));
}

int
draw_area_set_image (DrawArea *da, Image *image) {
  
  g_return_val_if_fail (da, 0);

  if (image == da->image)
    return 1;

  draw_area_draw_end (da);

  if (da->image) {
    if (da->cache)
      image_cache_unref (da->image, da->cache);
    da->cache = NULL;
    image_unref (da->image);
  }

  da->image = image ? image_ref (image) : NULL;
  if (da->image) {
    da->cache = image_cache_get (image, -1, -1);
    if (!da->cache)
      return 0;
    image_cache_ref (da->image, da->cache);

    return draw_area_draw_init (da);
  }
  
  return 1;
}

int                      
draw_area_draw_init (DrawArea *da) {
  g_return_val_if_fail (da, 0);
  g_return_val_if_fail (da->image && da->cache, 0);

  if (da->initialized)
    draw_area_draw_end (da);

  da->scale->bounding_box_width  = DRAW_AREA_APPLY_ZOOM (da, da->image->width);
  da->scale->bounding_box_height = DRAW_AREA_APPLY_ZOOM (da, da->image->height);
  
  if (!filter_init (FILTER (da->scale), da->image, 
		    da->cache->width, da->cache->height, da->cache->bpp,
		    da->cache->width, 1))
    return 0;

  if (!filter_init (FILTER (da->draw), da->image, 
		    FILTER (da->scale)->out_width, 
		    FILTER (da->scale)->out_height, 
		    FILTER (da->scale)->out_bpp,
		    FILTER (da->scale)->out_tile_width, 
		    FILTER (da->scale)->out_tile_height)) {
    filter_end (FILTER (da->scale));
    return 0;
  }
  
  da->initialized = 1;

  return 1;
}

void
draw_area_draw_end (DrawArea *da) {
  g_return_if_fail (da);

  if (da->initialized) {
    da->initialized = 0;
    filter_end (FILTER (da->scale));
    filter_end (FILTER (da->draw));
  }
}

void                       
draw_area_draw (DrawArea *da, 
		int x, int y,
		GdkRectangle *area) {
  Scale *scale;
  FilterTile in, out;
  int area_width, area_height;
  int c, dy; 

  g_return_if_fail (da);

  if (!da->initialized) {
    gdk_window_clear_area (da->draw->drawable,
			   area->x, area->y, area->width, area->height);
    return;
  }

  scale = da->scale;
  g_return_if_fail (scale->x_map && scale->y_map);
  g_return_if_fail (area);

  area_width  = area->width;
  area_height = area->height;

  if (x < 0) {
    area->x -= x;
    area_width += x;
    x = 0;
  }
  
  if (y < 0) {
    area->y -= y;
    area_height += y;
    y = 0;
  }


  c = FILTER (scale)->out_width - (x + area_width);
  if (c < 0)
    area_width += c;
  
  c = FILTER (scale)->out_height - (y + area_height);
  if (c < 0)
    area_height += c;
  
  if (area_width <= 0 || area_height <= 0)
    return;

  scale->out_row = y;
  scale->src_row = scale->y_map [scale->out_row];
  
  in.width  = da->cache->width;
  in.height = 1; //scale->y_map [y + area_height] - scale->src_row;
  in.data   = da->cache->data + scale->src_row * scale->src_row_stride;
  /*
  out.data  = g_malloc (area_width * area_height * FILTER (scale)->src_bpp);

  scale_process_tile_parts (scale, &in, &out, 
			    x, x + area_width);
  
 draw_set_output_pos (da->draw, area->x, area->y);
  
  (* FILTER_CLASS (da->draw)->process_tile) (FILTER (da->draw),
  &out,
  FILTER (draw)->buffer);

  g_free (out.data);

  */

  for (dy = area->y; dy < area->y + area_height;) {
    
    scale_process_tile_parts (scale, &in, FILTER (scale)->buffer, 
			      x, x + area_width);

    draw_set_output_pos (da->draw, area->x, dy);

    
    (* FILTER_CLASS (da->draw)->filter_process_tile) (FILTER (da->draw),
						      FILTER (scale)->buffer,
						      FILTER (da->draw)->buffer);
    
    dy += FILTER (scale)->buffer->height;
    in.data += scale->src_row_stride;
  }
}



