
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: filter.c,v 1.6 2000/04/28 20:46:34 dr Exp $ 
*/


#include "filter.h"

static void          _filter_class_init (ModuleClass *klass);
static void          _filter_init (Module *module);
static int           _filter_run (Filter *filter);

static int          filter_init_nolock (Filter *filter, Image *image,
					int src_width, int src_height, int src_bpp, 
					int src_tile_width, int src_tile_height);
static int          filter_end_nolock (Filter *filter);

#define FILTER_HAVE_THREAD(filter) ((filter)->gpe ? GPENGINE_HAVE_THREAD (filter->gpe) : 0)

#define ENTER_THREAD(filter)  FILTER_USE_X (filter) && FILTER_HAVE_THREAD (filter) ? gdk_threads_enter () : 0 
#define LEAVE_THREAD(filter)  FILTER_USE_X (filter) && FILTER_HAVE_THREAD (filter) ? gdk_threads_leave () : 0 

ModuleType 
filter_get_type () {
  static ModuleType type = 0;

  if (!type) {
    static ModuleTypeInfo info = {
      "Filter",
      NULL,
      NULL,
      sizeof (Filter),
      sizeof (FilterClass),
      _filter_init,
      _filter_class_init,
      NULL,
      NULL
    };

    type = module_type_unique (module_get_type (), &info);
  }

  return type;
}


static void          
_filter_class_init (ModuleClass *klass) {
  FilterClass *filter_class;

  filter_class             = (FilterClass *) klass;

  filter_class->run        = _filter_run;
}

static void          
_filter_init (Module *module) {

}

static int
filter_init_nolock (Filter *filter, Image *image,
		    int src_width, int src_height, int src_bpp, 
		    int src_tile_width, int src_tile_height) {

  g_return_val_if_fail (filter, 0);
  g_return_val_if_fail (!filter->initialized, 0);
  g_return_val_if_fail (src_width > 0 && src_height > 0 && src_bpp > 0 &&
			src_tile_width > 0 && src_tile_height > 0, 0);

  filter->image                = image;
  filter->src_width            = filter->out_width            = src_width;
  filter->src_height           = filter->out_height           = src_height;
  filter->src_bpp              = filter->out_bpp              = src_bpp;
  filter->src_tile_width       = filter->out_tile_width       = src_tile_width;
  filter->src_tile_height      = filter->out_tile_height      = src_tile_height;

  if (FILTER_CLASS (filter)->filter_init &&
      !(* FILTER_CLASS (filter)->filter_init) (filter))
    return 0;

  filter->initialized = 1;
  if (FILTER_CLASS (filter)->flags & FILTER_NEED_BUFFER) {
    
    filter->buffer_size          = filter->out_tile_width * filter->out_tile_height * filter->out_bpp *
      /* little waste of memory which makes things a lot easier :
	 instead of processing tile by tile the GPEngine processes filter->src_tile_height tiles 
	 supposes filter->out_bpp <= filter->src_bpp */
      filter->src_tile_height; 
    filter->buffer_size         += filter->buffer_size % 8;
    filter->buffer               = filter_tile_alloc (filter->buffer_size);
    filter->buffer->width        = filter->out_tile_width;
    filter->buffer->height       = filter->out_tile_height;
    filter->buffer->bpp          = filter->out_bpp;
  }

  return 1;
}

int
filter_init (Filter *filter, Image *image,
	     int src_width, int src_height, int src_bpp, 
	     int src_tile_width, int src_tile_height) {
  int success;

  ENTER_THREAD (filter);
  success = filter_init_nolock (filter, image, src_width, src_height, src_bpp,
				src_tile_width, src_tile_height);
  LEAVE_THREAD (filter);

  return success;
}

inline int              
filter_process_tile (Filter *filter, FilterTile *src) {
  int success;

  g_return_val_if_fail (filter, 0);
  g_return_val_if_fail (filter->initialized, 0);
  g_return_val_if_fail (src, 0);

  ENTER_THREAD (filter);
  success = FILTER_CLASS (filter)->filter_process_tile ? 
    (* FILTER_CLASS (filter)->filter_process_tile) (filter, src, filter->buffer) : 1;
  LEAVE_THREAD (filter);

  return success;
}

static int
filter_end_nolock (Filter *filter) {

  g_return_val_if_fail (filter, 0);

  if (filter->buffer) {
    g_free (filter->buffer->data);
    g_free (filter->buffer);
    filter->buffer = NULL;
  }

  if (filter->initialized && FILTER_CLASS (filter)->filter_end)
    (* FILTER_CLASS (filter)->filter_end) (filter);

  filter->initialized = 0;

  return 1;
}

int 
filter_end (Filter *filter) {
  ENTER_THREAD (filter);
  filter_end_nolock (filter);
  LEAVE_THREAD (filter);

  return 1;
}

int
filter_run (Filter *filter) {
  int success;
  g_return_val_if_fail (filter, 0);

  ENTER_THREAD (filter);
  success = FILTER_CLASS (filter)->run ?
    (* FILTER_CLASS (filter)->run) (filter) : 0;
  LEAVE_THREAD (filter);

  return success;
}

void
filter_free_outputs (Filter *filter) {

  ENTER_THREAD (filter);
  if (FILTER_CLASS (filter)->free_outputs)
    (* FILTER_CLASS (filter)->free_outputs) (filter);
  LEAVE_THREAD (filter);

}

static int
_filter_run (Filter *filter) {
  FilterTile  tile;
  ImageCache  *src_cache, *out_cache = NULL;
  guchar      *out = NULL;
  int         j, src_row_stride, out_row_stride;

  g_return_val_if_fail (filter, 0);

  if (!FILTER_CLASS (filter)->filter_process_tile)
    return 1;

  src_cache = gpengine_get_cache (filter->gpe);
  g_return_val_if_fail (src_cache, 0);

  if (!filter_init_nolock (filter, 
			   filter->image,
			   src_cache->width, src_cache->height, src_cache->bpp, 
			   src_cache->width, 1))
    return 0;

  src_row_stride = src_cache->width * src_cache->bpp;
  tile.width  = src_cache->width;
  tile.height = 1;
  tile.bpp    = src_cache->bpp;
  tile.data   = src_cache->data;

  if (filter->buffer) {
    out_cache = image_cache_new (filter->image, 
				 filter->out_width,
				 filter->out_height,
				 filter->out_bpp);
    out = out_cache->data;
  } 

  for (j = 0; j < src_cache->height; j++) {

    if (!(* FILTER_CLASS (filter)->filter_process_tile) (filter, &tile, filter->buffer)) {
      filter_end (filter);
      return 0;
    }

    if (filter->buffer) {
      out_row_stride = filter->buffer->width * filter->buffer->height * filter->out_bpp;
      memcpy (out, filter->buffer->data, out_row_stride);
      out += out_row_stride;
    }

    tile.data += src_row_stride;
  }

  filter_end_nolock (filter);

  if (out_cache) {
    gpengine_set_cache (filter->gpe, out_cache);
    image_cache_unref (filter->image, out_cache);
  }

  return 1;
}

FilterTile *
filter_tile_alloc (guint size) {
  FilterTile *tile;

  if (size > 0) {
    tile             = g_new (FilterTile, 1);
    tile->width      = 0;
    tile->height     = 0;
    tile->bpp        = 0;
    tile->data       = g_malloc (size);

    return tile;
  } 

  return NULL;
}

void             
filter_tile_free (FilterTile *tile) {
  g_return_if_fail (tile);

  g_free (tile->data);
  g_free (tile);
}
