
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: gpengine.c,v 1.9 2000/04/28 20:46:34 dr Exp $ 
*/


#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>

#include "gpengine.h"
#include "codec.h"
#include "mcache.h"

#define ENTER_THREAD(gpe)      GPENGINE_HAVE_THREAD (gpe) ? gdk_threads_enter () : 0
#define LEAVE_THREAD(gpe)      GPENGINE_HAVE_THREAD (gpe) ? gdk_threads_leave () : 0

static void          gpengine_class_init (ModuleClass *klass);
static void          gpengine_init (Module *module);
static void          gpengine_free (Module *module);

static void          gpengine_report_status (GPEngine *gpe, int scanlines);
static void *        gpengine_thread (void *data);
static void          gpengine_pipeline_end (GPEngine *gpe);
static int           gpengine_run (GPEngine *gpe, int start_filter);

ModuleType 
gpengine_get_type () {
  static ModuleType type = 0;

  if (!type) {
    static ModuleTypeInfo info = {
      "GPEngine",
      NULL,
      NULL,
      sizeof (GPEngine),
      sizeof (GPEngineClass),
      gpengine_init,
      gpengine_class_init,
      NULL,
      NULL
    };

    type = module_type_unique (module_get_type (), &info);
  }

  return type;
}

static void
gpengine_class_init (ModuleClass *klass) {

  klass->free              = gpengine_free;
}

static void
gpengine_init (Module *module) {
}

GPEngine *
gpengine_new () {
  GPEngine *gpe;

  gpe           = (GPEngine *) module_type_new (gpengine_get_type ());
  gpe->status   = GPENGINE_FINISHED;
  gpe->filters  = g_array_new (1, 0, sizeof (Filter *));
  gpe->xfilter  = -1;
  gpe->mcache   = (Filter *) mcache_new (0);
  gpe->mcache->gpe = gpe;

  return gpe;
}

/* BUG: the GPEngine structure is free while the other
   thread may be using it. 
*/
static void
gpengine_free (Module *module) {
  GPEngine *gpe;
  int i;
  
  gpe = (GPEngine *) module;

  gpengine_abort (gpe);

  module_free ((Module *) gpe->mcache);
  
  for (i = 0; i < gpe->filters->len; i++)
    module_free ((Module *) g_array_index (gpe->filters, Filter *, i));

  g_array_free (gpe->filters, 1);
}

ImageCache *
gpengine_get_cache (GPEngine *gpe) {
  g_return_val_if_fail (gpe, NULL);

  return gpe->cache;
}

void
gpengine_set_cache (GPEngine *gpe, ImageCache *cache) {
  g_return_if_fail (gpe);

  if (gpe->cache)
    image_cache_unref (gpe->image, gpe->cache);

  gpe->cache = cache ? image_cache_ref (gpe->image, cache) : NULL;
}

void
gpengine_add_filter (GPEngine *gpe, Filter *filter) {
  
  g_return_if_fail (gpe && filter);

  filter->gpe = gpe;

  if (FILTER_USE_X (filter)) {

    /* there's only place for a single 'xfilter' */
    g_return_if_fail (gpe->xfilter == -1);

    gpe->xfilter = gpe->filters->len;
    gpe->filters = g_array_append_val (gpe->filters, filter);
  } else {

    /* the 'xfilter' must be the last filter */
    g_return_if_fail (gpe->xfilter == -1);

    gpe->filters = g_array_append_val (gpe->filters, filter);
  }
}

int 
gpengine_start (GPEngine *gpe, guint flags, Image *image, ImageCache *cache, 
		GPEngineCallback callback, gpointer user_data) {
  int err;

  g_return_val_if_fail (gpe->status == GPENGINE_FINISHED, 0);

  /*
  if (gpe->thread_id &&
      (err = pthread_join (gpe->thread_id, NULL))) 
    g_warning ("Can't join thread %lx (%d: %s)", 
	       gpe->thread_id, err, strerror (err));
  */

  gpe->status        = GPENGINE_STARTED;
  gpe->abort         = 0;
  gpe->flags         = flags;
  gpe->image         = image_ref (image);
  gpe->cache         = cache ? image_cache_ref (image, cache) : NULL;
  gpe->callback      = callback;
  gpe->user_data     = user_data;
  gpe->read_rows     = 0;
  gpe->thread_id     = 0;

  if (gpe->flags & GPENGINE_NO_THREAD) {
    gpengine_thread (gpe);

  } else {
    pthread_attr_t attr;
    
    pthread_attr_init (&attr);
    pthread_attr_setdetachstate (&attr, PTHREAD_CREATE_DETACHED);

    if ((err = pthread_create (&gpe->thread_id, &attr, gpengine_thread, gpe))) {
      image_cache_unref (image, cache);
      image_unref (image);
      g_warning ("Can't start thread (%d: %s)", err, strerror (err));
      return 0;
    }

  }
  return 1;
}

int 
gpengine_is_running (GPEngine *gpe) {
  g_return_val_if_fail (gpe, 0);

  return gpe->status != GPENGINE_FINISHED;
}

void
gpengine_abort (GPEngine *gpe) {

  if (gpe->status == GPENGINE_FINISHED)
    return;
  
  g_return_if_fail (pthread_equal (pthread_self (), gpe->thread_id) == 0);

  /* the thread will terminate by itself
     when it'll see this flags.

     there's surely faster ways to do
     that (thread cancellation or non local jumps ?
     will surely be better but harder to implement)
  */
  gpe->abort = 1;
}

gfloat
gpengine_get_progress (GPEngine *gpe) {
  g_return_val_if_fail (gpe, 0.0);

  return 
    gpe->status == GPENGINE_FINISHED || gpe->rows == 0 ? 0 : (gfloat) gpe->read_rows / gpe->rows;
}

static void *
gpengine_thread (void *data) {
  GPEngine *gpe;
  Filter   *filter;
  int success, i;

  gpe = (GPEngine *) data;

  if (gpe->cache) 
    /* render from the cache */
    success = gpengine_run (gpe, 0);

  else {          
    /* either create a cache of the image
       or process each tile 1 by 1 
       (if in GPENGINE_RUN_INTERACTIVE mode)
    */

    filter  = g_array_index (gpe->filters, Filter *, 0);
    filter->image = gpe->image;
    success = filter_run (filter);

    gpengine_pipeline_end (gpe);
  }

  /* free ouputs data */  
  filter_free_outputs (gpe->mcache);
  for (i = 0; i < gpe->filters->len; i++) {
    filter = g_array_index (gpe->filters, Filter *, i);
    
    filter_free_outputs (filter);    
  }
  
  gpengine_set_cache (gpe, NULL);

  image_unref (gpe->image);
  gpe->image = NULL;

  gpe->status = GPENGINE_FINISHED;
  gpengine_report_status (gpe, 0);

  return (void *) success;
}

static void
gpengine_report_status (GPEngine *gpe, int scanlines) {
  ENTER_THREAD (gpe);
  
  if (gpe->status == GPENGINE_FINISHED && gpe->abort)
    (* gpe->callback) (gpe, GPENGINE_ABORTED, scanlines, gpe->user_data);
  else if (!gpe->abort)
    (* gpe->callback) (gpe, gpe->status, scanlines, gpe->user_data);

  LEAVE_THREAD (gpe);
}


int
gpengine_pipeline_init (GPEngine *gpe, 
			int width, int height, int bpp,
			int src_width, int src_height, int src_bpp,
			int src_tile_width, int src_tile_height) {
  Filter *filter, *lbf;
  int success = 1, i;


  gpe->rows = src_height;

  if (!gpe->image->width || 
      /* XXX temporary hack: needed because
	 the dcache module doesn't read the original
	 image dimensions
       */
      width > gpe->image->width) {
    gpe->image->width       = width;
    gpe->image->height      = height;
    gpe->image->bpp         = bpp;
  }

  filter = g_array_index (gpe->filters, Filter *, 0);

  filter_init (filter, gpe->image,
	       src_width, src_height, src_bpp,
	       src_tile_width, src_tile_height);
  
  if (gpe->flags & GPENGINE_RUN_INTERACTIVE) {
    lbf = filter;

    for (i = 1; success && i < gpe->filters->len; i++) {
      filter = g_array_index (gpe->filters, Filter *, i);

      if (gpe->abort)
	return 0;

      success = filter_init (filter, gpe->image,
			     lbf->out_width, lbf->out_height, lbf->out_bpp,
			     lbf->out_tile_width, lbf->out_tile_height);
      
      if (filter->buffer)
	lbf = filter;
    }

    gpe->status = GPENGINE_INITIALIZED;
    gpengine_report_status (gpe, 0);
    gpe->status = GPENGINE_PROCESSING;

  } else 
    success = filter_init (gpe->mcache, gpe->image, 
			   src_width, src_height, src_bpp,
			   src_tile_width, src_tile_height);
			   
  return success;
}

int
gpengine_pipeline_process_tile (GPEngine *gpe, FilterTile *tile) {
  int success = 1, i;

  if (gpe->abort)
    return 0;

  if (gpe->flags & GPENGINE_RUN_INTERACTIVE) {
    Filter *filter;
    FilterTile *lbt;

    lbt = tile;

    for (i = 1; success && i < gpe->filters->len; i++) {
      filter = g_array_index (gpe->filters, Filter *, i);
      
      success = filter_process_tile (filter, lbt);
      
      if (filter->buffer)
	lbt = filter->buffer;
    }

    gpengine_report_status (gpe, tile->height);

    gpe->read_rows += tile->height;
  } else {
    success = filter_process_tile (gpe->mcache, tile);
    gpe->read_rows += tile->height;
  }

  return success;
}

static void
gpengine_pipeline_end (GPEngine *gpe) {
  Filter *filter;
  int i;

  if (gpe->flags & GPENGINE_RUN_INTERACTIVE) {
    for (i = 0; i < gpe->filters->len; i++) {
      filter = g_array_index (gpe->filters, Filter *, i);
      
      filter_end (filter);
    }

  } else {
    ImageCache *cache;

    cache = mcache_get_cache (MCACHE (gpe->mcache));
    gpengine_set_cache (gpe, cache);

    filter_end (gpe->mcache);
    filter = g_array_index (gpe->filters, Filter *, 0);
    filter_end (filter);

    gpengine_run (gpe, 1);
  }      
}

static int
gpengine_run (GPEngine *gpe, int start_filter) {
  Filter *filter;
  int i, success = 1;

  if (!gpe->cache)
    return 0;

  for (i = start_filter; success && i < gpe->filters->len; i++) {
    filter = g_array_index (gpe->filters, Filter *, i);
    
    filter->image = gpe->image;
    success = filter_run (filter);
  }
    
  /* have to wait that all modules have run before doing that */
  gpe->status = GPENGINE_INITIALIZED;
  gpengine_report_status (gpe, 0);

  return success;
}

ParamValue *
gpengine_get_param_value (GPEngine  *gpe,
			  int       filter_no,
			  char      *name,
			  ParamType type) {
  Filter *filter;

  if (filter_no < 0) {
    ParamValue *value;
    int i;

    for (i = 0; i < gpe->filters->len; i++) {
      filter = g_array_index (gpe->filters, Filter *, i);
      
      value = module_get_param_value (MODULE (filter), name, type);
      if (value)
	return value;
    }

    return NULL;
  }
  
  g_return_val_if_fail (filter_no < gpe->filters->len, NULL);
  
  filter = g_array_index (gpe->filters, Filter *, filter_no);

  return module_get_param_value (MODULE (filter), name, type);  
}
    
