
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: gpengine.h,v 1.6 2000/04/28 20:46:34 dr Exp $ 
*/

#ifndef _GP_ENGINE_H_ 
#define _GP_ENGINE_H_

#include <pthread.h>
#include "module.h"
#include "param.h"

#define GPENGINE_TYPE            (gpengine_get_type ())
#define GPENGINE(module)         ((GPEngine *) module)
#define IS_GPENGINE(module)      MODULE_CHECK_CAST (MODULE (module), GPENGINE_TYPE)
#define GPENGINE_CLASS(module)   ((GPEngineClass *) MODULE_CLASS (module))

#define GPENGINE_HAVE_THREAD(gpe) !((gpe)->flags & GPENGINE_NO_THREAD)

typedef struct _GPEngine         GPEngine;
typedef struct _GPEngineClass    GPEngineClass;

#include "filter.h"

typedef enum {
  GPENGINE_STARTED      = -100,
  GPENGINE_INITIALIZED,
  GPENGINE_FINISHED,
  GPENGINE_PROCESSING,
  GPENGINE_ABORTED,
  GPENGINE_AFINISHED
} GPEngineState;

enum {
  GPENGINE_NO_THREAD       = 1 << 0,
  GPENGINE_RUN_INTERACTIVE = 1 << 1,

  /* internal */
};

typedef void   (*GPEngineCallback) (GPEngine *gpe, 
				    GPEngineState state, 
				    int scanlines,
				    gpointer *user_data);

struct _GPEngine {
  Module      module;

  guint       flags;

  Image       *image;
  ImageCache  *cache;

  Filter      *mcache;

  GArray      *filters;
  int         xfilter;

  pthread_t   thread_id;
  int         abort;

  /* for progress monitoring */
  guint       status;
  guint       rows, read_rows;

  GPEngineCallback callback;
  gpointer         user_data;
};

struct _GPEngineClass {
  ModuleClass parent_class;
};

ModuleType        gpengine_get_type ();

GPEngine *        gpengine_new ();
void              gpengine_add_filter (GPEngine *gpe, Filter *filter);

int               gpengine_start (GPEngine *gpe, guint flags, Image *image, ImageCache *cache,
				  GPEngineCallback callback, gpointer user_data);
void              gpengine_abort (GPEngine *gpe);
gfloat            gpengine_get_progress (GPEngine *gpe);
int               gpengine_is_running (GPEngine *gpe);

/* internal */
ParamValue *      gpengine_get_param_value (GPEngine *gpe, int filter_no, char *name, ParamType type);

int               gpengine_pipeline_init (GPEngine *gpe, 
					  int width, int height, int bpp,
					  int src_width, int src_height, int src_bpp,
					  int src_tile_width, int src_tile_height);
int               gpengine_pipeline_process_tile (GPEngine *gpe, FilterTile *tile);

ImageCache *      gpengine_get_cache (GPEngine *gpe);
void              gpengine_set_cache (GPEngine *gpe, ImageCache *cache);

#endif

