
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: image.c,v 1.6 2000/04/28 20:46:34 dr Exp $ 
*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <pthread.h>

#include "directory.h"
#include "codec.h"

#define DEBUG(str)      puts (str);

#define LOCK()        pthread_mutex_lock (&_images_mutex_);
#define UNLOCK()      pthread_mutex_unlock (&_images_mutex_);

static GCompareFunc       image_compare_funcs [] = {
  (GCompareFunc) image_compare_by_name,
  (GCompareFunc) image_compare_by_size,
  (GCompareFunc) image_compare_by_mtime
};

static pthread_mutex_t _images_mutex_   = PTHREAD_MUTEX_INITIALIZER;

static guint           d_images         = 0;
static guint           d_images_caches  = 0;
static guint           d_images_pixmaps = 0;

int 
is_readable_image (char *file_name, char *directory) {

  return codec_can_read_image (file_name, directory);
}

Image *                   
image_new (char *file_name, char *directory) {
  Image *image;
  
  g_return_val_if_fail (file_name && directory, NULL);

  image             = g_new (Image, 1);
  image->refcount   = 1;
  image->file_name  = g_strdup (file_name);
  image->directory  = directory_ref_by_name (directory);
  image->file_size  = 0;
  image->file_mtime = 0;
  image->width      = 0;
  image->height     = 0;
  image->bpp        = 0;

  image->caches     = NULL;
  image->pixmaps    = NULL;

  d_images ++;

  return image;
}

Image *                      
image_ref (Image *image) {

  g_return_val_if_fail (image, NULL);

  g_return_val_if_fail (image->refcount < UINT_MAX, NULL);
  image->refcount ++;

  return image;
}

void                      
image_unref (Image *image) {
  g_return_if_fail (image);

  image->refcount --;

  if (image->refcount == 0) {
    d_images --;

    directory_unref_by_name (image->directory);
    if (image->caches)
      g_list_free (image->caches);

    g_free (image);
  }
}

char *
image_get_path (Image *image) {
  char buf [PATH_MAX];

  g_return_val_if_fail (image, NULL);

  snprintf (buf, PATH_MAX, "%s/%s", image->directory, image->file_name);
  return g_strdup (buf);
}

/*
  Image caches 
*/

/*
  XXX: assumes caches have the same aspect ratio & same depth
*/
static gint
image_cache_compare (ImageCache *ic1, ImageCache *ic2) {
  return ic2->width - ic1->width;
}

ImageCache *                      
image_cache_new (Image *image, int width, int height, int bpp) {
  ImageCache *cache;

  g_return_val_if_fail (image, NULL);
  g_return_val_if_fail (width > 0 && height > 0 && bpp > 0, NULL);

  cache           = g_new (ImageCache, 1);
  cache->refcount = 1;
  cache->width    = width;
  cache->height   = height;
  cache->bpp      = bpp;
  cache->data     = g_malloc (width * height * bpp);

  LOCK ();
  image->caches   = g_list_insert_sorted (image->caches, 
					  cache, 
					  (GCompareFunc) image_cache_compare);
  UNLOCK ();
  d_images_caches ++;

  return cache;
}

void
image_cache_add (Image *image, ImageCache *cache) {

  g_return_if_fail (image && cache);

  LOCK ();
  image->caches   = g_list_insert_sorted (image->caches, 
					  cache, 
					  (GCompareFunc) image_cache_compare);
  UNLOCK ();

  image_cache_ref (image, cache);
}

ImageCache *
image_cache_ref (Image *image, ImageCache *cache) {

  g_return_val_if_fail (image && cache, NULL);  

  g_return_val_if_fail (cache->refcount < UINT_MAX, NULL);
  cache->refcount ++;

  return cache;
}

void                      
image_cache_unref (Image *image, ImageCache *cache) {
  GList *list;
  g_return_if_fail (image && cache);

  LOCK ();
  list = g_list_find (image->caches, cache);
  UNLOCK ();

  g_return_if_fail (cache);

  cache->refcount--;

  if (cache->refcount == 0) {
    d_images_caches --;

    image->caches = g_list_remove_link (image->caches, list);
    
    if (cache->data)
      g_free (cache->data);
    g_free (cache);
  }
}

ImageCache *              
image_cache_get (Image *image, int width, int height) {
  ImageCache *cache;
  GList *list;
  g_return_val_if_fail (image, NULL);

  if (width < 0)
    width = image->width;

  if (height < 0)
    height = image->height;

  list = image->caches;
  while (list) {
    cache = list->data;
    list = list->next;

    if (cache->width >= width && cache->height >= height)
      return cache;
  }

  return NULL;
}

/*
  Image pixmaps
*/

static gint
image_pixmap_compare (ImagePixmap *ip1, ImagePixmap *ip2) {
  return ip2->width - ip1->width;
}

ImagePixmap *             
image_pixmap_new (Image *image, int width, int height) {
  ImagePixmap *ipix;

  g_return_val_if_fail (image, NULL);
  g_return_val_if_fail (width > 0 && height > 0, NULL);

  ipix = g_new (ImagePixmap, 1);
  ipix->refcount = 1;
  ipix->width    = width;
  ipix->height   = height;
  ipix->pixmap   = gdk_pixmap_new (NULL, width, height,
				   gdk_visual_get_best_depth ());
  ipix->mask     = NULL;

  LOCK ();
  image->pixmaps = g_list_insert_sorted (image->pixmaps, 
					 ipix,
					 (GCompareFunc) image_pixmap_compare);

  UNLOCK ();

  d_images_pixmaps ++;

  return ipix;
}

ImagePixmap *             
image_pixmap_ref (Image *image, ImagePixmap *ipix) {
  g_return_val_if_fail (image && ipix, NULL);

  g_return_val_if_fail (ipix->refcount < UINT_MAX, NULL);
  ipix->refcount ++;

  return ipix;
}

void                      
image_pixmap_unref (Image *image, ImagePixmap *ipix) {
  
  g_return_if_fail (image && ipix);

  ipix->refcount --;

  if (ipix->refcount == 0) {
    d_images_pixmaps --;

    gdk_pixmap_unref (ipix->pixmap);
    g_free (ipix);
  }
}

ImagePixmap *             
image_pixmap_get (Image *image, int width, int height) {
  GList *list;
  ImagePixmap *ipix;

  if (width < 0) 
    width = image->width;

  if (height < 0)
    height = image->height;

  list = image->pixmaps;
  while (list) {
    ipix = list->data;
    list = list->next;

    if (ipix->width >= width && ipix->height >= height)
      return ipix;
  }

  return NULL;
}

/* 
   compare funcs
*/

int
image_compare_by_name (const Image **i1, const Image **i2) {
  return strcmp ((*i1)->file_name, (*i2)->file_name);
}

int
image_compare_by_size (const Image **i1, const Image **i2) {
  return (int) ((*i1)->file_size - (*i2)->file_size);
}

int
image_compare_by_mtime (const Image **i1, const Image **i2) {
  return (int) ((*i1)->file_mtime - (*i2)->file_mtime);
}
 
/*********************
 *     ImageSet      *
 *********************/

ImageSet *                
image_set_new () {
  ImageSet *set;

  set               = g_new (ImageSet, 1);
  set->images       = g_array_new (1, 0, sizeof (Image *));
  set->compare      = (GCompareFunc) image_compare_by_name;

  return set;
}

ImageSet *
image_set_copy (ImageSet *set) {
  ImageSet *cset;
  Image *image;
  int i;

  g_return_val_if_fail (set, NULL);

  cset = image_set_new ();
  
  for (i = 0; i < image_set_size (set); i++) {
    image = image_set_get_image (set, i);
    cset->images = g_array_append_val (cset->images, image);
    image_ref (image);
  }

  return cset;
}

void
image_set_free (ImageSet *set) {
  g_return_if_fail (set);

  image_set_clear (set);
  g_free (set);
}

void
image_set_clear (ImageSet *set) {
  int i;

  g_return_if_fail (set);

  for (i = 0; i < image_set_size (set); i++)
    image_unref (image_set_get_image (set, i));
  
  g_array_set_size (set->images, 0);
}

void                      
image_set_add (ImageSet *set, Image *image) {

  g_return_if_fail (set && image);

  image_ref (image);
  set->images       = g_array_append_val (set->images, image);
}

void                      
image_set_remove (ImageSet *set, Image *image) {
  int index;

  g_return_if_fail (set && image);

  index             = image_set_get_index (set, image);
  g_return_if_fail (index != -1);
  set->images       = g_array_remove_index (set->images, index);
  image_unref (image);
}

int
image_set_get_index (ImageSet *set, Image *image) {
  int i;
  
  g_return_val_if_fail (set && image, -1);

  for (i = 0; i < image_set_size (set); i++)
    if (image_set_get_image (set, i) == image)
      return i;

  return -1;
}

void                      
image_set_sort (ImageSet *set, ImageSetSortType type) {
  
  g_return_if_fail (set);

  set->compare = image_compare_funcs [type];
  qsort (set->images->data, set->images->len, sizeof (Image *), set->compare); 
}

void
image_print_leaks () {
  printf ("Memory leaks summary:\n"
	  "  images not free  :%d\n"
	  "  caches not free  :%d\n"
	  "  pixmaps not free :%d\n",
	  d_images,
	  d_images_caches,
	  d_images_pixmaps);
}
