
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: jpeg.c,v 1.4 2000/04/28 20:46:34 dr Exp $ 
*/


#include <stdio.h>
#include <math.h>
#include <setjmp.h>
#include <jpeglib.h>
#include "jpeg.h"

typedef struct _JpegErrorMgr      JpegErrorMgr;

struct _JpegErrorMgr {
  struct jpeg_error_mgr pub;
  jmp_buf setjmp_buffer;
};


static void            jpeg_class_init (ModuleClass *class);
static void            jpeg_init (Module *module);

static int             jpeg_can_read (Codec *codec);
static int             jpeg_read_image_infos (Codec *codec);
static int             jpeg_read_image (Codec *codec);
static int             jpeg_write_image (Codec *codec);

ModuleType 
jpeg_get_type () {
  static ModuleType type = 0;

  if (!type) {
    static ModuleTypeInfo jpeg_info = {
      "Jpeg",
      NULL,
      NULL,
      sizeof (Jpeg),
      sizeof (JpegClass),
      jpeg_init,
      jpeg_class_init,
      NULL,
      NULL
    };

    type = module_type_unique (codec_get_type (), &jpeg_info);
  }

  return type;
}

static void
jpeg_class_init (ModuleClass *class) {
  static char *jpeg_extensions = ".jpg.jpeg.";
  CodecFunc *operations;

  CodecClass *codec_class;

  codec_class = (CodecClass *) class;
  
  codec_class->extensions  = jpeg_extensions;
  operations               = codec_class->operations;            

  operations [CODEC_CAN_READ_IMAGE]    = jpeg_can_read;
  operations [CODEC_READ_IMAGE_INFOS]  = jpeg_read_image_infos;
  operations [CODEC_READ_IMAGE]        = jpeg_read_image;
  operations [CODEC_WRITE_IMAGE]       = jpeg_write_image;
}

static void
jpeg_init (Module *module) {

}

static int             
jpeg_can_read (Codec *codec) {
  guchar buff [4];
  
  if (!fread (buff, 4, 1, codec->file))
    return 0;

  return buff [0] == 0xff && buff [1] == 0xd8 && buff [2] == 0xff;
}


static void
jpeg_error_exit (j_common_ptr cinfo) {
    JpegErrorMgr *jem;

    jem = (JpegErrorMgr *) cinfo->err;
    
    (*cinfo->err->output_message) (cinfo);

    longjmp (jem->setjmp_buffer, 1);
}

static int             
jpeg_read_image_infos (Codec *codec) {

  return 1;
}

static int             
jpeg_read_image (Codec *codec) {
  struct jpeg_decompress_struct cinfo;
  JpegErrorMgr jem;
  FilterTile *buffer;
  int    i;

  g_return_val_if_fail (codec && codec->file, 0);

  cinfo.err            = jpeg_std_error (&jem.pub);
  jem.pub.error_exit   = jpeg_error_exit;

  jpeg_create_decompress (&cinfo);

  if (setjmp (jem.setjmp_buffer)) {
    jpeg_destroy_decompress (&cinfo);
    return 0;
  }

  jpeg_stdio_src (&cinfo, codec->file);
  jpeg_read_header (&cinfo, 1);

  i = codec->bounding_box_size > 0 ? cinfo.image_width / codec->bounding_box_size : 1;
  
  if (i > 7)
    cinfo.scale_denom = 8;
  else if (i > 3)
    cinfo.scale_denom = 4;
  else if (i > 1)
    cinfo.scale_denom = 2;
  else
    cinfo.scale_denom = 1;

  cinfo.scale_num = 1;
  cinfo.dct_method = JDCT_FASTEST;

  jpeg_calc_output_dimensions (&cinfo);


  codec_init (codec, 
	      cinfo.image_width, cinfo.image_height, cinfo.num_components,
	      cinfo.output_width, cinfo.output_height, cinfo.output_components, 
	      cinfo.output_width, 1);
  buffer = codec_get_buffer (codec);
  
  jpeg_start_decompress (&cinfo);

  while (cinfo.output_scanline < cinfo.output_height &&
	 jpeg_read_scanlines (&cinfo, &buffer->data, 1))
    if (!codec_add_tile (codec, buffer)) {
      jpeg_destroy_decompress (&cinfo);
      return 0;
    }

  jpeg_finish_decompress (&cinfo);
  jpeg_destroy_decompress (&cinfo);
  
  return 1;
}

static int             
jpeg_write_image (Codec *codec) {
  struct jpeg_compress_struct cinfo;
  JpegErrorMgr jem;
  ImageCache *cache;
  guchar *row;
  int    i, row_stride;

  g_return_val_if_fail (codec && codec->file, 0);

  cache = codec_get_cache (codec);
  g_return_val_if_fail (cache, 0);

  cinfo.err            = jpeg_std_error (&jem.pub);
  jem.pub.error_exit   = jpeg_error_exit;
  
  jpeg_create_compress (&cinfo);

  if (setjmp (jem.setjmp_buffer)) {
    jpeg_destroy_compress (&cinfo);
    return 0;
  }

  jpeg_stdio_dest (&cinfo, codec->file);
  
  cinfo.image_width      = cache->width;
  cinfo.image_height     = cache->height;
  cinfo.input_components = cache->bpp;
  cinfo.in_color_space   = cache->bpp == 3 ? JCS_RGB : JCS_GRAYSCALE;
  
  jpeg_set_defaults (&cinfo);
  jpeg_start_compress (&cinfo, TRUE);

  row = cache->data;
  row_stride = cache->width * cache->bpp;

  for (i = 0; i < cache->height; i++) {
    jpeg_write_scanlines (&cinfo, &row, 1);
    row += row_stride;
  }

  jpeg_finish_compress (&cinfo);
  jpeg_destroy_compress (&cinfo);
  
  return 1;
}






