
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: module_type.c,v 1.5 2000/04/28 20:46:34 dr Exp $ 
*/


#include <string.h>
#include <pthread.h>
#include "module_type.h"

#define MTI_PARENT(mti)        ( (ModuleTypeInfo *) (mti)->reserved1)
#define MTI_CLASS(mti)         ( (ModuleClass *) (mti)->reserved2)

static void               module_type_init_class (ModuleTypeInfo *info, ModuleClass *class);
static void               module_type_init_module (ModuleTypeInfo *info, Module *module);


static GList *                   s_type_info_list = NULL;
static int                       s_type_info_list_size = 0;
static pthread_mutex_t           s_type_info_list_mutex = PTHREAD_MUTEX_INITIALIZER;

/*
#define pthread_mutex_lock(t)
#define pthread_mutex_unlock(t)
*/

ModuleType 
module_type_unique (ModuleType parent_type, ModuleTypeInfo *info) {
  ModuleClass *class;
  ModuleTypeInfo *parent_info;

  g_return_val_if_fail (parent_type <= s_type_info_list_size && info, 0);
    
  pthread_mutex_lock (&s_type_info_list_mutex);

  /* create the type list (which realy should be an array)
     if it doesn't exists 
  */
  if (s_type_info_list == NULL) {
    s_type_info_list = g_list_append (s_type_info_list, NULL); /* type != 0 */
    s_type_info_list_size = 1;
  }

  class = g_malloc (info->class_size);
  parent_info = parent_type > 0 ? g_list_nth_data (s_type_info_list, parent_type) : NULL;
 
  bzero (class, info->class_size);

  /* setup the 'info' variable */
  MTI_PARENT(info) = parent_info;
  MTI_CLASS(info)  = class;

  /* setup the Module's class */
  class->parent_class = parent_info ? MTI_CLASS(parent_info) : NULL;
  class->type        = s_type_info_list_size;

  /* create a parameter node and link it to the
     parent's class one
  */
  class->params_node = param_node_new_nl (info->name);
  param_node_add_child (class->parent_class ?
			class->parent_class->params_node : NULL, 
			class->params_node);

  /* recusive initialization of the class */
  module_type_init_class (info, class);
  
  s_type_info_list = g_list_append (s_type_info_list, info);
  s_type_info_list_size ++;

  pthread_mutex_unlock (&s_type_info_list_mutex);
   
  return class->type;
}

Module *
module_type_new (ModuleType type) {
  ModuleTypeInfo *info;
  Module *module;

  g_return_val_if_fail (type > 0 && type < s_type_info_list_size, NULL);

  info = g_list_nth_data (s_type_info_list, type);
  g_return_val_if_fail (info, NULL);

  module = g_malloc (info->module_size);
  bzero (module, info->module_size);
  
  module->class = MTI_CLASS(info);
  module_type_init_module (info, module);

  return module;
}

void
module_type_free (Module *module) {
  ModuleClass *class;

  class = MODULE_CLASS(module);
  if (class->free)
    (* class->free) (module);
    
  g_free (module);
}

static void
module_type_init_class (ModuleTypeInfo *info, ModuleClass *class) {
 
  if (MTI_PARENT(info))
    module_type_init_class (MTI_PARENT(info), class);
  
  if (class && info->module_class_init)
    (*info->module_class_init) (class);
}

static void
module_type_init_module (ModuleTypeInfo *info, Module *module) {

  if (MTI_PARENT(info))
    module_type_init_module (MTI_PARENT(info), module);

  if (module && info->module_init)
    (*info->module_init) (module);
}

int
module_type_is_a (Module *module, ModuleType type) {
  ModuleClass *class;

  g_return_val_if_fail (module, 0);

  class = MODULE_CLASS(module);
  while (class) {
    if (class->type == type)
      return 1;

    class = class->parent_class;
  }

  return 0;
}









