
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: scale.c,v 1.5 2000/04/28 20:46:35 dr Exp $ 
*/


#include <stdio.h>
#include "scale.h"

#define ALPHA_BLEND(src, bg, alpha)        (((alpha) * (src) + (255 - (alpha)) * (bg)) >> 8)
//#define ALPHA_BLEND(src, bg, alpha)        (((alpha) * (src)  + (255 - (alpha)) * (bg)) / 255);


static void      scale_class_init (ModuleClass *class);
static void      scale_init (Module *module);

static int       scale_filter_init (Filter *filter);
static int       scale_filter_process_tile (Filter *filter, FilterTile *in, FilterTile *out);
static void      scale_filter_end (Filter *filter);


ModuleType
scale_get_type () {
  static ModuleType type = 0;

  if (!type) {
    static ModuleTypeInfo scale_info = {
      "Scale",
      NULL,
      NULL,
      sizeof (Scale),
      sizeof (ScaleClass),
      scale_init,
      scale_class_init,
      /* reserved1 */ NULL,
      /* reserved2 */ NULL
    };

    type = module_type_unique (filter_get_type (), &scale_info);
  }

  return type;
}

static void
scale_class_init (ModuleClass *class) {
  ScaleClass  *scale_class;
  FilterClass *filter_class;
  
  filter_class     = (FilterClass *) class;
  scale_class      = (ScaleClass *) class;

  filter_class->flags                  |= FILTER_NEED_BUFFER;

  filter_class->filter_init            = scale_filter_init;
  filter_class->filter_process_tile    = scale_filter_process_tile;
  filter_class->filter_end             = scale_filter_end;

}

static void
scale_init (Module *module) {

}

Module *
scale_new (int flags, 
	   int bounding_box_width, int bounding_box_height, 
	   GdkColor *background) {
  Module *module;
  Scale *scale;

  module = module_type_new (scale_get_type ());

  scale  = (Scale *) module;
  scale->flags               = flags;
  scale->bounding_box_width  = bounding_box_width;
  scale->bounding_box_height = bounding_box_height;


  if (background) {
    scale->background.red   = background->red >> 8;
    scale->background.green = background->green >> 8;
    scale->background.blue  = background->blue >> 8;
  } else 
    scale->background.red = 
    scale->background.green = 
    scale->background.blue = 0;

  return module;
}

static int
scale_filter_init (Filter *filter) {
  Scale *scale;
  double scale_factor, p;
  int x, y;

  g_return_val_if_fail (filter, 0);

  scale = (Scale *) filter;
  scale->src_row_stride = filter->src_width * filter->src_bpp;

  if ((scale->bounding_box_width <= 0 || scale->bounding_box_height <= 0) ||
      ((scale->flags & SCALE_DONT_MAGNIFY) &&
       filter->src_width < scale->bounding_box_width &&
       filter->src_height < scale->bounding_box_height))
    scale_factor       = 1.0;
  else 
    scale_factor = MAX ((double) filter->src_width / scale->bounding_box_width,
			(double) filter->src_height / scale->bounding_box_height);

  filter->out_width  = 2 + (double) filter->src_width / scale_factor;
  filter->out_height = 2 + (double) filter->src_height / scale_factor;

  filter->out_tile_width  = filter->out_width;
  filter->out_tile_height = 1 + (double) 1 / scale_factor;

  scale->x_map = g_new (int, filter->out_width + 1);
  scale->y_map = g_new (int, filter->out_height + 1);

  for (x = 0, p = 0.0; (int) p < filter->src_width; x++, p += scale_factor)
    scale->x_map [x] = filter->src_bpp * (int) p;
  scale->x_map [x] = -1;
  g_return_val_if_fail (x <= filter->out_width, 0);
  filter->out_width = x;

  for (y = 0, p = 0.0; (int) p < filter->src_height; y++, p += scale_factor)
    scale->y_map [y] = (int) p;
  scale->y_map [y] = -1;
  g_return_val_if_fail (y <= filter->out_height, 0);
  filter->out_height = y;

  filter->out_bpp = 
    (filter->src_bpp == 2) || (filter->src_bpp == 4) ? 
    filter->src_bpp - 1 : filter->src_bpp;

  scale->out_row = 0;
  scale->src_row = 0;

  if (filter->src_bpp == 2) 
    scale->background.red = 
      scale->background.red | 
      scale->background.green | 
      scale->background.blue;

  return 1;
}

int
scale_process_tile_parts (Scale *scale, FilterTile *in, FilterTile *out, int sx, int ex) {
  register guchar *src, *dest;
  guchar *out_row;
  register int *xms, *xme;
  int y, out_stride;

  g_return_val_if_fail (scale && in && out, 0);
  g_return_val_if_fail (sx >= 0 && ex <= FILTER (scale)->out_width, 0);
  
  dest        = out->data;
  src         = in->data;
  out->width  = ex - sx;
  out->height = 0;
  out_stride  = out->width * FILTER (scale)->out_bpp;
  xme         = &(scale->x_map [ex]);

  for (y = 0; y < in->height; y ++) {

    if (scale->y_map [scale->out_row] == -1 ||
	scale->y_map [scale->out_row] != scale->src_row) {
      scale->src_row ++;
      src += scale->src_row_stride;
      continue;
    }
  
    out_row = dest;
    xms     = &(scale->x_map [sx]);

    switch (FILTER (scale)->src_bpp) {
    case 1:
      for (; xms < xme; xms ++)
	*dest ++ = src [*xms];
      
      break;
    case 2:
		
      for (; xms < xme; xms ++) 
	*dest++ = ALPHA_BLEND (src [*xms], 
			       scale->background.red, 
			       src [*xms + 1]);
      
      break;
    case 3:
      
      for (; xms < xme; xms ++) {
	dest [0] = src [*xms + 0];
	dest [1] = src [*xms + 1];
	dest [2] = src [*xms + 2];
	dest  += 3;
      }
      
      break;
    case 4:
      
      for (; xms < xme; xms ++) {
	dest [0] = ALPHA_BLEND (src [*xms + 0], 
				scale->background.red, 
				src [*xms + 3]);
	dest [1] = ALPHA_BLEND (src [*xms + 1], 
				scale->background.green , 
				src [*xms + 3]);
	dest [2] = ALPHA_BLEND (src [*xms + 2], 
				scale->background.blue  , 
				src [*xms + 3]);
	dest += 3;
      }
      
      break;
    default:
      break;
    }
    
    scale->out_row ++;
  
    out->height ++;
    
    while (scale->y_map [scale->out_row] == scale->src_row) {
      memcpy (dest, out_row, out_stride);

      dest += out_stride;
      scale->out_row ++;
      out->height ++;
    }
    
    scale->src_row ++;
    src += scale->src_row_stride;
  }

  return 1;
}
			
static int
scale_filter_process_tile (Filter *filter, FilterTile *in, FilterTile *out) {

  return scale_process_tile_parts (SCALE (filter), in, out, 0, filter->out_width);
}
				
static void	
scale_filter_end (Filter *filter) {
  Scale *scale;

  scale = (Scale *) filter;

  g_return_if_fail (scale);

  if (scale->x_map) {
    g_free (scale->x_map);
    scale->x_map = NULL;
  }

  if (scale->y_map) {
    g_free (scale->y_map);
    scale->y_map = NULL;
  }
}

