/*
 * page.c
 *	Page menu callbacks for mgv.
 *
 * Copyright (C) 1996  Eric A. Howe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:	Matthew D. Francey
 *		Eric A. Howe (mu@trends.net)
 */
#include <wlib/rcs.h>
MU_ID("$Mu: mgv/page.c,v 1.97 $")

#include <assert.h>
#include <ctype.h>
#include <stdlib.h>
#include <Xm/List.h>
#include <Xm/ToggleB.h>
#include <Xm/ScrollBar.h>

#include <wlib/wlib.h>
#include <wlib/typesanity.h>
#include <mine/mgv.h>
#include <mine/page.h>
#include <mine/log.h>
#include <mine/sens.h>
#include <mine/auto.h>
#include <mine/util.h>
#include <mine/page_control.h>

/*
 * popup->back, popup->forward
 */
void
mgv_page_backforward(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m = (MGV *)closure;
	USEUP(call);

	if(*XtName(w) == 'b')
		mgv_pgstack_back(m->pgstack);
	else
		mgv_pgstack_forward(m->pgstack);
}

/*
 * page->next, page->previous
 */
void
mgv_page_nextprev(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m    = (MGV *)closure;
	char	*name = XtName(w);
	int	offset;
	USEUP(call);
	assert(MgvOK(m));

	offset = (*name == 'n') ? 1 : -1;
	mgv_pgstack_goto(m->pgstack, m->page + offset);
}

/*
 * page list callback
 */
void
mgv_page_goto(Widget w, XtPointer closure, XtPointer call)
{
	XmLSS	*cbs = (XmLSS *)call;
	MGV	*m   = (MGV *)closure;
	USEUP(w);
	assert(MgvOK(m));

	if(m->page == cbs->item_position - 1)
		return;

	/*
	 * Jeez, XmList is one-based (probably so you can use position
	 * zero to indicate the end of the list when using the
	 * XmList.*() convenience functions, -1 would probably be a better
	 * value to indicate "the end"), how silly in a C environment
	 * (even windoze got this one right!).
	 */
	mgv_pgstack_goto(m->pgstack, cbs->item_position - 1);
}

/*
 * show/hide the page labels/numbers
 */
void
mgv_page_labels(Widget w, XtPointer closure, XtPointer call)
{
	XmTBS	*cbs = (XmTBS *)call;
	MGV	*m   = (MGV *)closure;
	Widget	p;
	assert(MgvOK(m));

	if(m->dsc == NULL
	|| m->dsc->pages == NULL)
		return;
	if(strcmp(XtName(w), "showLabels") == 0)
		m->labels = cbs->set;
	else
		m->pgnums = cbs->set;

	if(!m->labels && !m->pgnums) {
		w = wl_find1(m->main, "*popup*hideList");
		XmToggleButtonSetState(w, True, True);
		return;
	}
	if(!m->labels != !m->pgnums
	&& !XtIsManaged(XtParent(m->list))) {
		p = wl_find1(m->main, "*popup*hideList");
		assert(p != NULL);
		XmToggleButtonSetState(p, False, True);
	}

	XmListDeleteAllItems(m->list);
	mgv_list_update(m, -1);

	/*
	 * There doesn't seem to be any simple way to retain
	 * the scroll position.  If you replace in the loop rather
	 * than rebuilding the list, the list won't resize properly
	 * when you the labels go away (i.e. it won't shrink).
	 */
	XmListSetBottomPos(m->list, m->page + 1);

	/*
	 * And the sizing trick (ugh!).
	 */
	p = XtParent(m->list);
	if(!XtIsManaged(p))
		return;
	XtUnmanageChild(p);
	XtVaSetValues(p, XmNrightAttachment, XmATTACH_FORM, NULL);
	XtManageChild(p);
}

/*
 * page->center
 */
void
mgv_page_center(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m = (MGV *)closure;
	WL_SBV	v;
	USEUP(w); USEUP(call);
	assert(MgvOK(m));

	memset((void *)&v, '\0', sizeof(v));

	w = wl_getsbvalues(m->main, "*view*VertScrollBar", &v);
	v.value = (v.min + v.max - v.slider)/2;
	wl_setsbvalues(w, &v, True);

	w = wl_getsbvalues(m->main, "*view*HorScrollBar", &v);
	v.value = (v.min + v.max - v.slider)/2;
	wl_setsbvalues(w, &v, True);
}

/*
 * page->redisplay
 */
void
mgv_page_redisplay(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m = (MGV *)closure;
	USEUP(w); USEUP(call);
	assert(MgvOK(m));

	mgv_show(m, m->page);
}

/*
 * scrolling action
 * You're not supposed to fiddle with the scrollbars in a scrolled window
 * but it works so screw the OSF.
 *
 * I really should be using application-defined scrolling these days but
 * I want to get 2.0 out so I can start breaking things again.
 *,,,
 */
#define	UP	1
#define	DOWN	2
#define	LEFT	3
#define	RIGHT	4
void
mgv_action_scroll(Widget w, XEvent *ev, String *argv, Cardinal *argc)
{
	MGV	*m;
	char	action[32], magical[8];
	int	i, way, force, value, incr, dir;
	WL_SBV	v;
	USEUP(argc); USEUP(ev);

	w = wl_findup(w, "mainWindow");
	assert(w != NULL);
	XtVaGetValues(w, XmNuserData, &m, NULL);
	assert(MgvOK(m));

	/*
	 * A general action parser (which used the Xt type converters) would
	 * be a nice addition to wlib but we'll have to see if the type
	 * converters can be used this way.
	 */
	memset((void *)&action[0], '\0', sizeof(action));
	if(*argc >= 1) {
		for(i=0; i < (int)sizeof(action) - 1 && argv[0][i] != '\0'; ++i)
			action[i] = tolower(argv[0][i]);
	}
	memset((void *)&magical[0], '\0', sizeof(magical));
	if(*argc >= 2) {
		for(i = 0; i < (int)sizeof(magical)-1 && argv[1][i] != '\0';++i)
			magical[i] = tolower(argv[1][i]);
	}
	force = 1;
	if(*argc >= 3) {
		if((force = atoi(argv[2])) < 0)
			force *= -1;
		else if(force == 0)
			force  =  1;
	}
	memset((void *)&v, '\0', sizeof(v));
	if(strcmp(action, "down") == 0) {
		dir =  1;
		way = DOWN;
		w   = wl_getsbvalues(w, "*view*VertScrollBar", &v);
	}
	else if(strcmp(action, "up") == 0) {
		dir = -1;
		way = UP;
		w   = wl_getsbvalues(w, "*view*VertScrollBar", &v);
	}
	else if(strcmp(action, "right") == 0) {
		dir =  1;
		way = RIGHT;
		w   = wl_getsbvalues(w, "*view*HorScrollBar",  &v);
	}
	else if(strcmp(action, "left") == 0) {
		dir = -1;
		way = LEFT;
		w   = wl_getsbvalues(w, "*view*HorScrollBar",  &v);
	}
	else {
		XBell(XtDisplay(w), 100);
		return;
	}

	/*
	 * Check magic scrolling.
	 * We abuse the MgvS things a little bit here but it gives
	 * a fairly simple solution.  We also query the scrollbar values
	 * again after we move to the new page since the page size can
	 * change.
	 *
	 * This chunk is a little convoluted because of the magic resistance.
	 */
	if((way == UP || way == DOWN)
	&& (m->bits & MgvMAGICSCROLL)
	&& strcmp(magical, "magical") == 0) {
		/*
		 * Some would consider this wonderful little construction
		 * an abuse--I call it a valid and wholesome technique.
		 */
		int hehe = (way == UP)  + 2*(m->last_magic >  0)
					+ 4*(m->last_magic == 0);
		switch(hehe) {
		case 0:	case 4:	m->last_magic -= force;	break;
		case 1:		m->last_magic  = 0;	break;
		case 2:		m->last_magic  = 0;	break;
		case 3: case 5:	m->last_magic += force;	break;
		default:
			/* can't happen (unless your compiler is busted) */
			assert("busted compiler!" != NULL);
			break;
		}

		if(way     == UP
		&& v.value == v.min
		&& (m->sens & MgvSNOTFIRST)) {
			if(m->last_magic < m->magic_resistance)
				return;
			mgv_pgstack_goto(m->pgstack, m->page - 1);
			w = wl_getsbvalues(m->main, "*view*VertScrollBar", &v);
			v.value = v.max - v.slider;
			wl_setsbvalues(w, &v, True);
			m->last_magic = 0;
			return;
		}
		if(way == DOWN
		&& v.value + v.slider == v.max
		&& (m->sens & MgvSNOTLAST)) {
			if(m->last_magic > -m->magic_resistance)
				return;
			mgv_pgstack_goto(m->pgstack, m->page + 1);
			w = wl_getsbvalues(m->main, "*view*VertScrollBar", &v);
			v.value = 0;
			wl_setsbvalues(w, &v, True);
			m->last_magic = 0;
			return;
		}
	}
	m->last_magic = 0;

	/*
	 * We move m->scrollfactor of the _slider_ size (the slider
	 * represents one screen full in the scrollbar's coordinate
	 * system).  This makes things easy and sensible for both me,
	 * the programmer, and you, the not-me's (I could have said
	 * "users" but I'm a user too so that would be a misleading
	 * thing to say).
	 *
	 * Once we have the final value (unadjusted for page size
	 * limits), we can compute the individual increments based
	 * on the m->smoothness value (which is the percentage of
	 * the total increment that each little hop should be).
	 */
	value = v.slider * m->scrollfactor;
	if((incr = value / m->smoothness) <= 0)
		incr = 1;
	incr *= dir;
	value = v.value + dir*value;
	if(value < v.min)
		value = v.min;
	else if(value > v.max - v.slider)
		value = v.max - v.slider;

	if((m->bits & MgvSMOOTH) && incr < 0) {
		while((v.value += incr) > value)
			wl_setsbvalues(w, &v, True);
	}
	else if((m->bits & MgvSMOOTH) && incr > 0) {
		while((v.value += incr) < value)
			wl_setsbvalues(w, &v, True);
	}
	v.value = value;
	wl_setsbvalues(w, &v, True);
}

/*
 * page->auto size
 */
void
mgv_page_size(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m   = (MGV *)closure;
	XmTBS	*cbs = (XmTBS *)call;
	assert(MgvOK(m));

	if(!cbs->set)
		return;

	/*
	 * "What's in a name...", a bloody hell of a lot I'd say.
	 */
	switch(*XtName(w)) {
	case 's':	m->autosizer = mgv_auto_size;		break;
	case 'h':	m->autosizer = mgv_auto_height;		break;
	case 'w':	m->autosizer = mgv_auto_width;		break;
	case 'n':	m->autosizer = mgv_auto_none;		break;
	default:	assert("unknown toggle!" != NULL);	break;
	}
	m->autosizer(m);
}
