/*
 * about.c
 *	Standard about box convenience function.
 *
 * Copyright (C) 1996  Eric A. Howe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:	Matthew D. Francey
 *		Eric A. Howe (mu@trends.net)
 */
#include <wlib/rcs.h>
MU_ID("$Mu: wlib/about.c,v 1.23 $")

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <Xm/Text.h>

#include <wlib/wlib.h>
#include <wlib/sanity.h>
#include <wlib/about.h>

static WLC bldopts_cb[] = {
	{"*dismiss", XmNactivateCallback, 0, NULL, wl_dismiss},
	{NULL},
};
static WLW bldopts_kids[] = {
	{"dismiss",		wlPushB,	0,		NULL},
	{"sep",			wlSeparatorG,	0,		NULL},
	{"text",		wlScrolledText,	WL_MANAGE,	NULL},
	{NULL},
};
static WLW bldopts[] = {
	{"bldoptsDialog",	wlFormDialog,	0,	bldopts_kids},
	{NULL},
};

#define	ABOUT_MAGIC1	0x41424f55
#define	ABOUT_MAGIC2	0x554f4221
typedef struct {
	long	magic1;
	Widget	about;
	Widget	bldopts;
	char	*(*version)(char *buffer);
	long	magic2;
} ABOUT;
#define	about_ok(a) (			\
	   (a)          != NULL		\
	&& (a)->magic1  == ABOUT_MAGIC1	\
	&& (a)->magic2  == ABOUT_MAGIC2	\
	&& (a)->version != NULL		\
	&& (a)->about   != NULL		\
)

/*
 * help->about->options
 */
static void
about_options(Widget w, XtPointer closure, XtPointer call)
{
	ABOUT	*a = (ABOUT *)closure;
	char	buf[2048];
	Widget	text;
	USEUP(w); USEUP(call);
	assert(about_ok(a));

	if(a->bldopts != NULL) {
		XtManageChild(a->bldopts);
		XtPopup(XtParent(a->bldopts), XtGrabNone);
		return;
	}

	if((a->bldopts = wl_create(a->about, bldopts)) == NULL)
		return;

	wl_find(a->bldopts, &text, "*textSW*text", NULL);
	memset((void *)&buf[0], '\0', sizeof(buf));
	XmTextInsert(text, 0, a->version(&buf[0]));
	XmTextSetCursorPosition(text, 0);

	wl_callback(a->bldopts, &bldopts_cb[0], (XtPointer)a);
	XtManageChild(a->bldopts);
	XtPopup(XtParent(a->bldopts), XtGrabNone);
}

static void
pixmap_free(Widget w, XtPointer closure, XtPointer call)
{
	Pixmap	pix;

	USEUP(closure); USEUP(call);
	pix = None;
	XtVaGetValues(w, XmNlabelPixmap, &pix, NULL);
	wl_free_pixmap(XtDisplay(wl_top(w)), pix);
}

static void
death(Widget w, XtPointer closure, XtPointer call)
{
	ABOUT	*a = (ABOUT *)closure;
	USEUP(w); USEUP(call);
	assert(about_ok(a));
	free((void *)a);
}

/*
 * the about dialog
 */
static WLC about_cb[] = {
	{"*options",     XmNactivateCallback, 0, NULL, about_options},
	{"*dismiss",     XmNactivateCallback, 0, NULL, wl_dismiss},
	{"*icon",        XmNdestroyCallback,  0, NULL, pixmap_free},
	{"*aboutDialog", XmNdestroyCallback,  0, NULL, death},
	{NULL}
};
static WLW aboutform[] = {
	{"icon",	wlLabel,		0,	NULL},
	{"header",	wlLabelG,		0,	NULL},
	{NULL}
};
static WLW aboutdlg[] = {
	{"options",	wlPushB,	0,		NULL},
	{"dismiss",	wlPushB,	0,		NULL},
	{"sep",		wlSeparatorG,	0,		NULL},
	{"form",	wlForm ,	WL_MANAGE,	aboutform},
	{"footer",	wlLabelG,	0,		NULL},
	{"text",	wlScrolledText,	WL_MANAGE,	NULL},
	{NULL},
};
static WLW about[] = {
	{"aboutDialog",	wlFormDialog,	0,		aboutdlg},
	{NULL},
};

/*
 *,,, This should probably be elsewhere.
 */
static char *minigpl[] = {
	"This program is free software; you can redistribute it and/or modify\n",
	"it under the terms of the GNU General Public License as published by\n",
	"the Free Software Foundation; either version 2 of the License, or\n",
	"(at your option) any later version.\n",
	"\n",
	"This program is distributed in the hope that it will be useful\n",
	"but WITHOUT ANY WARRANTY; without even the implied warranty of\n",
	"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n",
	"GNU General Public License for more details.\n",
	"\n",
	"You should have received a copy of the GNU General Public License\n",
	"along with this program; if not, write to the Free Software\n",
	"Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n",
	NULL
};

/*
 * help->about
 */
Widget
wl_about(Widget parent, const char *header, const char *footer,
		const char **cpright,
		Pixmap (*mkpix)(Widget, Pixel), char *(*v)(char *))
{
	ABOUT	*a;
	Widget	text, icon;
	char	**s;
	Pixel	bg;
	Pixmap	pix;

	if((a = calloc(1, sizeof(ABOUT))) == NULL
	|| (a->about = wl_create(parent, about)) == NULL) {
		free((void *)a);
		return NULL;
	}
	a->magic1  = ABOUT_MAGIC1;
	a->magic2  = ABOUT_MAGIC2;
	a->version = v == NULL ? wl_version : v;

	wl_find(a->about,
		&text,	"*textSW*text",
		&icon,	"*icon",
		NULL);

	XtVaSetValues(wl_find1(a->about, "*header"),
		XtVaTypedArg,	XmNlabelString,
				XmRString, header,
				strlen(header) + 1,
		NULL);
	XtVaSetValues(wl_find1(a->about, "*footer"),
		XtVaTypedArg,	XmNlabelString,
				XmRString, footer,
				strlen(footer) + 1,
		NULL);

	for(s = cpright == NULL ? minigpl : (char **)cpright; *s != NULL; ++s)
		XmTextInsert(text, XmTextGetLastPosition(text), *s);
	XmTextSetCursorPosition(text, 0);

	wl_callback(a->about, &about_cb[0], (XtPointer)a);

	XtVaGetValues(parent, XmNbackground, &bg, NULL);
	if(mkpix != NULL
	&& (pix = mkpix(parent, bg)) != None) {
		XtVaSetValues(icon,
			XmNlabelType,	XmPIXMAP,
			XmNlabelPixmap,	pix,
			NULL);
	}

	assert(about_ok(a));
	return a->about;
}
