/*
 * help-html.c
 *	Extra help support for HTML.
 *
 * Copyright (C) 1996  Matthew D. Francey
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:	Matthew D. Francey
 *		Eric A. Howe (mu@trends.net)
 */
#include <wlib/rcs.h>
MU_ID("$Mu: wlib/help-html.c,v 1.33 $")

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/WinUtil.h>

#include <wlib/wlib.h>
#include <wlib/wlibP.h>

typedef struct {
	int	port;
} CCIPORT;

static void
defcciport(Widget w, int offset, XrmValue *v)
{
	USEUP(w); USEUP(offset);
	v->size = sizeof(int);
	v->addr = XtMalloc(sizeof(int));
	*(int *)v->addr = getpid() % 2048 + 1024;
}

static int
cci_port(Widget w)
{
	CCIPORT	p;
	XtResource r = {
		XwlNhelpCCIPort, XwlCHelpCCIPort,
		XtRInt, sizeof(int), XtOffsetOf(CCIPORT, port),
		XtRCallProc, (XtPointer)defcciport
	};
	XtGetApplicationResources(wl_top(w), (XtPointer)&p, &r, 1, NULL, 0);
	return p.port;
}

static char *
urlify(char *url, Widget w, WLP_HELP *h)
{
	char	*browser, *prefix;

	browser = wlp_s(w, WlpShelpBrowser);
	prefix  = wlp_s(w, WlpShelpPrefix);

	if(h->section == NULL)
		sprintf(url, "%s%s.html", prefix, h->chapter);
	else
		sprintf(url, "%s%s.html#%s", prefix, h->chapter, h->section);

	return browser;
}

static pid_t
launch(char **argv)
{
	pid_t	brat;

	brat = fork();
	switch(brat) {
	case 0:
		execvp(*argv, argv);
		fprintf(stderr, "could not execute %s: %s\n",
						*argv, strerror(errno));
		exit(EXIT_FAILURE);
		break;
	case -1:
		break;
	default:
		break;
	}
	return brat;
}

/*
 * plain vanilla HTML help with no browser communication 
 */
Boolean
wlp_help_html(Widget w, WLP_HELP *h)
{
	char	*display, *fmt;
	char	url[512], *argv[5];
	int	n;

	memset((void *)&url[0], '\0', sizeof(url));
	n = 0;
	argv[n++] = urlify(url, w, h);
	if((display = DisplayString(XtDisplay(wl_top(w)))) != NULL) {
		argv[n++] = "-display";
		argv[n++] = display;
	}
	argv[n++] = url;
	argv[n++] = NULL;
	assert(ARGSANITY(argv, n));

	if(launch(&argv[0]) == -1) {
		int e = errno;
		fmt = wlp_s(w, WlpScantBrowse);
		wl_warn(wl_top(w), fmt, argv[0], strerror(e));
	}

	return True;
}

/*
 * attempt to launch Mosaic with a CCI connection
 */
static pid_t
opencci(Widget w, WLP_HELP *h)
{
	int	n;
	char	*argv[7];
	char	url[1024], buf[1024];
	char	*display;

	n = 0;
	argv[n++] = urlify(url, w, h);

	if((display = DisplayString(XtDisplay(wl_top(w)))) != NULL) {
		argv[n++] = "-display";
		argv[n++] = display;
	}
	argv[n++] = "-xrm";
	sprintf(buf, "*cciPort: %d", cci_port(w));
	argv[n++] = buf;
	argv[n++] = url;
	argv[n]   = NULL;

	return launch(&argv[0]);
}

/*
 * pull the host chunk out of an URL, if there is no host, then
 * default to "localhost"
 */
static char *
yankhost(char *url, char *s)
{
	char	*p, *q;

	strcpy(s, "localhost");
	if(strncmp(url, "file://", sizeof("file://") - 1) == 0)
		return s;
	if((p = strstr(url, "://"))  != NULL
	&& (q = strchr(p += 3, '/')) != NULL) {
		strncpy(s, p, q - p);
		s[q - p] = '\0';
	}
	return s;
}

/*
 * HTML help with Mosaic/CCI
 */
Boolean
wlp_help_cci(Widget w, WLP_HELP *h)
{
static	FILE	*fp   = NULL;
static	int	gotem = FALSE;
	char	url[1024], buf[1024], server[1024];
	char	*browser;

	browser = urlify(&url[0], w, h);
	if(!gotem) {
		if(opencci(w, h) == -1) {
			wl_warn(wl_top(w), wlp_s(w, WlpScantBrowse),
						browser, strerror(errno));
			return True;
		}
		gotem = TRUE;
		return True;
	}
	if(fp == NULL) {
		sprintf(server, "%s:%d", yankhost(url, buf), cci_port(w));
		if((fp = wlp_sopen(server, "r+b")) == NULL
		|| wlp_sgets(buf, sizeof(buf), fp) == NULL) {
			if(fp != NULL)
				wlp_sclose(fp);
			fp    = NULL;
			gotem = FALSE;
			wl_warn(wl_top(w), wlp_s(w, WlpScantBrowse),
						browser, strerror(errno));
			return True;
		}
	}

	/*
	 * flushing and what-not is done by the wlp_s* functions
	 */
	if(wlp_sprintf(fp, "get url <%s> output current\r\n", url) < 0
	|| wlp_sgets(&buf[0], sizeof(buf), fp) == NULL) {
		wlp_sclose(fp);
		fp    = NULL;
		gotem = FALSE;
		wl_warn(w, wlp_s(w, WlpScantBrowse),
						browser, strerror(errno));
		return True;
	}
	if(strncmp(buf, "501 ", 4) == 0)
		wl_warn(wl_top(w), wlp_s(w, WlpScantBrowse), browser, buf);
	return True;
}

/*
 * X error handler for catching BadWindow
 */
static	int		badwindow      = FALSE;
static	XErrorHandler	default_xerror = NULL;
static int
badwin_catcher(Display *dpy, XErrorEvent *ev)
{
	badwindow = (ev->error_code == BadWindow);
	return badwindow ? 0 : default_xerror(dpy, ev);
}

static Window
winping(Display *dpy, Window win, char *name, char *classname)
{
	XClassHint	hints = {NULL, NULL};
	int		status;

	if(win == None)
		return None;

	/*
	 * we have to watch for BadWindow errors in here
	 */
	XSync(dpy, False);
	default_xerror = XSetErrorHandler(badwin_catcher);
	win = XmuClientWindow(dpy, win);
	XSync(dpy, False);
	if(win == None || badwindow) {
		win = None;
		goto Hell;
	}
	status = XGetClassHint(dpy, win, &hints);
	XSync(dpy, False);
	if(status == 0 || badwindow) {
		win = None;
		goto Hell;
	}
	if(strcmp(name, hints.res_name)       != 0
	|| strcmp(classname, hints.res_class) != 0)
		win = None;
	XFree(hints.res_name);
	XFree(hints.res_class);
Hell:
	XSetErrorHandler(default_xerror);
	badwindow = FALSE;
	return win;
}

static Window
winfind(Display *dpy, Window guess, char *name, char *cls)
{
	Window		root, win, *kids;
	unsigned	n_kids, i;

	if((win = winping(dpy, guess, name, cls)) != None)
		return win;
	root = RootWindow(dpy, DefaultScreen(dpy));
	if(XQueryTree(dpy, root, &win, &win, &kids, &n_kids) == 0)
		return None;
	for(i = 0, win = None; win == None && i < n_kids; ++i)
		win = winping(dpy, kids[i], name, cls);
	XFree(kids);

	return win;
}

/*
 * HTML help with netscape's -remote thing
 */
Boolean
wlp_help_remote(Widget w, WLP_HELP *h)
{
static	Window	ns = None;
	Display	*dpy;
	char	*display;
	char	url[512], *argv[8];
	char	id[1024], act[1024];
	int	n;
	pid_t	brat;

	dpy = XtDisplay(wl_top(w));
	memset((void *)&url[0], '\0', sizeof(url));
	n = 0;
	argv[n++] = urlify(&url[0], w, h);
	if((display = DisplayString(dpy)) != NULL) {
		argv[n++] = "-display";
		argv[n++] = display;
	}
	if((ns = winfind(dpy, ns, "Navigator", "Netscape")) != None) {
		sprintf(id, "%ld", (long)ns);
		sprintf(act, "openURL(%s)", url);
		argv[n++] = "-id";
		argv[n++] = &id[0];
		argv[n++] = "-remote";
		argv[n++] = &act[0];
	}
	else {
		argv[n++] = url;
	}
	argv[n++] = NULL;
	assert(ARGSANITY(argv, n));

	if((brat = launch(&argv[0])) == -1) {
		int e = errno;
		wl_warn(wl_top(w), wlp_s(w,WlpScantBrowse),argv[0],strerror(e));
	}
	if(ns != None)
		waitpid(brat, NULL, 0);

	return True;
}
