/*
 * help-libhelp.c
 *	Extra help support for the libhelp help system.
 *
 * Copyright (C) 1997 Eric A. Howe
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:	Matthew D. Francey
 *		Eric A. Howe (mu@trends.net)
 */
#include <wlib/rcs.h>
MU_ID("$Mu: wlib/help-libhelp.c,v 1.11 $")

#include <stdlib.h>
#include <string.h>

#include <wlib/wlib.h>
#include <wlib/wlibP.h>

#if !defined(HAS_LIBHELP)
#	error "You forgot to define HAS_LIBHELP"
#endif

#if HAS_LIBHELP

#include <help.h>

#if !defined(HAS_PUTENV)
#	define	HAS_PUTENV	0
#endif

/*
 * putenv() isn't in POSIX so it either doesn't exist (not likely) or
 * we can't get at it (likely) so we do it ourselves.
 */
#if !HAS_PUTENV
#define	putenv	putenv_

extern	char	**environ;
static int
putenv_(const char *e)
{
static	char	**my_environ;
	char	*value;
	char	**p;
	int	len, i;
	char	**tmp;

	if((value = strrchr(e, '=')) == NULL)
		return -1;
	++value;
	len = strlen(e) - strlen(value) - 1;

	/*
	 * see if it is already there
	 */
	for(p = environ, i = 0; p != NULL && *p != NULL; ++p, ++i)
		if(strncmp(e, *p, len) == 0)
			break;
	if(p == NULL || *p == NULL) {
		/*
		 * not there, build a new environment array
		 */
		if(environ == my_environ) {
			tmp = realloc((void *)my_environ, (i+2)*sizeof(char *));
			if(tmp == NULL)
				return -1;
		}
		else {
			tmp = malloc((i+2)*sizeof(char *));
			if(tmp == NULL)
				return -1;
			memcpy((void *)tmp, (void *)environ, i*sizeof(char *));
		}
		my_environ   =
		environ      = tmp;
		environ[i+1] = NULL;
	}

	environ[i] = (char *)e;

	return 0;
}

#endif	/* HAS_PUTENV */

/*
 * Libhelp initialization.  You'll notice that we don't free "path" here.
 * Since libhelp doesn't make copies of its string resources, we have
 * no choice but to use a static fixed size buffer or let the memory
 * leak.  The "cls" and "indexfile" variables are static for the same
 * reasons.  I sent a patch to the libhelp author to fix this bug but I
 * never received a response (the patch also removed the dependence on
 * environment variables) so I'll just have to keep using this kludge;
 * if anyone wants the patch, I think I still have it around.
 */
static int
init(Widget w)
{
static	char	cls[128];
static	char	indexfile[512];
	char	name[128];
	char	*path;		/* this leaks, we have no choice	*/
	char	*libhelppath, *prefix, *s;
	int	size;

	/*
	 * We have to set the LIBHELPPATH in the environment since
	 * there is no other way to set the search path.  I could
	 * just assume that people will set LIBHELPPATH but overloading
	 * helpPrefix seems like a sensible thing to do.  I sent a
	 * patch to the libhelp people awhile ago that put this into
	 * the help_set_resource() API but I haven't heard from them.
	 */
	libhelppath = getenv("LIBHELPPATH");
	prefix      = wlp_s(w, WlpShelpPrefix);
	if((s = strchr(prefix, ':')) != NULL
	&& *(s + 1) == '/'
	&& *(s + 2) == '/')
		prefix = s + 3;
	size = sizeof("LIBHELPPATH=") + strlen(prefix);
	if(libhelppath == NULL) {
		path = calloc(1, size);
		s    = "LIBHELPPATH=%s";
	}
	else {
		path = calloc(1, size + strlen(libhelppath) + 1);
		s    = "LIBHELPPATH=%s:%s";
	}
	if(path == NULL)
		return FALSE;
	sprintf(path, s, prefix, libhelppath);
	putenv(path);

	memset((void *)&name[0], '\0', sizeof(name));
	memset((void *)&cls[0],  '\0', sizeof(cls));
	wl_appnames(w, &name[0], sizeof(name) - 1, &cls[0], sizeof(cls) - 1);
	sprintf(indexfile, "%s.html", name);

	help_set_resource(help_class_name, (XtPointer)cls);
	help_set_resource(help_index, (XtPointer)indexfile);
	help_set_resource(help_server, (XtPointer)XMHELP_SERVER);

	return TRUE;
}

/*
 * libhelp stuff
 */
Boolean
wlp_help_libhelp(Widget w, WLP_HELP *h)
{
static	int	initted = FALSE;
	char	buffer[1024];

	if(!initted
	&& !(initted = init(w)))
		return False;
	sprintf(buffer, "%s.html%s%s", h->chapter,
					h->section == NULL ? "" : "#",
					h->section == NULL ? "" : h->section);
	get_help(w, (XtPointer)&buffer[0], NULL);
	return True;
}

#else

/*
 * stub for lackers of libhelp
 */
Boolean
wlp_help_libhelp(Widget w, WLP_HELP *h)
{
static	Boolean	warned = False;
	if(!warned) {
		warned = True;
		wl_warn(wl_top(w), wlp_s(w, WlpSnotSupported),"libhelp","HTML");
	}
	return wlp_help_html(w, h);
}

#endif
