/*
 * $Id: sun.c,v 1.4 1995/09/16 14:01:57 traister Exp traister $
 * sun.c: reads a Sun rasterfile into an IMAGE struct
 * Copyright (c), 1995 Joseph J. Traister
 * 
 * $Log: sun.c,v $
 * Revision 1.4  1995/09/16  14:01:57  traister
 * Rewrote error reporting throughout.
 * Added file type table to facilitate adding new image types.
 *
 * Revision 1.3  1995/09/03  02:14:54  traister
 * Added RLE-encoded, color-mapped files
 *
 * Revision 1.2  1995/07/30  19:20:47  traister
 * Updated copyright prior to first public release
 *
 * Revision 1.1  1995/05/13  01:39:44  traister
 * Initial revision
 *
*/
/*   This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include "vimage.h"

#define RAS_MAGIC	0x59a66a95
enum { RAS_OLD, RAS_STANDARD, RAS_BYTE_ENCODED, RAS_EXPERIMENTAL };
enum { RMT_NONE, RMT_EQUAL_RGB, RMT_RAW }; /* this is a guess */

#define UCHAR		unsigned char

typedef struct rasterhead_tag {
  int ras_magic;
  int ras_width;
  int ras_height;
  int ras_depth;
  int ras_length;
  int ras_type;
  int ras_maptype;
  int ras_maplength;
} RASTERHEAD;

RASTERHEAD filehead;

unsigned ReadUInt(FILE *fpIn);

/*
  ReadRasterfile: reads a Sun rasterfile into an IMAGE struct
*/
int ReadRasterfileHead(FILE *fpIn, IMAGE *image)
{
  UCHAR redpal[256], grnpal[256], blupal[256];
  int i;
  
  filehead.ras_magic = ReadUInt(fpIn);
  filehead.ras_width = ReadUInt(fpIn);
  filehead.ras_height = ReadUInt(fpIn);
  filehead.ras_depth = ReadUInt(fpIn);
  filehead.ras_length = ReadUInt(fpIn);
  filehead.ras_type = ReadUInt(fpIn);
  filehead.ras_maptype = ReadUInt(fpIn);
  filehead.ras_maplength = ReadUInt(fpIn);
  if (filehead.ras_magic != RAS_MAGIC)
    return 0;
  if (filehead.ras_type == RAS_EXPERIMENTAL) {
    strcpy(errmsg, "Cannot read experimental Sun Rasterfiles");
    return -1;
  }
  if (filehead.ras_maptype == RMT_EQUAL_RGB && filehead.ras_maplength != 0) {
    fread(&redpal, 1, filehead.ras_maplength/3, fpIn);
    fread(&grnpal, 1, filehead.ras_maplength/3, fpIn);
    fread(&blupal, 1, filehead.ras_maplength/3, fpIn);
    for (i=0; i < filehead.ras_maplength/3; i++) {
      image->palette[i*3] = redpal[i];
      image->palette[i*3+1] = grnpal[i];
      image->palette[i*3+2] = blupal[i];
    }
  }
  image->height = filehead.ras_height;
  image->width = filehead.ras_width;
  switch (filehead.ras_depth) {
   case 1:
     image->palsize = -1;
     break;
   case 8:
     image->palsize = filehead.ras_maplength/3;
     break;
   case 24:
     image->palsize = -24;
     break;
   case 32:
     image->palsize = -32;
     break;
   }
  return 1;
}

int ReadRasterfile(FILE *fpIn, IMAGE *image)
{
  int i, j, k, l, m, rasterwidth, bitswidth;
  UCHAR *rasterline, *cp;
  
  switch (filehead.ras_depth) {
    case 1:
      bitswidth = (image->width>>3)+(image->width&0x07 ? 1 : 0);
      image->bits = (UCHAR*)malloc(bitswidth*image->height);
      rasterwidth = bitswidth;
      if (bitswidth%2)
        rasterwidth++;
      rasterline = (UCHAR*)malloc(rasterwidth);
      for (i=0; i < image->height; i++) {
        fread(rasterline, 1, rasterwidth, fpIn);
        memcpy(image->bits+i*bitswidth, rasterline, bitswidth);
      }
      free(rasterline);
      break;
    
    case 8:
      image->bits = (UCHAR*)malloc(image->height*image->width);
      rasterwidth = image->width+(image->width%2);
      switch(filehead.ras_type) {
      case RAS_STANDARD:
      case RAS_OLD:
	rasterline = (UCHAR*)malloc(rasterwidth);
	for (i=0; i < image->height; i++) {
	  fread(rasterline, 1, rasterwidth, fpIn);
	  memcpy(image->bits+i*image->width, rasterline, image->width);
	}
	free(rasterline);
	break;

      case RAS_BYTE_ENCODED:
	m=0;
	for (cp = image->bits; cp-image->bits < image->width*image->height; ) {
	  j = getc(fpIn);
	  if (j != 0x80) {
	    *cp++ = j;
	    m++;
	    if (!(m%rasterwidth) && rasterwidth != image->width)
	      cp--;
	  } else {
	    j = getc(fpIn);
	    if (!j) {
	      *cp++ = 0x80;
	      m++;
	      if (!(m%rasterwidth) && rasterwidth != image->width)
		cp--;
	    } else {
	      k = getc(fpIn);
	      for (l=0; l <= j; l++) {
		*cp++ = k;
		m++;
		if (!(m%rasterwidth) && rasterwidth != image->width)
		  cp--;
	      }
	    }
	  }
	}
	break;
      }
      break;

    case 24:
      bitswidth = image->width*3;
      image->bits = (UCHAR*)malloc(bitswidth*image->height);
      rasterwidth = bitswidth+(bitswidth%2);
      rasterline = (UCHAR*)malloc(rasterwidth);
      for (i=0; i < image->height; i++) {
        fread(rasterline, 1, rasterwidth, fpIn);
        for (j=0; j < image->width; j++) {
          image->bits[i*bitswidth+j*3] = rasterline[j*3+2];
          image->bits[i*bitswidth+j*3+1] = rasterline[j*3+1];
          image->bits[i*bitswidth+j*3+2] = rasterline[j*3];
        }
      }
      free(rasterline);
      break;
    
    case 32:
      strcpy(errmsg, "32-bit Sun rasterfiles not supported, yet!");
      return -1;
      break;
      
    default:
      strcpy(errmsg, "Unknown raster depth in Sun rasterfile");
      return -1;
  }
  return 0;
}

/*
  ReadInt: reads a big-endian integer from fpIn and returns the proper int
*/
unsigned ReadUInt(FILE *fpIn)
{
  unsigned result;
  
  result = (getc(fpIn)&0xff)<<24;
  result |= (getc(fpIn)&0xff)<<16;
  result |= (getc(fpIn)&0xff)<<8;
  result |= (getc(fpIn)&0xff);
  return result;
}
