/*
 * $Id: targa.c,v 1.4 1995/09/16 14:01:57 traister Exp traister $
 * targa.c: function for reading Targa image files
 * Only supports a subset of possible Targa images (codes 1, 2, 9 & 10)
 * Copyright (c) 1995, Joseph J. Traister
 *  
 * $Log: targa.c,v $
 * Revision 1.4  1995/09/16  14:01:57  traister
 * Rewrote error reporting throughout.
 * Added file type table to facilitate adding new image types.
 *
 * Revision 1.3  1995/09/03  02:14:54  traister
 * Added colormapped Targa files.
 *
 * Revision 1.2  1995/07/30  19:20:47  traister
 * Updated copyright prior to first public release
 *
 * Revision 1.1  1995/05/13  01:39:59  traister
 * Initial revision
 *
*/
/*   This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include "vimage.h"

enum { TGA_MAP=1, TGA_RGB=2, TGA_RLE_MAP=9, TGA_RLE_RGB=10 };
enum { NO_COLORMAP, COLORMAP };

#define ATTR_BITS  0x0F
#define RESERVED   0x10
#define ORIGIN     0x20
#define INTERLACE  0xC0

typedef struct tga_head_tag {
  UCHAR IDLength;
  UCHAR ColorMapType;
  UCHAR ImageCodeType;
  USHORT ColorMapOrigin;
  USHORT ColorMapLength;
  UCHAR ColorMapEntrySize;
  USHORT x, y, Width, Height;
  UCHAR PixelSize, Descriptor;
  char IDString[256];
} TGA_HEAD;

TGA_HEAD tgahead;

int ReadTargaHead(FILE *fpIn, IMAGE *image)
{
  int i, j;

  tgahead.IDLength = getc(fpIn);
  tgahead.ColorMapType = getc(fpIn);
  tgahead.ImageCodeType = getc(fpIn);
  tgahead.ColorMapOrigin = getc(fpIn);
  tgahead.ColorMapOrigin |= getc(fpIn)<<8;
  tgahead.ColorMapLength = getc(fpIn);
  tgahead.ColorMapLength |= getc(fpIn)<<8;
  tgahead.ColorMapEntrySize = getc(fpIn);
  tgahead.x = getc(fpIn);
  tgahead.x |= getc(fpIn)<<8;
  tgahead.y = getc(fpIn);
  tgahead.y |= getc(fpIn)<<8;
  tgahead.Width = getc(fpIn);
  tgahead.Width |= getc(fpIn)<<8;
  tgahead.Height = getc(fpIn);
  tgahead.Height |= getc(fpIn)<<8;
  tgahead.PixelSize = getc(fpIn);
  tgahead.Descriptor = getc(fpIn);
  if (tgahead.ColorMapType & 0xFE || tgahead.Descriptor & RESERVED)
    return 0;
  if (tgahead.IDLength)
    fread(tgahead.IDString, 1, tgahead.IDLength, fpIn);
  if (tgahead.ColorMapType) {
    if (tgahead.ColorMapLength > 256) {
	strcpy(errmsg, "Targa color map too large");
	return -1;
    }
    switch (tgahead.ColorMapEntrySize) {
    case 16:
      for (i=0; i < tgahead.ColorMapLength; i++) {
	j = getc(fpIn);
	j |= getc(fpIn)<<8;
	if (tgahead.ImageCodeType != TGA_RGB &&
	    tgahead.ImageCodeType != TGA_RLE_RGB) {
	  image->palette[i*3] = (j&0x7C00)>>7;
	  image->palette[i*3+1] = (j&0x03e0)>>2;
	  image->palette[i*3+2] = (j&0x1F)<<3;
	}
      }
      break;
      
    case 24:
    case 32:
      for (i=0; i < tgahead.ColorMapLength; i++) {
	if (tgahead.ImageCodeType != TGA_RGB &&
	    tgahead.ImageCodeType != TGA_RLE_RGB) {
	  image->palette[i*3+2] = getc(fpIn);
	  image->palette[i*3+1] = getc(fpIn);
	  image->palette[i*3] = getc(fpIn);
	  if (tgahead.ColorMapEntrySize == 32)
	    getc(fpIn);
	} else {
	  getc(fpIn);
	  getc(fpIn);
	  getc(fpIn);
	  if (tgahead.ColorMapEntrySize == 32)
	    getc(fpIn);
	}
      }
      break;

    default:
      return 0;
    }
  }
  image->width = tgahead.Width;
  image->height = tgahead.Height;
  if (tgahead.ImageCodeType == TGA_MAP ||
      tgahead.ImageCodeType == TGA_RLE_MAP)
    image->palsize = tgahead.ColorMapLength;
  else {
    if (tgahead.PixelSize == 16)
      image->palsize = -15;
    else if (tgahead.PixelSize == 32)
      image->palsize = -24;
    else
      image->palsize = -tgahead.PixelSize;
  }
  return 1;
}

int ReadTarga(FILE *fpIn, IMAGE *image)
{
  int i, j, k, l, c, r, g, b;
  UCHAR *tempbuf, *pch, *cp;
  
  switch (tgahead.ImageCodeType) {
  case TGA_RGB:
    switch (tgahead.PixelSize) {
    case 16:
      image->bits = (UCHAR*)malloc(image->width*image->height*2);
      fread(image->bits, image->width*image->height*2, 1, fpIn);
      if (!(tgahead.Descriptor&ORIGIN)) {
	tempbuf = (UCHAR*)malloc(image->width*image->height*2);
	for (i=0; i < image->height; i++)
	  memcpy(tempbuf+i*image->width*2, 
		 image->bits+(image->height-i-1)*image->width*2,
		 image->width*2);
	free(image->bits);
	image->bits = tempbuf;
      }
      break;

    case 24:
      image->bits = (UCHAR*)malloc(image->width*image->height*3);
      for (i=0; i < image->height*image->width; i++) {
	if ((c = getc(fpIn)) != EOF)
	  image->bits[i*3+2] = c;
	else {
	  free(image->bits);
	  strcpy(errmsg, "Unexpected EOF reading Targa file");
	  return -1;
	}
	if ((c = getc(fpIn)) != EOF)
	  image->bits[i*3+1] = c;
	else {
	  free(image->bits);
	  strcpy(errmsg, "Unexpected EOF reading Targa file");
	  return -1;
	}
	if ((c = getc(fpIn)) != EOF)
	  image->bits[i*3] = c;
	else {
	  free(image->bits);
	  strcpy(errmsg, "Unexpected EOF reading Targa file");
	  return -1;
	}
      }
      if (!(tgahead.Descriptor&ORIGIN)) {
	tempbuf = (UCHAR*)malloc(image->width*image->height*3);
	for (i=0; i < image->height; i++)
	  memcpy(tempbuf+i*image->width*3, 
		 image->bits+(image->height-i-1)*image->width*3,
		 image->width*3);
	free(image->bits);
	image->bits = tempbuf;
      }
      break;

    default:
      fclose(fpIn);
    }
    break;
  
  case TGA_RLE_RGB:
    switch (tgahead.PixelSize) {
    case 24:
      image->bits = (UCHAR*)malloc(image->width*image->height*3);
      pch = image->bits;
      i=0;
      while (i < image->height*image->width) {
	c = getc(fpIn);
	if (c&0x80) {
	  b = getc(fpIn);
	  g = getc(fpIn);
	  r = getc(fpIn);
	  c &= 0x7F;
	  c++;
	  while (c--) {
	    *pch++ = r;
	    *pch++ = g;
	    *pch++ = b;
	    i++;
	  }
	} else {
	  c++;
	  while (c--) {
	    b = getc(fpIn);
	    g = getc(fpIn);
	    r = getc(fpIn);
	    *pch++ = r;
	    *pch++ = g;
	    *pch++ = b;
	    i++;
	  }
	}
      }
      if (!(tgahead.Descriptor&ORIGIN)) {
	tempbuf = (UCHAR*)malloc(image->width*image->height*3);
	for (i=0; i < image->height; i++)
	  memcpy(tempbuf+i*image->width*3, 
		 image->bits+(image->height-i-1)*image->width*3,
		 image->width*3);
	free(image->bits);
	image->bits = tempbuf;
      }
      break;

    case 16:
      image->bits = (UCHAR*)malloc(image->width*image->height*2);
      pch = image->bits;
      i=0;
      while (i < image->height*image->width) {
	c = getc(fpIn);
	if (c&0x80) {
	  j = getc(fpIn);
	  k = getc(fpIn);
	  c &= 0x7F;
	  c++;
	  while (c--) {
	    *pch++ = j;
	    *pch++ = k;
	    i++;
	  }
	} else {
	  c++;
	  while (c--) {
	    j = getc(fpIn);
	    k = getc(fpIn);
	    *pch++ = j;
	    *pch++ = k;
	    i++;
	  }
	}
      }
      if (!(tgahead.Descriptor&ORIGIN)) {
	tempbuf = (UCHAR*)malloc(image->width*image->height*2);
	for (i=0; i < image->height; i++)
	  memcpy(tempbuf+i*image->width*2, 
		 image->bits+(image->height-i-1)*image->width*2,
		 image->width*2);
	free(image->bits);
	image->bits = tempbuf;
      }
      break;

    default:
      strcpy(errmsg, "Unknown pixel size in Targa file");
      return -1;
    }
    break;

  case TGA_MAP:
    image->bits = (UCHAR*)malloc(image->width*image->height);
    for (cp = image->bits; 
	 cp-image->bits < image->width*image->height; cp++) {
      for (i=0, j=0; i < tgahead.PixelSize/8; i++)
	j |= getc(fpIn)<<(i*8);
      *cp = j - tgahead.ColorMapOrigin;
    }
    if (!(tgahead.Descriptor&ORIGIN)) {
      tempbuf = (UCHAR*)malloc(image->width*image->height);
      for (i=0; i < image->height; i++)
	memcpy(tempbuf+i*image->width, 
	       image->bits+(image->height-i-1)*image->width,
	       image->width);
      free(image->bits);
      image->bits = tempbuf;
    }
    break;

  case TGA_RLE_MAP:
    image->bits = (UCHAR*)malloc(image->width*image->height);
    for (cp=image->bits; cp-image->bits < image->width*image->height; ) {
      i = getc(fpIn);
      if (i&0x80) {
	for (j=0, k=0; j < tgahead.PixelSize/8; j++)
	  k |= getc(fpIn)<<(j*8);
	memset(cp, k-tgahead.ColorMapOrigin, (i&0x7f)+1);
	cp += (i&0x7f)+1;
      } else {
	for (j=0; j < (i&0x7f)+1; j++) {
	  for (k=0, l=0; k < tgahead.PixelSize/8; k++)
	    l |= getc(fpIn)<<(k*8);
	  *cp++ = l - tgahead.ColorMapOrigin;
	}
      }
    }
    if (!(tgahead.Descriptor&ORIGIN)) {
      tempbuf = (UCHAR*)malloc(image->width*image->height);
      for (i=0; i < image->height; i++)
	memcpy(tempbuf+i*image->width, 
	       image->bits+(image->height-i-1)*image->width,
	       image->width);
      free(image->bits);
      image->bits = tempbuf;
    }
    break;

  default:
    strcpy(errmsg, "Unknown Targa file type");
    return -1;
  }
  return 0;
}
