/*
 * $Id: tiff.c,v 1.2 1995/09/16 14:01:57 traister Exp traister $
 * tiff.c: function ReadTIFF reads a TIFF file into an IMAGE struct
 * Copyright (c) 1995, Joseph J. Traister
 * 
 * $Log: tiff.c,v $
 * Revision 1.2  1995/09/16  14:01:57  traister
 * Rewrote error reporting throughout.
 * Added file type table to facilitate adding new image types.
 *
 * Revision 1.1  1995/09/03  02:14:16  traister
 * Initial revision
 *
 *
*/
/*   This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <stdio.h>
#include <stdlib.h>
#include "vimage.h"
#include "tiff.h"

#define USHORT unsigned short
#define UINT unsigned int

UCHAR tempc;
#define swapshort(a) { \
			 tempc = ((UCHAR*)&(a))[0]; \
			 ((UCHAR*)&(a))[0] = ((UCHAR*)&(a))[1]; \
			 ((UCHAR*)&(a))[1] = tempc; \
		     }

#define swaplong(a) { \
			 tempc = ((UCHAR*)&(a))[0]; \
			 ((UCHAR*)&(a))[0] = ((UCHAR*)&(a))[3]; \
			 ((UCHAR*)&(a))[3] = tempc; \
			 tempc = ((UCHAR*)&(a))[1]; \
			 ((UCHAR*)&(a))[1] = ((UCHAR*)&(a))[2]; \
			 ((UCHAR*)&(a))[2] = tempc; \
		     }

enum byteorder { LITTLEENDIAN=0x4949, BIGENDIAN=0x4D4D };
enum datatypes { BYTE=1, ASCII, SHORT, LONG, RATIONAL };

typedef struct tiffhead_tag {
  USHORT byteorder, version;
  UINT IFDoffset;
} TIFFHEAD;

typedef struct IFD_tag {
  USHORT tag, type;
  UINT count, offset;
} IFD;

static UINT photointerp, byteorder, comptype;
static UINT rowsperstrip, stripoffsets, planarconfig, offsettype;
static UINT stripbytecounts, bytecounttype, predictor;
UCHAR *lzwbuf;

int DecompLZW(UCHAR *out, int isize);

int ReadTIFFHead(FILE *fpin, IMAGE *image)
{
  int i, j, k, IFDcount;
  IFD *ifds=NULL;
  TIFFHEAD tiffhead;
  UINT bitspersample=1, colormap=0, samplesperpixel=1;

  stripbytecounts = stripoffsets = 0;
  photointerp = predictor = comptype = planarconfig = 1;
  rowsperstrip = 0xffffffff;
  image->width = image->height = -1;
  fread(&tiffhead, sizeof(TIFFHEAD), 1, fpin);
  if (tiffhead.byteorder == BIGENDIAN) {
    swapshort(tiffhead.version);
    swaplong(tiffhead.IFDoffset);
  } else if (tiffhead.byteorder != LITTLEENDIAN)
    return 0;
  if (tiffhead.version != 42)
    return 0;
  byteorder = tiffhead.byteorder;
  fseek(fpin, tiffhead.IFDoffset, SEEK_SET);
  IFDcount = getc(fpin);
  IFDcount |= getc(fpin)<<8;
  if (byteorder == BIGENDIAN)
    swapshort(IFDcount);
  if (IFDcount > 0) {
    if (!(ifds = (IFD*)malloc(sizeof(IFD)*IFDcount))) {
      strcpy(errmsg, "Out of memory reading TIFF header");
      return -1;
    }
  } else {
    strcpy(errmsg, "No image found in TIFF");
    return -1;
  }
  fread(ifds, sizeof(IFD), IFDcount, fpin);
  if (byteorder == BIGENDIAN)
    for (i=0; i < IFDcount; i++) {
      swapshort(ifds[i].tag);
      swapshort(ifds[i].type);
      swaplong(ifds[i].count);
      if (ifds[i].type == SHORT && ifds[i].count == 1) {
	swapshort(ifds[i].offset);
      } else {
	swaplong(ifds[i].offset);
      }
    }
  for (i=0; i < IFDcount; i++)
    switch (ifds[i].tag) {
    case ImageWidth:
      image->width = ifds[i].offset;
      break;

    case ImageLength:
      image->height = ifds[i].offset;
      break;

    case Compression:
      comptype = ifds[i].offset;
      break;

    case Predictor:
      predictor = ifds[i].offset;
      break;

    case PhotometricInterpretation:
      photointerp = ifds[i].offset;
      break;

    case StripOffsets:
      stripoffsets = ifds[i].offset;
      offsettype = ifds[i].type;
      break;

    case StripByteCounts:
      stripbytecounts = ifds[i].offset;
      bytecounttype = ifds[i].type;
      break;

    case PlanarConfiguration:
      planarconfig = ifds[i].offset;
      break;

    case RowsPerStrip:
      rowsperstrip = ifds[i].offset;
      break;

    case ColorMap:
      colormap = ifds[i].offset;
      break;

    case BitsPerSample:
      bitspersample = ifds[i].offset;
      break;

    case SamplesPerPixel:
      samplesperpixel = ifds[i].offset;
      break;
    }
  free(ifds);
  if (comptype != 1 && comptype != 5) {
    strcpy(errmsg, "Unsupported compression type in TIFF");
    return -1;
  }
  if (comptype == 5 && (predictor < 1 || predictor > 2)) {
    strcpy(errmsg, "Unsupported predictor in TIFF");
    return -1;
  }
  if (image->height < 0 || image->width < 0) {
    strcpy(errmsg, "Height or width (or both) missing in TIFF");
    return -1;
  }
  if (!(stripoffsets && stripbytecounts)) {
    strcpy(errmsg, "Strip information missing in TIFF");
    return -1;
  }
  if (planarconfig != 1 && planarconfig != 2) {
    strcpy(errmsg, "Unsupported planar configuration in TIFF");
    return -1;
  }
  switch (photointerp) {
  case 0:
  case 1:
    if (bitspersample != 1 && bitspersample != 8) {
      strcpy(errmsg, "Unsupported bits per pixel in TIFF");
      return -1;
    }
    image->palsize = -bitspersample;
    break;

  case 2:
    if (samplesperpixel == 3) {
      fseek(fpin, bitspersample, SEEK_SET);
      for (i=0; i < 3; i++) {
	j = getc(fpin);
	j |= getc(fpin)<<8;
	if (byteorder == BIGENDIAN)
	  swapshort(j);
	if (j != 8) {
	  strcpy(errmsg, "Unsupported bits per sample in TIFF");
	  return -1;
	}
      }
      image->palsize = -24;
    } else {
      strcpy(errmsg, "Unsupported samples per pixel in TIFF");
      return -1;
    }
    break;

  case 3:
    if (bitspersample != 8) {
      strcpy(errmsg, "Unsupported bits per sample in TIFF");
      return -1;
    }
    image->palsize = 256;
    fseek(fpin, colormap, SEEK_SET);
    for (k=0; k < image->palsize; k++) {
      if (byteorder == LITTLEENDIAN)
	getc(fpin);
      image->palette[k*3] = getc(fpin);
      if (byteorder == BIGENDIAN)
	getc(fpin);
    }
    for (k=0; k < image->palsize; k++) {
      if (byteorder == LITTLEENDIAN)
	getc(fpin);
      image->palette[k*3+1] = getc(fpin);
      if (byteorder == BIGENDIAN)
	getc(fpin);
    }
    for (k=0; k < image->palsize; k++) {
      if (byteorder == LITTLEENDIAN)
	getc(fpin);
      image->palette[k*3+2] = getc(fpin);
      if (byteorder == BIGENDIAN)
	getc(fpin);
    }
    break;

  default:
    strcpy(errmsg, "Unsupported photo interpretation in TIFF");
    return -1;
  }
  if (comptype == 5 && predictor == 2 && image->palsize != -24 &&
      image->palsize != -8) {
    strcpy(errmsg, "Unsupported predictor in TIFF");
    return -1;
  }
  if (rowsperstrip > image->height)
    rowsperstrip = image->height;
  return 1;
}

int ReadTIFF(FILE *fpin, IMAGE *image)
{
  UINT *offsets, *bytecounts, imagesize;
  UCHAR *pos, *tempbits;
  int cstrips=1, i, j, r, g, b;

  cstrips = (image->height+rowsperstrip-1)/rowsperstrip;
  if (planarconfig == 2)
    cstrips *= 3;
  if (!(offsets = (UINT*)malloc(cstrips*sizeof(UINT)))) {
    strcpy(errmsg, strerror(errno));
    return -1;
  }
  if (cstrips > 1) {
    fseek(fpin, stripoffsets, SEEK_SET);
    if (offsettype == LONG) {
      fread(offsets, sizeof(UINT), cstrips, fpin);
      if (byteorder == BIGENDIAN)
	for (i=0; i < cstrips; i++)
	  swaplong(offsets[i]);
    } else {
      for (i=0; i < cstrips; i++)
	if (byteorder == LITTLEENDIAN) {
	  offsets[i] = getc(fpin);
	  offsets[i] |= getc(fpin)<<8;
	} else {
	  offsets[i] = getc(fpin)<<8;
	  offsets[i] |= getc(fpin);
	}
    }
  } else
    offsets[0] = stripoffsets;
  if (!(bytecounts = (UINT*)malloc(cstrips*sizeof(UINT)))) {
    strcpy(errmsg, strerror(errno));
    return -1;
  }
  if (cstrips > 1) {
    fseek(fpin, stripbytecounts, SEEK_SET);
    if (bytecounttype == LONG) {
      fread(bytecounts, sizeof(UINT), cstrips, fpin);
      if (byteorder == BIGENDIAN)
	for (i=0; i < cstrips; i++)
	  swaplong(bytecounts[i]);
    } else {
      for (i=0; i < cstrips; i++)
	if (byteorder == LITTLEENDIAN) {
	  bytecounts[i] = getc(fpin);
	  bytecounts[i] |= getc(fpin)<<8;
	} else {
	  bytecounts[i] = getc(fpin)<<8;
	  bytecounts[i] |= getc(fpin);
	}
    }
  } else
    bytecounts[0] = stripbytecounts;
  if (image->palsize == -1)
    imagesize = ((image->width+7)/8)*image->height;
  else
    imagesize = image->width*image->height;
  if (image->palsize == -24)
    imagesize *= 3;
  if (!(image->bits = (UCHAR*)malloc(imagesize))) {
    strcpy(errmsg, strerror(errno));
    return -1;
  }
  if (cstrips == 1) {
    if (comptype == 5) {
      lzwbuf = (UCHAR*)malloc(bytecounts[0]);
      fseek(fpin, offsets[0], SEEK_SET);
      fread(lzwbuf, sizeof(char), bytecounts[0], fpin);
      if (DecompLZW(image->bits, bytecounts[0]) < 0) {
	strcpy(errmsg, "Error decompressing TIFF");
	free(image->bits);
	free(lzwbuf);
	free(offsets);
	free(bytecounts);
	return -1;
      }
      free(lzwbuf);
    } else {
      fseek(fpin, offsets[0], SEEK_SET);
      fread(image->bits, sizeof(char), bytecounts[0], fpin);
    }
  } else {
    if (comptype == 5) {
      pos = image->bits;
      for (i=0; i < cstrips; i++) {
	lzwbuf = (UCHAR*)malloc(bytecounts[i]);
	fseek(fpin, offsets[i], SEEK_SET);
	fread(lzwbuf, sizeof(char), bytecounts[i], fpin);
	if ((j=DecompLZW(pos, bytecounts[i])) < 0) {
	  strcpy(errmsg, "Error decompressing TIFF");
	  free(image->bits);
	  free(lzwbuf);
	  free(offsets);
	  free(bytecounts);
	  return -1;
	}
	pos += j;
	free(lzwbuf);
      }
    } else {
      pos = image->bits;
      for (i=0; i < cstrips; i++) {
	fseek(fpin, offsets[i], SEEK_SET);
	fread(pos, sizeof(char), bytecounts[i], fpin);
	pos += bytecounts[i];
      }
    }
  }
  free(offsets);
  free(bytecounts);
  if (comptype == 5 && predictor == 2) {
    if (image->palsize == -8)
      for (i=0; i < image->height; i++) {
	g = image->bits[i*image->width];
	for (j=1; j < image->width; j++) {
	  g += image->bits[i*image->width+j];
	  image->bits[i*image->width+j] = (char)g;
	}
      }
    else
      for (i=0; i < image->height; i++) {
	r = image->bits[i*image->width*3];
	g = image->bits[i*image->width*3+1];
	b = image->bits[i*image->width*3+2];
	for (j=1; j < image->width; j++) {
	  r += image->bits[(i*image->width+j)*3];
	  image->bits[(i*image->width+j)*3] = (char)r;
	  g += image->bits[(i*image->width+j)*3+1];
	  image->bits[(i*image->width+j)*3+1] = (char)g;
	  b += image->bits[(i*image->width+j)*3+2];
	  image->bits[(i*image->width+j)*3+2] = (char)b;
	}
      }
  }
  if (photointerp && image->palsize == -1)
    for (i=0; i < imagesize; i++)
      image->bits[i] ^= 0xff;
  if (!photointerp && image->palsize == -8)
    for (i=0; i < imagesize; i++)
      image->bits[i] = 255-image->bits[i];
  if (planarconfig == 2 && image->palsize == -24) {
    tempbits = (UCHAR*)malloc(image->height*image->width*3);
    for (i=0; i < image->height*image->width; i++) {
      tempbits[i*3] = image->bits[i];
      tempbits[i*3+1] = image->bits[i+image->width*image->height];
      tempbits[i*3+2] = image->bits[i+2*image->width*image->height];
    }
    free(image->bits);
    image->bits = tempbits;
  }
  return 0;
}

#define CLEARCODE 256
#define EOICODE 257
#define MAXCODE 4095

short prefix[MAXCODE+1];
UCHAR suffix[MAXCODE+1];
UCHAR stack[MAXCODE+1];
int bitsleft, maxbytes, curbyte;

UCHAR TIFFDecomposeCode(int code, UCHAR **out);
int TIFFGetCode(int bits);

int DecompLZW(UCHAR *out, int isize)
{
  int curbits, nextcode, new, old;
  UCHAR *nexto;

  nextcode = CLEARCODE+2;
  old = CLEARCODE;
  curbyte = 0;
  bitsleft = 8;
  maxbytes = isize;
  nexto = out;
  curbits = 9;
  while (curbyte < isize) {
    new = TIFFGetCode(curbits);
    if (new > MAXCODE)
      return -1;
    if (old == CLEARCODE && new != CLEARCODE) {
      *nexto++ = new;
      old = new;
      continue;
    }
    if (new == CLEARCODE) {
      nextcode = CLEARCODE+2;
      curbits = 9;
      old = TIFFGetCode(curbits);
      if (old > MAXCODE)
	return -1;
      *nexto++ = old;
      continue;
    }
    if (new == EOICODE)
      break;
    if (new < CLEARCODE) {
      *nexto++ = new;
      prefix[nextcode] = old;
      suffix[nextcode++] = new;
    } else if (new < nextcode) {
      prefix[nextcode] = old;
      suffix[nextcode++] = TIFFDecomposeCode(new, &nexto);
    } else if (new == nextcode && old < CLEARCODE) {
      *nexto++ = old;
      *nexto++ = old;
      prefix[nextcode] = old;
      suffix[nextcode++] = old;
    } else if (new == nextcode) { /* old >= clearcode */
      prefix[nextcode] = old;
      suffix[nextcode] = TIFFDecomposeCode(old, &nexto);
      *nexto++ = suffix[nextcode++];
    } else {
      strcpy(errmsg, "Bad code in TIFF");
      return -1;
    }
    if (nextcode == (1<<curbits)-1 && curbits < 12)
      curbits++;
    old = new;
  }
  return nexto-out;
}

int TIFFGetCode(int bits)
{
  int bitstogo = bits, result = 0, goodbits;
  UCHAR mask[] = { 0x00, 0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f, 0xFF };

  if (bits < 9 || bits > 12) {
    strcpy(errmsg, "Bad TIFF code width");
    return MAXCODE+1;
  }
  while (bitstogo) {
    if (!bitsleft) {
      bitsleft = 8;
      curbyte++;
      if (curbyte >= maxbytes)
	return 0;
    }
    if (bitsleft <= bitstogo) {
      goodbits = lzwbuf[curbyte] & mask[bitsleft];
      result <<= bitsleft;
      result |= goodbits;
      bitstogo -= bitsleft;
      bitsleft = 0;
    } else {
      goodbits = (lzwbuf[curbyte]&mask[bitsleft])&(~mask[bitsleft-bitstogo]);
      result <<= bitstogo;
      result |= goodbits >> (8-bitstogo);
      bitsleft -= bitstogo;
      bitstogo = 0;
    }
  }
  return result;    
}

UCHAR TIFFDecomposeCode(int code, UCHAR **out)
{
  int sp = 0;
  UCHAR startchar;

  while (code > CLEARCODE) {
    stack[sp++] = suffix[code];
    code = prefix[code];
  }
  startchar = code;
  stack[sp++] = code;
  while (sp) {
    **out = stack[--sp];
    (*out)++;
  }
  return startchar;
}
