/*
 * $Id: video.c,v 1.5 1995/09/16 14:01:57 traister Exp traister $
 * video.c - graphics routines for vimage
 * Copyright (c) 1995, Joseph J. Traister
 * 
 * $Log: video.c,v $
 * Revision 1.5  1995/09/16  14:01:57  traister
 * Rewrote error reporting throughout.
 * Added file type table to facilitate adding new image types.
 *
 * Revision 1.4  1995/09/03  02:14:54  traister
 * Cleaned up pallete handling.
 *
 * Revision 1.3  1995/07/30  19:20:47  traister
 * Updated copyright prior to first public release
 *
 * Revision 1.2  1995/05/13  22:40:17  traister
 * Sped up display by using drawscansegment. Removed use of gl
 * framebuffer library. Added "break" in slide show when ESC is hit.
 *
 * Revision 1.1  1995/05/13  01:40:17  traister
 * Initial revision
 *
 */

/*   This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <errno.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <malloc.h>
#include <vga.h>
#include "vimage.h"

int modetable[5][4] = 
{ 
  {   G320x200x256,   G320x200x32K,   G320x200x64K,   G320x200x16M },
  {   G640x480x256,   G640x480x32K,   G640x480x64K,   G640x480x16M },
  {   G800x600x256,   G800x600x32K,   G800x600x64K,   G800x600x16M },
  {  G1024x768x256,  G1024x768x32K,  G1024x768x64K,  G1024x768x16M },
  { G1280x1024x256, G1280x1024x32K, G1280x1024x64K, G1280x1024x16M }
};

vga_modeinfo *modeinfo=NULL;
int vgamode=-1;
uid_t euid, uid;

/*
  InitVGA: sets VGA adaptor into mode set in GSVGAMODE variable
*/
int InitVGA(char *modename)
{
  int i, j;

  seteuid(euid);
  if (!automode) {
    if (!strlen(modename)) {
      vgamode = vga_getdefaultmode();
      if (vgamode == -1)
	vgamode = G320x200x256;
    } else {
      vgamode = vga_getmodenumber(modename);
      if (vgamode < 0) {
	seteuid(uid);
	sprintf(errmsg, "Unknown video mode name: %s", modename);
	return -1;
      }
    }
    if (!vga_hasmode(vgamode)) {
      seteuid(uid);
      sprintf(errmsg, "Video mode not valid: %s", vga_getmodename(vgamode));
      return -1;
    }
  } else {
    for (i=0; i < 5; i++)
      for (j=0; j < 4; j++)
	if (!vga_hasmode(modetable[i][j]))
	  modetable[i][j] = -1;
  }
  seteuid(uid);
  return 0;
}

void SetVideoMode(void)
{
  seteuid(euid);
  if (vga_getcurrentmode() != vgamode) {
    vga_lockvc();
    vga_setmode(vgamode);
    vga_unlockvc();
  } else
    vga_clear();
  seteuid(uid);
}

void SetModeInfo(IMAGE *image)
{
  int x=0, y=0, i;

  if (automode) {
    if (image->palsize == -24) {
      if (notruecolor)
	x = 2;
      else
	x = 3;
    }
    if (image->palsize == -15)
      x = 1;
    if (image->width > 320 || image->height > 200)
      y = 1;
    if (image->width > 640 || image->height > 480)
      y = 2;
    if (image->width > 800 || image->height > 600)
      y = 3;
    if (image->width > 1024 || image->height > 768)
      y = 4;
    i = x;
    while (modetable[y][i] < 0) {
      i--;
      if (i < 0) {
	i = x;
	y--;
      }
    }
    vgamode = modetable[y][i];
  }
  modeinfo = vga_getmodeinfo(vgamode);
}

/*
   DisplayImage: displays image
*/
int DisplayImage(IMAGE *image)
{
  int startline, startcol, x=0, y=0, palsize=0, i=0;
  unsigned r, g, b;
  int palette[768];
  UCHAR *outline;

  seteuid(euid);
  startline = (modeinfo->height-image->height)/2;
  if (startline < 0)
    startline = 0;
  startcol = (modeinfo->width-image->width)/2;
  if (startcol < 0)
    startcol = 0;
  if (modeinfo->colors > 256) {
    if (!(outline = (UCHAR*)malloc(image->width*modeinfo->bytesperpixel))) {
      seteuid(uid);
      strcpy(errmsg, strerror(errno));
      return -1;
    }
    for (y=0; y<image->height && y<modeinfo->height; y++) {
      for (x=0; x<image->width && x<modeinfo->width; x++) {
	switch (image->palsize) {
	case -24:
	  i = y*image->width*3+x*3;
	  r = image->bits[i];
	  g = image->bits[i+1];
	  b = image->bits[i+2];
	  break;
	  
	case -15:
	  i = y*image->width*2+x*2;
	  i = image->bits[i]|(image->bits[i+1]<<8);
	  r = (i&0x7C00)>>7;
	  g = (i&0x03e0)>>2;
	  b = (i&0x001F)<<3;
	  break;
	 
	case -8:
	  r = g = b = image->bits[y*image->width+x];
	  break;
	  
	case -1:
	  i = (image->width>>3)+(image->width&0x07 ? 1 : 0);
	  if (image->bits[(y*i+(x>>3))]&1<<(7-(x&0x07)))
	    r = g = b = 0;
	  else
	    r = g = b = 255;
	  break;
	  
	default:
	  i = y*image->width+x;
	  r = image->palette[image->bits[i]*3];
	  g = image->palette[image->bits[i]*3+1];
	  b = image->palette[image->bits[i]*3+2];
	}
	switch (modeinfo->colors) {
	case 16777216:
	  outline[x*3] = b;
	  outline[x*3+1] = g;
	  outline[x*3+2] = r;
	  break;

	case 65536:
	  i = ((r<<8)&0xf800)|((g<<3)&0x07e0)|(b>>3);
	  outline[x*2] = i&0xff;
	  outline[x*2+1] = i>>8;
	  break;

	case 32768:
	  i = ((r<<7)&0x7c00)|((g<<2)&0x03e0)|(b>>3);
	  outline[x*2] = i&0xff;
	  outline[x*2+1] = i>>8;
	  break;
	}
      }
      vga_drawscansegment(outline, startcol, y+startline, 
			  image->width*modeinfo->bytesperpixel);
    }
    free(outline);
  } else {
    switch (image->palsize) {
    case -8:
      for (i=0; i < 256; i++)
	palette[i*3] = palette[i*3+1] = palette[i*3+2] = (i>>2);
      palsize = 256;
      break;

    case -1:
      palette[0] = palette[1] = palette[2] = 0;
      palette[3] = palette[4] = palette[5] = 63;
      palsize = 2;
      break;
      
    default:
      for (i=0; i < image->palsize*3; i++)
	palette[i] = image->palette[i]>>2;
      palsize = image->palsize;
    }
    vga_setpalvec(0, palsize, palette);
    if (image->palsize == -1) {
      for (y=startline; y<image->height+startline && y<modeinfo->height; y++)
	for (x=startcol; x<image->width+startcol && x<modeinfo->width; x++) {
	  i = (image->width>>3)+(image->width&0x07 ? 1 : 0);
	  if (image->bits[(y-startline)*i+((x-startcol)>>3)]&
	      1<<(7-((x-startcol)&0x07)))
	    vga_setcolor(0);
	  else
	    vga_setcolor(1);
	  vga_drawpixel(x, y);
	}
    } else {
      for (y=startline; y<image->height+startline && y<modeinfo->height; y++)
	vga_drawscansegment(image->bits+(y-startline)*image->width,
			    startcol, y, image->width);
    }
  }
  seteuid(uid);
  return 0;
}

void VGAShutdown(void)
{
  seteuid(euid);
  if (vga_getcurrentmode() != TEXT)
    vga_setmode(TEXT);
  seteuid(uid);
}

void SetID(void)
{
  euid = geteuid();
  uid = getuid();
  seteuid(uid);
}
