/* function.c */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef DISP_CURSES
#include <curses.h>
#endif

#include "calc.h"
#include "error.h"

#ifndef PI
#define PI 3.14159265
#endif

int func_add(void);
int func_sub(void);
int func_mult(void);
int func_div(void);
int func_pow(void);
int func_log(void);
int func_ln(void);
int func_sqr(void);
int func_sqrt(void);
int func_inv(void);
int func_neg(void);
int func_exp(void);
int func_fact(void);
int func_sin(void);
int func_cos(void);
int func_tan(void);
int func_asin(void);
int func_acos(void);
int func_atan(void);

int func_sl(void);
int func_sr(void);
int func_and(void);
int func_or(void);
int func_xor(void);

int func_deg(void);
int func_rad(void);
int func_num(void);
int func_fix(void);
int func_todeg(void);
int func_torad(void);
int func_ipart(void);
int func_fpart(void);
int func_flt(void);
int func_tohex(void);
int func_todec(void);
int func_swap(void);
int func_drop(void);
int func_rot(void);
int func_roll(void);
int func_clr(void);
int func_dropn(void);
int func_dup(void);
int func_sto(void);
int func_rcl(void);
int func_del(void);
int func_stack(void);
int func_save(void);
int func_load(void);
int func_loadregs(void);
int func_execsc(void);
int func_setsc(void);
int func_delsc(void);
int func_help(void);
int func_quit(void);

struct functable funclist[] = {
	{ "+", func_add, 2, TTYP_NUM },
	{ "-", func_sub, 2, TTYP_NUM },
	{ "*", func_mult, 2, TTYP_NUM },
	{ "/", func_div, 2, TTYP_NUM },
	{ "^", func_pow, 2, TTYP_NUM },
	{ "log", func_log, 1, TTYP_NUM },
	{ "ln", func_ln, 1, TTYP_NUM },
	{ "sqr", func_sqr, 1, TTYP_NUM },
	{ "sqrt", func_sqrt, 1, TTYP_NUM },
	{ "inv", func_inv, 1, TTYP_NUM },
	{ "neg", func_neg, 1, TTYP_NUM },
	{ "exp", func_exp, 1, TTYP_NUM },
	{ "fact", func_fact, 1, TTYP_NUM },
	{ "sin", func_sin, 1, TTYP_NUM },
	{ "cos", func_cos, 1, TTYP_NUM },
	{ "tan", func_tan, 1, TTYP_NUM },
	{ "asin", func_asin, 1, TTYP_NUM },
	{ "acos", func_acos, 1, TTYP_NUM },
	{ "atan", func_atan, 1, TTYP_NUM },
	{ "sl", func_sl, 2, TTYP_NUM },
	{ "sr", func_sr, 2, TTYP_NUM },
	{ "and", func_and, 2, TTYP_INT },
	{ "or", func_or, 2, TTYP_INT },
	{ "xor", func_xor, 2, TTYP_INT },
	{ "deg", func_deg, 0, 0 },
	{ "rad", func_rad, 0, 0 },
	{ "fix", func_fix, 1, TTYP_NUM },
	{ "num", func_num, 0, 0 },
	{ "todeg", func_todeg, 1, TTYP_NUM },
	{ "torad", func_torad, 1, TTYP_NUM },
	{ "ipart", func_ipart, 1, TTYP_NUM },
	{ "fpart", func_fpart, 1, TTYP_NUM },
	{ "flt", func_flt, 1, TTYP_NUM },
	{ "hex", func_tohex, 1, TTYP_NUM },
	{ "dec", func_todec, 1, TTYP_NUM },
	{ "tohex", func_tohex, 1, TTYP_NUM },
	{ "todec", func_todec, 1, TTYP_NUM },
	{ "swap", func_swap, 2, TTYP_ANY },
	{ "drop", func_drop, 1, TTYP_ANY },
	{ "rot", func_rot, 3, TTYP_ANY },
	{ "roll", func_roll, 0, 0 },
	{ "clr", func_clr, 0, 0 },
	{ "dropn", func_dropn, 1, TTYP_NUM },
	{ "dup", func_dup, 1, TTYP_ANY },
	{ "sto", func_sto, 2, TTYP_ANY },
	{ "rcl", func_rcl, 1, TTYP_NUM },
	{ "del", func_del, 1, TTYP_NUM },
	{ "stack", func_stack, 1, TTYP_NUM },
	{ "save", func_save, 1, TTYP_STR },
	{ "load", func_load, 1, TTYP_STR },
	{ "loadregs", func_loadregs, 1, TTYP_STR },
	{ "sc", func_execsc, 1, TTYP_NUM },
	{ "setsc", func_setsc, 2, TTYP_ANY },
	{ "delsc", func_delsc, 1, TTYP_NUM },
	{ "help", func_help, 0, 0 },
	{ "quit", func_quit, 0, 0 },
	{ NULL, NULL, 0 } };

ac_int_t getnumi(token *t)
{
	if(!(t->typ & TTYP_NUM)) {
		return 0;
	}

	if(t->typ & TTYP_FLT) {
		return (ac_int_t)(t->dat.dnum);
	}

	return t->dat.lnum;
}

ac_flt_t getnumf(token *t)
{
	if(!(t->typ & TTYP_NUM)) {
		return 0;
	}

	if(t->typ & TTYP_FLT) {
		return t->dat.dnum;
	}

	return (ac_flt_t)t->dat.lnum;
}

int func_add(void)
{
	token *arg1, *arg2;
	token res;
	
	arg1 = ac_pop_p();
	arg2 = ac_pop_p();

	if (arg1->typ == TTYP_FLT || arg2->typ == TTYP_FLT) {
		/* One or both values is a fp; result is fp */
		res.typ = TTYP_FLT;
		res.dat.dnum = getnumf(arg1) + getnumf(arg2);
	} else {
		res.typ = arg1->typ;
		res.dat.lnum = getnumi(arg1) + getnumi(arg2);
	}

	ac_push(&res);
	free(arg1);
	free(arg2);
	return 0;
}

int func_sub(void)
{
	token *arg1, *arg2;
	token res;
	
	arg2 = ac_pop_p();
	arg1 = ac_pop_p();

	if (arg1->typ == TTYP_FLT || arg2->typ == TTYP_FLT) {
		/* One or both values is a fp; result is fp */
		res.typ = TTYP_FLT;
		res.dat.dnum = getnumf(arg1) - getnumf(arg2);
	} else {
		res.typ = arg1->typ;
		res.dat.lnum = getnumi(arg1) - getnumi(arg2);
	}

	ac_push(&res);
	free(arg1);
	free(arg2);
	return 0;
}

int func_mult(void)
{
	token *arg1, *arg2;
	token res;
	
	arg1 = ac_pop_p();
	arg2 = ac_pop_p();

	if (arg1->typ == TTYP_FLT || arg2->typ == TTYP_FLT) {
		/* One or both values is a fp; result is fp */
		res.typ = TTYP_FLT;
		res.dat.dnum = getnumf(arg1) * getnumf(arg2);
	} else {
		res.typ = arg1->typ;
		res.dat.lnum = getnumi(arg1) * getnumi(arg2);
	}

	ac_push(&res);
	free(arg1);
	free(arg2);
	return 0;
}

int func_div(void)
{
	token *arg1, *arg2;
	token res;
	
	arg2 = ac_pop_p();
	arg1 = ac_pop_p();

	if(getnumf(arg2) == 0.0) {
		free(arg1);
		free(arg2);
		strcpy(error_buf, "Divide by 0.");
		return -1;
	}

	res.typ = TTYP_FLT;
	res.dat.dnum = getnumf(arg1) / getnumf(arg2);

	ac_push(&res);
	free(arg1);
	free(arg2);
	return 0;
}

int func_pow(void)
{
	token *arg1, *arg2;
	token res;
	
	arg2 = ac_pop_p();
	arg1 = ac_pop_p();

	if(arg1->typ != TTYP_FLT && arg2->typ != TTYP_FLT &&
					(arg2->dat.lnum > -1 || arg1->dat.lnum == 0)) {
		/* Both arguments are integers; find the integral solution */
		res.typ = arg1->typ;
		if(arg2->dat.lnum == 0)
			res.dat.lnum = 1;
		else if(arg1->dat.lnum == 0)
			res.dat.lnum = 0;
		else {
			int i;
			res.dat.lnum = arg1->dat.lnum;
			for(i = 1; i < arg2->dat.lnum; i++)
				res.dat.lnum *= arg1->dat.lnum;
		}
	} else {
		res.typ = TTYP_FLT;
		res.dat.dnum = pow(getnumf(arg1), getnumf(arg2));
	}
	
	ac_push(&res);
	free(arg1);
	free(arg2);
	return 0;
}

int func_log(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = log10(getnumf(arg1));
	ac_push(&res);
	free(arg1);
	return 0;
}

int func_ln(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = log(getnumf(arg1));
	ac_push(&res);
	free(arg1);
	return 0;
}

int func_sqr(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	res.typ = arg1->typ;
	if(arg1->typ == TTYP_FLT)
		res.dat.dnum = arg1->dat.dnum * arg1->dat.dnum;
	else
		res.dat.lnum = arg1->dat.lnum * arg1->dat.lnum;

	ac_push(&res);
	free(arg1);
	return 0;
}

int func_sqrt(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = sqrt(getnumf(arg1));
	ac_push(&res);
	free(arg1);
	return 0;
}

int func_inv(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	if(getnumf(arg1) == 0.0) {
		free(arg1);
		strcpy(error_buf, "Divide by 0.");
		return -1;
	}
	
	res.typ = TTYP_FLT;
	res.dat.dnum = 1.0 / getnumf(arg1);
	ac_push(&res);
	free(arg1);
	return 0;
}

int func_neg(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	if(arg1->typ != TTYP_FLT) {
		res.typ = arg1->typ;
		res.dat.lnum = 0 - arg1->dat.lnum;
	} else {
		res.typ = TTYP_FLT;
		res.dat.dnum = 0.0 - arg1->dat.dnum;
	}
	
	ac_push(&res);
	free(arg1);
	return 0;
}

int func_exp(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = exp(getnumf(arg1));
	
	ac_push(&res);
	free(arg1);
	return 0;
}

int func_fact(void)
{
	token *arg1;
	token res;
	ac_int_t i;
	
	arg1 = ac_pop_p();

	res.typ = TTYP_DEC;
	for(i = 2, res.dat.lnum = 1; i <= getnumi(arg1); i++)
		res.dat.lnum *= i;
	
	ac_push(&res);
	free(arg1);
	return 0;
}

int func_sin(void)
{
	token *arg1;
	token res;
	
	if(opt.deg) func_torad();
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = sin(getnumf(arg1));

	ac_push(&res);
	free(arg1);
	return 0;
}

int func_cos(void)
{
	token *arg1;
	token res;
	
	if(opt.deg) func_torad();
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = cos(getnumf(arg1));

	ac_push(&res);
	free(arg1);
	return 0;
}

int func_tan(void)
{
	token *arg1;
	token res;
	
	if(opt.deg) func_torad();
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = tan(getnumf(arg1));

	ac_push(&res);
	free(arg1);
	return 0;
}

int func_asin(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	if(getnumf(arg1) < -1.0 || getnumf(arg1) > 1.0) {
		sprintf(error_buf, "Invalid argument.");
		free(arg1);
		return -1;
	}

	res.typ = TTYP_FLT;
	res.dat.dnum = asin(getnumf(arg1));

	ac_push(&res);
	free(arg1);
	
	if(opt.deg) func_todeg();
	
	return 0;
}

int func_acos(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	if(getnumf(arg1) < -1.0 || getnumf(arg1) > 1.0) {
		sprintf(error_buf, "Invalid argument.");
		free(arg1);
		return -1;
	}

	res.typ = TTYP_FLT;
	res.dat.dnum = acos(getnumf(arg1));

	ac_push(&res);
	free(arg1);

	if(opt.deg) func_todeg();
	
	return 0;
}

int func_atan(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = atan(getnumf(arg1));

	ac_push(&res);
	free(arg1);

	if(opt.deg) func_todeg();
	
	return 0;
}

int func_sl(void)
{
	token *arg1, *arg2;
	token res;
	
	arg2 = ac_pop_p();
	arg1 = ac_pop_p();

	res.typ = (arg1->typ == TTYP_HEX ? TTYP_HEX : TTYP_DEC);
	res.dat.lnum = getnumi(arg1) << getnumi(arg2);
	
	ac_push(&res);
	free(arg1);
	free(arg2);
	return 0;
}

int func_sr(void)
{
	token *arg1, *arg2;
	token res;
	
	arg2 = ac_pop_p();
	arg1 = ac_pop_p();

	res.typ = (arg1->typ == TTYP_HEX ? TTYP_HEX : TTYP_DEC);
	res.dat.lnum = getnumi(arg1) >> getnumi(arg2);
	
	ac_push(&res);
	free(arg1);
	free(arg2);
	return 0;
}

int func_and(void)
{
	token *arg1, *arg2;
	token res;
	
	arg2 = ac_pop_p();
	arg1 = ac_pop_p();

	res.typ = (arg1->typ == TTYP_HEX ? TTYP_HEX : TTYP_DEC);
	res.dat.lnum = getnumi(arg1) & getnumi(arg2);
	
	ac_push(&res);
	free(arg1);
	free(arg2);
	return 0;
}

int func_or(void)
{
	token *arg1, *arg2;
	token res;
	
	arg2 = ac_pop_p();
	arg1 = ac_pop_p();

	res.typ = (arg1->typ == TTYP_HEX ? TTYP_HEX : TTYP_DEC);
	res.dat.lnum = getnumi(arg1) | getnumi(arg2);
	
	ac_push(&res);
	free(arg1);
	free(arg2);
	return 0;
}

int func_xor(void)
{
	token *arg1, *arg2;
	token res;
	
	arg2 = ac_pop_p();
	arg1 = ac_pop_p();

	res.typ = (arg1->typ == TTYP_HEX ? TTYP_HEX : TTYP_DEC);
	res.dat.lnum = getnumi(arg1) ^ getnumi(arg2);
	
	ac_push(&res);
	free(arg1);
	free(arg2);
	return 0;
}

int func_deg(void)
{
	opt.deg = TRUE;
	return 0;
}

int func_rad(void)
{
	opt.deg = FALSE;
	return 0;
}

int func_fix(void)
{
	token *arg1;
	
	arg1 = ac_pop_p();

	if(getnumi(arg1) < 0 || getnumi(arg1) > 16) {
		strcpy(error_buf, "Invalid argument");
		return -1;
	}

	opt.fix = getnumi(arg1);
	free(arg1);

	return 0;
}

int func_num(void)
{
	opt.num = !opt.num;	
	return 0;
}

int func_todeg(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = getnumf(arg1) * 180.0 / PI;

	ac_push(&res);
	free(arg1);
	return 0;
}

int func_torad(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = getnumf(arg1) * PI / 180.0;

	ac_push(&res);
	free(arg1);
	return 0;
}

int func_ipart(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	res.typ = TTYP_DEC;
	res.dat.lnum = getnumi(arg1);

	ac_push(&res);
	free(arg1);

	return 0;
}

int func_fpart(void)
{
	token *arg1;
	token res;
	double dummy;
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = fabs(modf(getnumf(arg1), &dummy));

	ac_push(&res);
	free(arg1);

	return 0;
}

int func_flt(void)
{
	token *arg1;
	token res;
	
	arg1 = ac_pop_p();

	res.typ = TTYP_FLT;
	res.dat.dnum = getnumf(arg1);

	ac_push(&res);
	free(arg1);
	return 0;
}

int func_tohex(void)
{
	token *arg1;
	token res;

	arg1 = ac_pop_p();

	res.typ = TTYP_HEX;
	res.dat.lnum = getnumi(arg1);
	
	ac_push(&res);
	free(arg1);
	return 0;
}

int func_todec(void)
{
	token *arg1;
	token res;

	arg1 = ac_pop_p();

	res.typ = TTYP_DEC;
	res.dat.lnum = getnumi(arg1);
	
	ac_push(&res);
	free(arg1);
	return 0;
}

int func_swap(void)
{
	token *arg1, *arg2;
	
	arg1 = ac_pop_p();
	arg2 = ac_pop_p();

	ac_push_p(arg1);
	ac_push_p(arg2);
	return 0;
}

int func_drop(void)
{
	token *arg1;

	arg1 = ac_pop_p();

	free_tok(arg1);
	free(arg1);
	return 0;
}

int func_rot(void)
{
	token *arg1, *arg2, *arg3;
	
	arg1 = ac_pop_p();
	arg2 = ac_pop_p();
	arg3 = ac_pop_p();

	ac_push_p(arg1);
	ac_push_p(arg3);
	ac_push_p(arg2);
	return 0;
}

int func_roll(void)
/* Unimplemented */
{
	return 0;
}

int func_clr(void)
{
	while(s->h) 
		func_drop();
		
	return 0;
}	

int func_dropn(void)
{
	token *arg1;
	ac_int_t i;
	
	arg1 = ac_pop_p();
	
	if(getnumi(arg1) < 0) {
		strcpy(error_buf, "Invalid argument");
		free(arg1);
		return -1;
	}

	for(i = 0; i < getnumi(arg1) && s->h; i++)
		func_drop();

	free(arg1);
	return 0;
}

int func_dup(void)
{
	token *arg1;
	token arg2;
	
	arg1 = ac_pop_p();
	
	arg2 = *arg1;
	if(arg1->typ == TTYP_STR) {
		arg2.dat.str = strdup(arg1->dat.str);
	}

	ac_push_p(arg1);
	ac_push(&arg2);
	return 0;
}

int func_sto(void)
{
	token *arg1, *arg2;
	
	arg2 = ac_pop_p();
	arg1 = ac_pop_p();

	if(!(arg2->typ & TTYP_INT)) {
		strcpy(error_buf, "Invalid argument.");
		free(arg1);
		free(arg2);
		return -1;
	}

	set_reg(getnumi(arg2), *arg1);

	free(arg1);
	return 0;
}

int func_rcl(void)
{
	token *arg1;
	token *res;
	
	arg1 = ac_pop_p();
	
	res = get_reg(getnumi(arg1));
	
	if(res == NULL) {
		sprintf(error_buf, "Nothing in register %lld", getnumi(arg1));
		free(arg1);
		return -1;
	}

	ac_push(res);
	free(arg1);
	return 0;
}

int func_del(void)
{
	token *arg1;

	arg1 = ac_pop_p();
	
	del_reg(getnumi(arg1));
	
	free(arg1);
	return 0;
}

int func_stack(void)
{
	token *arg1;
	
	arg1 = ac_pop_p();

	s = get_stack(getnumi(arg1));

	free(arg1);
	return 0;
}
			
int func_save(void)
{
	token *arg1;
	FILE *f;
	
	llist_node *n, *m;
	stack_node *stkn;
	reg_node *regn;
		
	arg1 = ac_pop_p();

	if(!(f = fopen(arg1->dat.str, "w"))) {
		strcpy(error_buf, "Unable to open file.");
		free(arg1);
		return -1;
	}
	
	for(n = stacks.h; n; n = n->next) {
		stkn = n->data;
		if(!stkn->stk.h)
			continue;

		fprintf(f, "!STACK ID: %lld\n", stkn->id);
		fprintf(f, "!STACK SIZE: %d\n", stkn->stk.size);
		for(m = stkn->stk.h; m; m = m->next) {
			fprintf(f, "%s\n", tok_to_savstr(*(token *)(m->data)));
		}
	}

	for(n = regs.h; n; n = n->next) {
		regn = n->data;
		fprintf(f, "!REGISTER ID: %lld\n", regn->id);
		fprintf(f, "%s\n", tok_to_savstr(regn->tok));
	}
		
	fclose(f);
	free(arg1);
	return 0;
}

int func_load(void)
{
	int i;
	ac_int_t id, size;
	token *arg1;
	token tok;
	FILE *f;
	char buf[1024];
	
	arg1 = ac_pop_p();

	if(!(f = fopen(arg1->dat.str, "r"))) {
		strcpy(error_buf, "Unable to open file.");
		free(arg1);
		return -1;
	}

	free_stacklist();
	
	while(fgets(buf, 1023, f)) {
		if(sscanf(buf, "!STACK ID: %lld\n", &id) == 0)
			continue;
			
		s = get_stack(id);
		
		fgets(buf, 1023, f);	
		sscanf(buf, "!STACK SIZE: %lld\n", &size);

		for(i = 0; i < size; i++) {
			fgets(buf, 1023, f);
			if(buf[strlen(buf) - 1] == '\n') {
				buf[strlen(buf) - 1] = '\0';
			}
				
			tok = savstr_to_tok(buf);
			
			llist_append(s, &tok);
		}
	}

	s = get_stack(0);
	
	fclose(f);

	free(arg1);
	return 0;
}

int func_loadregs(void)
{
	ac_int_t id;
	token *arg1;
	token tok;
	FILE *f;
	char buf[1024];
	
	arg1 = ac_pop_p();

	if(!(f = fopen(arg1->dat.str, "r"))) {
		strcpy(error_buf, "Unable to open file.");
		free(arg1);
		return -1;
	}

	free_reglist();
	
	while(fgets(buf, 1023, f)) {
		if(sscanf(buf, "!REGISTER ID: %lld\n", &id) == 0)
			continue;
			
		fgets(buf, 1023, f);	
		if(buf[strlen(buf) - 1] == '\n') {
			buf[strlen(buf) - 1] = '\0';
		}
				
		tok = savstr_to_tok(buf);
			
		set_reg(id, tok);
	}

	fclose(f);

	free(arg1);
	return 0;
}

int func_execsc(void)
{
	token *arg1;
	char *cmd;
	int ret;
	
	arg1 = ac_pop_p();

	cmd = get_shortcut(getnumi(arg1));

	if(cmd) 
		ret = exec_string(cmd);
	else {
		strcpy(error_buf, "No such shortcut.");
		free(arg1);
		return -1;
	}
	
	free(arg1);
	return ret;
}

int func_setsc(void)
{
	token *arg1, *arg2;
	char ret;
	
	arg2 = ac_pop_p();
	arg1 = ac_pop_p();

	if(!(arg2->typ & TTYP_INT)) {
		strcpy(error_buf, "Invalid argument.");
		free(arg1);
		free(arg2);
		return -1;
	}
	if(!(arg1->typ & TTYP_STR)) {
		strcpy(error_buf, "Invalid argument.");
		free(arg1);
		free(arg2);
		return -1;
	}

	ret = add_shortcut(arg2->dat.lnum, arg1->dat.str);
	free(arg1);
	free(arg2);
	
	if(ret) return 0;
	else {
		strcpy(error_buf, "Couldn't add shortcut.");
		return -1;
	}
}

int func_delsc(void)
{
	token *arg1;
	
	arg1 = ac_pop_p();

	del_shortcut(arg1->dat.lnum);
	
	free(arg1);
	return 0;
}
		
int func_help(void)
{
	do_help();
	return 0;
}
				
int func_quit(void)
{
	flags.quit = TRUE;
	return 0;
}
