/* iface_g.c
 *
 * GTK+ interface for acalc */

#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "calc.h"

#ifdef PIXMAPS
#include "pixmaps.h"
#define NUM_INDICATORS 2 
#else
#define NUM_INDICATORS 0
#endif

#define WINDOW_TITLE "Aardvark Calculator 0.0.48"

/* Indicator structure */
struct ind {
	GtkWidget *icon;
	GdkPixmap **pixmap;
	GdkBitmap **bitmap;
	short state;
};

struct mainwin {
	GtkWidget *win; /* Main window */
	GtkWidget *mbox; /* Main box */
	
	GtkWidget *ibox; /* Indicator box */
	struct ind *indicator;
	GtkWidget *itextbox; /* Widget containing the status text label */
	GtkWidget *itext; /* Status text */

	GtkWidget *tbox; /* Toolbar box */
	GtkWidget **tbuttons; /* Toolbar buttons */
	
	GtkWidget *sep1;
	GtkWidget *sbox; /* Box for the stack display and scrollbar */
	GtkWidget *stack; /* Stack display area */
	GtkWidget *stacksb; /* Stack scrollbar */
	GtkWidget *sep2;

	GtkWidget *ebox; /* Box for the entry widgets */
	GtkWidget *entry; /* Entry line */
	GtkWidget *enter; /* Enter Button */
} w;

void interface_init(int *argc, char ***argv);
	/* Initialize GTK+ and create the window */

void init_indicator(int inum, int mapnum, gpointer data);
	/* Initialize the given map for the given indicator, with data pointing to
	 * the pixmap data */

void interface_end(void);

void interface_loop(void);
	/* Main calculator function; called after data structures are set up and
	 * exits when it ends.  In this case, simply calls gtk_main() */

void entry_activate(GtkEditable *wid, gpointer data);
void button_activate(GtkButton *wid, gpointer data);
	/* called When the specified entry widget is activated */

void toolbar_clicked(GtkButton *wid, gpointer data);
	/* called when the toolbar button, stored in data, is pushed */

gint delete_event(GtkWidget *widget, GdkEvent *event, gpointer data);
	/* just calls the gtk_main_quit function */
gint handle_keypress(GtkWidget *widget, GdkEvent *event, gpointer data);
	/* look at a keypress:
	 * if the cursor is not at the end of the line, or there's an odd number of 
	 * '\"' in the line, break to the default handler
	 * if it's an IMMED shortcut --
	 *   add a space and the associated function to the entry widget, then 
	 *   activate the entry. Abort the default handler.
	 * otherwise, if it's a standard shortcut --
	 *   if the entry widget it empty, (or contains only spaces?), add the 
	 *   appropriate function and activate, aborting the default handler
	 * if it gets here, run the default handler */
	/* data contains the signal id of the signal emitted */

void enter_line(char *s);
	/* Takes the given line of entered text and handles it; updating the display
	 * as well */

void update_display(void);
	/* Updates the stack and indicators to match the current state */

void do_help(void);
	/* Give a list of all available functions */
void help_ok(GtkButton *button, gpointer data);
gboolean help_quit(GtkWidget *widget, GdkEvent *event, gpointer data);
	/* Simply close the help window */


void interface_init(int *argc, char ***argv)
{
	int i;
	
	gtk_init(argc, argv);

	/* Create widgets */
	w.win = gtk_window_new(GTK_WINDOW_TOPLEVEL);

	w.mbox = gtk_vbox_new(FALSE, 0);
	
	w.ibox = gtk_hbox_new(FALSE, 0);
	
#ifdef PIXMAPS
	w.indicator = malloc(NUM_INDICATORS * sizeof(struct ind));
#endif

	w.itextbox = gtk_alignment_new(1, 0, 0, 1);
	w.itext = gtk_label_new("");

	if(tbnum) {
		w.tbox = gtk_hbox_new(FALSE, 0);
		w.tbuttons = malloc(sizeof(struct GtkWidget *) * tbnum);
		for(i = 0; i < tbnum; i++)
			w.tbuttons[i] = gtk_button_new_with_label(tbbuttons[i]);
	}
	
	/* w.sep1 = gtk_hseparator_new(); */
	w.sbox = gtk_hbox_new(FALSE, 0);
	w.stack = gtk_text_new(NULL, NULL);
	w.stacksb = gtk_vscrollbar_new(GTK_TEXT(w.stack)->vadj);
	/* w.sep2 = gtk_hseparator_new(); */

	w.ebox = gtk_hbox_new(FALSE, 0);
	w.entry = gtk_entry_new();
	w.enter = gtk_button_new_with_label("Enter");

#ifdef PIXMAPS
	/* Create the icon pixmaps */

	/* Indicator 0 is the degrees/radians indicator.  2 states: deg and rad */
	w.indicator[0].pixmap = malloc(2 * sizeof(GdkPixmap *));
	w.indicator[0].bitmap = malloc(2 * sizeof(GdkBitmap *));
	init_indicator(0, 0, deg_icon_xpm);
	init_indicator(0, 1, rad_icon_xpm);
	w.indicator[0].state = (opt.deg) ? 0 : 1;
	
	/* Indicator 1 is the error indicator.  3 states: 0 - err_none, 1 - err, and
	 * 2 - err_syntax */
	w.indicator[1].pixmap = malloc(3 * sizeof(GdkPixmap *));
	w.indicator[1].bitmap = malloc(3 * sizeof(GdkBitmap *));
	init_indicator(1, 0, err_none_icon_xpm);
	init_indicator(1, 1, err_icon_xpm);
	init_indicator(1, 2, err_syntax_icon_xpm);
	w.indicator[1].state = 0;
	
	for(i = 0; i < NUM_INDICATORS; i++) {
		w.indicator[i].icon = gtk_pixmap_new(
						w.indicator[i].pixmap[w.indicator[i].state], 
						w.indicator[i].bitmap[w.indicator[i].state]);
	}
#endif

	/* Set widget options */
	gtk_window_set_title(GTK_WINDOW(w.win), WINDOW_TITLE);
	gtk_window_set_policy(GTK_WINDOW(w.win), FALSE, TRUE, FALSE);
	gtk_signal_connect(GTK_OBJECT(w.win), "delete_event",
	        GTK_SIGNAL_FUNC(delete_event), NULL);
	
	gtk_hbutton_box_set_spacing_default(0);
	
	for(i = 0; i < tbnum; i++)
		gtk_signal_connect(GTK_OBJECT(w.tbuttons[i]), "clicked",
						GTK_SIGNAL_FUNC(toolbar_clicked), (gpointer) i);

	gtk_text_set_editable(GTK_TEXT(w.stack), FALSE);

	gtk_signal_connect(GTK_OBJECT(w.entry), "activate",
	        GTK_SIGNAL_FUNC(entry_activate), NULL);
	gtk_signal_connect(GTK_OBJECT(w.entry), "key_press_event",
					GTK_SIGNAL_FUNC(handle_keypress), NULL);
	gtk_signal_connect(GTK_OBJECT(w.enter), "clicked",
	        GTK_SIGNAL_FUNC(button_activate), NULL);

	/* Pack widgets */
	gtk_container_add(GTK_CONTAINER(w.win), w.mbox);
	
	gtk_box_pack_start(GTK_BOX(w.mbox), w.ibox, FALSE, FALSE, 0);
	
#ifdef PIXMAPS
	for(i = 0; i < NUM_INDICATORS; i++) {
		gtk_box_pack_start(GTK_BOX(w.ibox), w.indicator[i].icon, 
						FALSE, FALSE, 0);
	}
#endif

	gtk_container_add(GTK_CONTAINER(w.itextbox), w.itext);
	gtk_box_pack_start(GTK_BOX(w.ibox), w.itextbox, TRUE, TRUE, 0);

	if(tbnum) {
		gtk_box_pack_start(GTK_BOX(w.mbox), w.tbox, FALSE, FALSE, 0);
		for(i = 0; i < tbnum; i++) {
			gtk_box_pack_start(GTK_BOX(w.tbox), w.tbuttons[i], TRUE, TRUE, 0);
		}
	}

	/* gtk_box_pack_start(GTK_BOX(w.mbox), w.sep1, FALSE, FALSE, 0); */
	
	gtk_box_pack_start(GTK_BOX(w.mbox), w.sbox, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(w.sbox), w.stack, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(w.sbox), w.stacksb, FALSE, FALSE, 0);
	
	/* gtk_box_pack_start(GTK_BOX(w.mbox), w.sep2, FALSE, FALSE, 0); */

	gtk_box_pack_start(GTK_BOX(w.mbox), w.ebox, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(w.ebox), w.entry, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(w.ebox), w.enter, FALSE, FALSE, 0);

	/* Show widgets */
	gtk_widget_show_all(w.win);
}

void init_indicator(int inum, int mapnum, gpointer data)
{
	GdkPixmap *pixmap;
	GdkBitmap *bitmap;
	GtkStyle *bgstyle;
	GdkColormap *map;
	
	bgstyle = gtk_widget_get_style(w.win);
	map = gdk_colormap_get_system();
	pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, map, &bitmap,
					&bgstyle->bg[GTK_STATE_NORMAL], data);
	w.indicator[inum].pixmap[mapnum] = pixmap;
	w.indicator[inum].bitmap[mapnum] = bitmap;
}

void interface_loop(void)
{
	/* Set up the initial display */
	update_display();

	gtk_main();
}

void interface_end(void)
{
	/* Nothing to be done */
	return;
}

void entry_activate(GtkEditable *wid, gpointer data)
{
	gchar *s;
	
	s = gtk_editable_get_chars(GTK_EDITABLE(w.entry), 0, -1);
	enter_line((char *)s);
	
	g_free(s);

	/* Clear the entry for the next line of input */
	gtk_entry_set_text(GTK_ENTRY(w.entry), "");
}

void button_activate(GtkButton *wid, gpointer data)
{
	entry_activate((GtkEditable *)wid, data);
}

void toolbar_clicked(GtkButton *wid, gpointer data)
{
	char buf[7];
	
	sprintf(buf, " %d sc", (int) data + 1);
	gtk_entry_append_text(GTK_ENTRY(w.entry), buf);
	entry_activate((GtkEditable *)wid, data);
}

gint delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gtk_main_quit();
	return FALSE;
}

void enter_line(char *s)
{
	int exec_return;
				
	if(strlen(s) == 0) /* Nothing entered */
		return;
	exec_return = exec_string(s);
	if(exec_return == -1) /* Syntax error */
		flags.syntax = TRUE;
	else if (exec_return == -2) /* Calculation error */
		flags.error = TRUE;

	if(flags.quit)
		delete_event(NULL, NULL, NULL);
	
	update_display();

	/* Reset flags */
	flags.error = FALSE;
	flags.syntax = FALSE;
}

void update_display(void)
{
	int i;
	gchar display[256];
	char *tok;
	GtkAdjustment *adj;
	llist_node *n;
	
	/* Update the stack display */
	/* For now, simply wipe out and reenter the entire stack. */

	gtk_text_freeze(GTK_TEXT(w.stack));
	gtk_editable_delete_text(GTK_EDITABLE(w.stack), 0, -1);

	for(i = s->size, n = s->t; n; n = n->prev, i--) {
		tok = tok_to_str(*(token *)(n->data));
		if(opt.num)
			sprintf(display, "%d: %s", i, tok);
		else
			strcpy(display, tok);
		
		gtk_text_insert(GTK_TEXT(w.stack), NULL, 
						&(w.stack->style->fg[GTK_STATE_ACTIVE]), NULL, display, 
		        strlen(display));
		if(i > 1) gtk_text_insert(GTK_TEXT(w.stack), NULL, NULL, NULL, "\n", 1);
	}
	gtk_text_thaw(GTK_TEXT(w.stack));
	
	/* Update the status line text */
	if(flags.error || flags.syntax)
		strcpy(display, error_buf);
	else
		display[0] = '\0';

	gtk_label_set_text(GTK_LABEL(w.itext), display);

	/* Update the status line indicators */
#ifdef PIXMAPS
	w.indicator[0].state = (opt.deg) ? 0 : 1;
	w.indicator[1].state = flags.syntax ? 2 : (flags.error ? 1 : 0);
	for(i = 0; i < NUM_INDICATORS; i++)
		gtk_pixmap_set(GTK_PIXMAP(w.indicator[i].icon),
						w.indicator[i].pixmap[w.indicator[i].state], 
						w.indicator[i].bitmap[w.indicator[i].state]);
#endif

	/* Set focus to the text entry widget */
	gtk_widget_grab_focus(GTK_WIDGET(w.entry));

	/* Adjust the stack window to the very bottom possible */
	adj = gtk_range_get_adjustment(&(GTK_SCROLLBAR(w.stacksb)->range));
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj), adj->upper);
}

gint handle_keypress(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	int i, l;
	char *s;

	/* break if not at end of line, or if there's a odd number of double quotes */
	if(gtk_editable_get_position(GTK_EDITABLE(w.entry)) 
					< (l = strlen(s = gtk_entry_get_text(GTK_ENTRY(w.entry))))) {
		return FALSE;
	}
	
	/* s = pointer to start of entry string */
	for(i = 0; *s; s++)
 		if(*s == '\"') i++;
	if(i % 2) {
		return FALSE;
	}

	/* If the last character in the string is an 'e', and the key pressed is
	 * '-', then break.  (Sort of a hack, but it's necessary to enter negative
	 * exponents in a sane fashion.) */
	s = gtk_entry_get_text(GTK_ENTRY(w.entry));
	if(l && (s[l - 1] == 'e' || s[l - 1] == 'E') && event->key.keyval == '-')
	{
		return FALSE;
	}
	
	for(i = 0; keys[i].key; i++) {
		/* l = length of entry string */
		if(((l == 0) ? TRUE : (keys[i].flags & IMMED)) 
					&& (event->key.keyval == keys[i].key)) {
			gtk_entry_append_text(GTK_ENTRY(w.entry), " ");
			gtk_entry_append_text(GTK_ENTRY(w.entry), keys[i].cmd);
			gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
			gtk_signal_emit_by_name(GTK_OBJECT(w.entry), "activate");
			return TRUE;
		}
	}

	return FALSE;
}

void do_help(void)
{
	GtkWidget *win;
	GtkWidget *box;
	GtkWidget *listview;
	GtkWidget *list;
	GtkWidget *align;
	GtkWidget *button;
	GtkRequisition rsize;

	int i, j;
	int buf[10], *bp;
	char pbuf[80], pbuf2[80], *pbp[2];

	/* create widgets */
	win = gtk_window_new(GTK_WINDOW_DIALOG);
	box = gtk_vbox_new(FALSE, 5);
	listview = gtk_scrolled_window_new(NULL, NULL);
	list = gtk_clist_new(2);
	align = gtk_alignment_new(.5, 0, 0, 0);
	button = gtk_button_new_with_label("OK");

	/* Set widget options */
	gtk_window_set_title(GTK_WINDOW(win), "Help");
	gtk_window_set_policy(GTK_WINDOW(win), FALSE, TRUE, FALSE);

	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(listview),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	gtk_clist_set_column_auto_resize(GTK_CLIST(list), 0, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(list), 1, TRUE);
	/* Load the function names and shortcuts into the clist */
	
	pbp[0] = pbuf; pbp[1] = pbuf2;
	for(i = 0; funclist[i].str; i++) {
		strcpy(pbuf, funclist[i].str);
		for(j = 0, bp = buf; keys[j].key; j++) 
			if(!strcmp(funclist[i].str, keys[j].cmd)) 
				*(bp++) = keys[j].key;
		*bp = 0;
		pbuf2[0] = '\0';
		if(buf[0]) {
			for(j = 0; buf[j]; j++) {
				if(j) strcat(pbuf2, ", ");
				strcat(pbuf2, gdk_keyval_name(buf[j]));
			}
		}
		gtk_clist_append(GTK_CLIST(list), pbp);
	}

	gtk_widget_get_child_requisition(GTK_WIDGET(list), &rsize);

	gtk_window_set_default_size(GTK_WINDOW(win), rsize.width + 25, 400);

	gtk_signal_connect(GTK_OBJECT(button), "clicked",
					GTK_SIGNAL_FUNC(help_ok), win);
	gtk_signal_connect(GTK_OBJECT(win), "delete_event",
					GTK_SIGNAL_FUNC(help_quit), win);

	/* Pack widgets */
	gtk_container_add(GTK_CONTAINER(win), box);	
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(listview), list);
	gtk_box_pack_start(GTK_BOX(box), listview, TRUE, TRUE, 0);
	gtk_container_add(GTK_CONTAINER(align), button);
	gtk_box_pack_start(GTK_BOX(box), align, FALSE, FALSE, 0);

	gtk_widget_show_all(win);
}

void help_ok(GtkButton *button, gpointer data)
{
	gtk_object_destroy(GTK_OBJECT(data));
}
	
gboolean help_quit(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gtk_object_destroy(GTK_OBJECT(data));
	return FALSE;
}
