/* llist.h */

#ifndef _llist_h
#define _llist_h

#include "type.h"

#define FOREACH_NODE(x,l) for((x) = (l).h; (x); (x) = (x)->next)

struct llist_node_t {
	struct llist_node_t *next;
	struct llist_node_t *prev;
	void *data;
};

typedef struct llist_node_t llist_node;

struct llist_t {
	llist_node *h; /* the list head */
	llist_node *t; /* the list tail */
	int db; /* size of each element */
	int size; /* how many elements are currently in the list */
};

typedef struct llist_t llist;

llist llist_new( int db );
	/* return a new linked list structure, with db bytes per piece of data. */
	
void llist_free( llist *l );
	/* Free the given llist and all allocated node/data memory. */

void llist_free_fn( llist *l, void (* free_fn)(void *ptr) );
	/* Free the given llist, using the specified function rather than free()
	 * to free the data memory */
	
void llist_shallow_free( llist *l );
	/* Same as above, minus data deletion */

void *llist_prepend( llist *l, void *data );
	/* Prepend the given data to the list.  (Use this instead of append,
	 * usually; this function doesn't involve searching through the list.) */ 
void *llist_prepend_p( llist *l, void *data );
	/* Prepend the *already allocated* memory to the list (this simply copies
	 * the pointer, rather than making a copy of the data.)  Be sure to use
	 * the *_shallow_* functions if you don't want the list handlers to free
	 * the data */
llist_node *llist_prepend_n( llist *l, void *data );
llist_node *llist_prepend_pn( llist *l, void *data );
	/* Return the node instead of the data */

void *llist_append( llist *l, void *data );
	/* Append the given data to the list; copying the first db bytes of data */
void *llist_append_p( llist *l, void *data );
	/* Append the already allocated data to the list (see the caveats for
	 * llist_prepend_p) */
llist_node *llist_append_n( llist *l, void *data );
/* Same as llist_append, but return the node instead of the data */
llist_node *llist_append_pn( llist *l, void *data );
/* Same as llist_append_p, but return the node instead of the data */

void *llist_insert( llist *l, void *data, 
					int (*compare) (void *, void* ));
void *llist_insert_p( llist *l, void *data, 
					  int (*compare) (void *, void* ));
llist_node *llist_insert_n( llist *l, void *data, 
							int (*compare) (void *, void* ));
llist_node *llist_insert_pn( llist *l, void *data, 
							 int (*compare) (void *, void* ));
/* If you're interested in keeping the list sorted, use this function */
/* Note: It's broken.  Should you want to use it, you need to fix the
 * llist_insert_pn function to properly handle the new list structure */

void llist_remove( llist *l, llist_node *n );
	/* Removes the node and data from the given list */
	
void llist_shallow_remove( llist *l, llist_node *n );
	/* Removes just the node; the data pointer is left intact. */
	
bool llist_remove_data( llist *l, void *data, bool search_entire );
    /* Removes all nodes containing given data; returns TRUE if removal
	 * was successful */

bool llist_shallow_remove_data( llist *l, void *data, bool search_entire );
    /* Same, but doesn't free data */

llist_node *llist_find( llist *l, void *data );
    /* Find given data and return the (first) node containing it */

int llist_size( llist *l );
/* returns the number of elements in a given list */

void *llist_arrayify( llist *l, int *s );
/* returns an array containing the elements in the list, and stores
   the length of the array in *s */

void *llist_arrayify_pointers( llist *l, int *s );
/* returns an array containing pointers to the elements in the list,
   and stores the length of the array in *s */

llist_node *llist_find_idx( llist *l, int n );
/* Find the nth element in a linked list */

void llist_foreach( llist *l, void (*fn)( void *data ) );
/* Run a given routine on the data of each element in the list */

#endif
