#define XK_MISCELLANY
#define XK_LATIN1
#include <Xm/XmAll.h>
#include <X11/keysymdef.h>
#include <Mrm/MrmPublic.h>
#include <features.h>
#include <nl_types.h>
#include <locale.h>
#include <math.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "calc.h"
#include "calc_msg.h"

#define STACK_SIZE 30

#ifdef LITTLE_ENDIAN
#define R1 3
#define R2 2
#define R3 1
#define R4 0
#define D1 1
#define D2 0
#define D3 1
#define D4 0
#else
#define R1 0
#define R2 1
#define R3 2
#define R4 3
#define D1 0
#define D2 1
#define D3 0
#define D4 1
#endif
#ifdef INT_64
typedef long long Int;
typedef unsigned long long UInt;
typedef union UNUM {
  UInt L;
  unsigned int I[2];
} UNum;
typedef unsigned long UShort;
#define UINT_SHIFT 63
#else
typedef long int Int;
typedef unsigned long int UInt;
typedef union UNUM {
  unsigned long L;
  unsigned short I[2];
} UNum;
typedef unsigned short UShort;
#define UINT_SHIFT 31
#endif

typedef enum {GetNumOrUnary, GetNum, ProcessNum, 
	      ProcessedEqual, ProcessedImmedOp, ExponentMode} CalcInputState;
typedef struct ACCUM
{
  Int IAccum;
  double FAccum;
} Accum;

typedef struct COLOR_RESOURCE {
  Pixel NumColor;
  Pixel CtlColor;
  Pixel MathColor;
  Pixel LogicColor;
  Pixel ModeColor;
  Pixel OperColor;
} ColorResource;

typedef struct BUTTON_STATE
{
  short Id;
  Boolean State;
} ButtonState;

void AppendDigit(int Num);
void AppendExponent(int Num);
void ExitProgram(Widget CBWidget, XtPointer CallbackData, 
		 XtPointer WidgetInfo);
void InitCalc(void);
int main(int argc, char *argv[]);
int IntegerMultiply(Int P1, Int P2, Int *Product);
void KeyStroke(Widget SrcWidget, XEvent *EventCode, String Input, 
	      Cardinal Count);
char *NumToA(Int Num, int Base);
void PopupError(nl_catd Cat, int Set, int MsgNum, int Flags, char * Msg, ...);
void ProcessImmedOp(int Request);
void ProcessInput(int Request);
int ProcessPrevOperator(void);
int ProcessUnaryOp(void);
void PutMessage(nl_catd Cat, int Set, int Msg, char *DefMsg, ...);
void RegisterWidget(Widget CBWidget, XtPointer CallbackData, 
		    XtPointer WidgetInfo);
void ResetBase(void);
void SetupMainWindow(int *ArgCount, char *Args[]);
void SelectWidget(Widget CBWidget, XtPointer CallbackData, 
		  XtPointer WidgetInfo);
void SetBase(Widget CBWidget, XtPointer CallbackData, XtPointer WidgetInfo);
void SetDisplayMode(Widget CBWidget, XtPointer CallbackData, 
		    XtPointer WidgetInfo);
void SetTempBase(int NewBase, ButtonState *ButtonP);
void SetTrigMode(Widget CBWidget, XtPointer CallbackData, 
		 XtPointer WidgetInfo);
void UpdateDisplay(Accum *Acc);

XtActionsRec Actions[] = {
  {"Key", (XtPointer) KeyStroke}
};
XtAppContext AppContext;
int Base;
ButtonState BinaryButtons[] = {
  {BTN_0, True}, {BTN_1, True}, {BTN_2, False}, {BTN_3, False},
  {BTN_4, False}, {BTN_5, False}, {BTN_6, False}, {BTN_7, False},
  {BTN_8, False}, {BTN_9, False}, {BTN_A, False}, {BTN_B, False},
  {BTN_C, False}, {BTN_D, False}, {BTN_E, False}, {BTN_F, False}
};
CalcInputState CalcState;
Widget CalcWidget[NUM_WIDGETS];
MrmRegisterArg Callbacks[] =
{
  {"RegisterWidget", (XtPointer) RegisterWidget},
  {"SelectWidget", (XtPointer) SelectWidget},
  {"SetBase", (XtPointer) SetBase},
  {"SetDisplayMode", (XtPointer) SetDisplayMode},
  {"SetTrigMode", (XtPointer) SetTrigMode}
};
ColorResource Colors;
XtResource ColorResources[] = {
  {"numColor", "NumColor", XtRPixel, sizeof(Pixel), 
   XtOffsetOf(ColorResource, NumColor), XtRString, XtDefaultForeground},
  {"ctlColor", "CtlColor", XtRPixel, sizeof(Pixel),
   XtOffsetOf(ColorResource, CtlColor), XtRString, XtDefaultForeground},
  {"mathColor", "MathColor", XtRPixel, sizeof(Pixel),
   XtOffsetOf(ColorResource, MathColor), XtRString, XtDefaultForeground},
  {"logicColor", "LogicColor", XtRPixel, sizeof(Pixel),
   XtOffsetOf(ColorResource, LogicColor), XtRString, XtDefaultForeground},
  {"modeColor", "ModeColor", XtRPixel, sizeof(Pixel),
   XtOffsetOf(ColorResource, ModeColor), XtRString, XtDefaultForeground},
  {"operColor", "OperColor", XtRPixel, sizeof(Pixel),
   XtOffsetOf(ColorResource, OperColor), XtRString, XtDefaultForeground}
};
short CtlColorButtons[] = {
  BTN_QUIT, BTN_CLEAR_ALL, BTN_CE
};
Accum CurrentAccum;
ButtonState DecimalButtons[] = {
  {BTN_0, True}, {BTN_1, True}, {BTN_2, True}, {BTN_3, True},
  {BTN_4, True}, {BTN_5, True}, {BTN_6, True}, {BTN_7, True},
  {BTN_8, True}, {BTN_9, True}, {BTN_A, False}, {BTN_B, False},
  {BTN_C, False}, {BTN_D, False}, {BTN_E, False}, {BTN_F, False}
};
int DisplayMode;
int Exponent;
ButtonState HexButtons[] = {
  {BTN_0, True}, {BTN_1, True}, {BTN_2, True}, {BTN_3, True},
  {BTN_4, True}, {BTN_5, True}, {BTN_6, True}, {BTN_7, True},
  {BTN_8, True}, {BTN_9, True}, {BTN_A, True}, {BTN_B, True},
  {BTN_C, True}, {BTN_D, True}, {BTN_E, True}, {BTN_F, True}
};
ButtonState LogicButtons[] = {
  {BTN_FAC, False}, {BTN_POW_X, False}, {BTN_SIN, False}, {BTN_SQRT, False},
  {BTN_INV_POW_X, False}, {BTN_COS, False}, {BTN_SQUARE, False}, 
  {BTN_LOG, False}, {BTN_TAN, False}, {BTN_HYP, False}, {BTN_LN, False},
  {BTN_ASIN, False}, {BTN_POW_10, False}, {BTN_ACOS, False},
  {BTN_POW_E, False}, {BTN_ATAN, False}, {BTN_DOT, False},
  {BTN_NOT, True}, {BTN_XOR, True}, {BTN_OR, True}, {BTN_AND, True},
  {BTN_LSHIFT, True}, {BTN_RSHIFT, True}, {BTN_DEC_INPUT, True},
  {BTN_BIN_INPUT, True}, {BTN_OCT_INPUT, True}, {BTN_HEX_INPUT, True},
  {BTN_EE, False}, {BTN_MOD, False}, {BTN_FLOOR, False}, {BTN_CEIL, False},
  {BTN_INT, False}, {BTN_FRAC, False}, {BTN_SCI, False}, {BTN_NORM, False},
  {BTN_DEG, False}, {BTN_RAD, False}, {BTN_GRA, False}
};
short LogicColorButtons[] = {
  BTN_NOT, BTN_XOR, BTN_OR, BTN_AND, BTN_LSHIFT, BTN_RSHIFT
};
Widget MainWindow;
MrmType MainWindowClass;
ButtonState MathButtons[] = {
  {BTN_FAC, True}, {BTN_POW_X, True}, {BTN_SIN, True}, {BTN_SQRT, True},
  {BTN_INV_POW_X, True}, {BTN_COS, True}, {BTN_SQUARE, True}, 
  {BTN_LOG, True}, {BTN_TAN, True}, {BTN_HYP, True}, {BTN_LN, True},
  {BTN_ASIN, True}, {BTN_POW_10, True}, {BTN_ACOS, True},
  {BTN_POW_E, True}, {BTN_ATAN, True}, {BTN_DOT, True},
  {BTN_NOT, False}, {BTN_XOR, False}, {BTN_OR, False}, {BTN_AND, False},
  {BTN_LSHIFT, False}, {BTN_RSHIFT, False}, {BTN_DEC_INPUT, False},
  {BTN_BIN_INPUT, False}, {BTN_OCT_INPUT, False}, {BTN_HEX_INPUT, False},
  {BTN_EE, True}, {BTN_MOD, True}, {BTN_FLOOR, True}, {BTN_CEIL, True},
  {BTN_INT, True}, {BTN_FRAC, True}, {BTN_SCI, True}, {BTN_NORM, True},
  {BTN_DEG, True}, {BTN_RAD, True}, {BTN_GRA, True}
};
short MathColorButtons[] = {
  BTN_FAC, BTN_POW_X, BTN_SIN, BTN_SQRT, BTN_INV_POW_X, BTN_COS, BTN_SQUARE,
  BTN_LOG, BTN_TAN, BTN_HYP, BTN_LN, BTN_ASIN, BTN_POW_10, BTN_ACOS,
  BTN_POW_E, BTN_ATAN, BTN_MOD, BTN_FLOOR, BTN_CEIL, BTN_INT, BTN_FRAC
};
short ModeColorButtons[] = {
  BTN_BIN_INPUT, BTN_OCT_INPUT, BTN_DEC_INPUT, BTN_HEX_INPUT
};
int MsgCat;
short NumColorButtons[] = {
  BTN_0, BTN_1, BTN_2, BTN_3, BTN_4, BTN_5, BTN_6, BTN_7, BTN_8, BTN_9,
  BTN_DOT, BTN_EE
};
ButtonState OctalButtons[] = {
  {BTN_0, True}, {BTN_1, True}, {BTN_2, True}, {BTN_3, True},
  {BTN_4, True}, {BTN_5, True}, {BTN_6, True}, {BTN_7, True},
  {BTN_8, False}, {BTN_9, False}, {BTN_A, False}, {BTN_B, False},
  {BTN_C, False}, {BTN_D, False}, {BTN_E, False}, {BTN_F, False}
};
short OperColorButtons[] = {
  BTN_PLUS, BTN_MINUS, BTN_TIMES, BTN_SLASH, BTN_LPAREN, BTN_RPAREN
};
char PointSeen;
char PrevOp;
int SaveBase;
int Scale;
struct {
  int Level;
  Accum Data[STACK_SIZE];
} Stack;
Widget TopLevel;
double TrigConvFactor;
char TrigMode;
int UnaryCount;
char UnaryOp[10];

int main(int argc, char *argv[])
{
  setlocale(LC_ALL, "");
  MsgCat = catopen("calc", MCLoadBySet);
  if (argc != 1)
    {
      PutMessage(MsgCat, XMSGG_1Set, XMSGG_1USAGE, "Usage: %s\n",
		 argv[0]);
      exit(1);
    }
  InitCalc();
  SetupMainWindow(&argc, argv);
  exit(0);
}

void AppendDigit(int Num)
{
    /*
     * Append digit to end of current number in working accumulator.
     */
  if ((CalcState == ProcessedEqual) || (CalcState == GetNumOrUnary) || 
      (CalcState == GetNum) || (CalcState == ProcessedImmedOp))
    {
      CurrentAccum.IAccum = 0;
      CurrentAccum.FAccum = 0.0;
    }
  if (Num == BTN_DOT)
    {
      PointSeen = True;
      Scale = Base;
      return;
    }
  if (CurrentAccum.IAccum >= 0)
    {
      if (PointSeen)
	{
	  CurrentAccum.FAccum = CurrentAccum.FAccum + ((double) Num / Scale);
	  Scale = Scale * Base;
	}
      else
	{
	  CurrentAccum.FAccum = (CurrentAccum.FAccum * (double) Base) +
	    (double) Num;
	  CurrentAccum.IAccum = (CurrentAccum.IAccum * Base) + Num;
	}
    }
  else
    {
      if (PointSeen)
	{
	  CurrentAccum.FAccum = CurrentAccum.FAccum - ((double) Num / Scale);
	  Scale = Scale * Base;
	}
      else
	{
	  CurrentAccum.FAccum = (CurrentAccum.FAccum * (double) Base) -
	    (double) Num;
	  CurrentAccum.IAccum = (CurrentAccum.IAccum * Base) - Num;
	}
    }
}

void AppendExponent(int Num)
{
    /*
     * Append current digit to exponent and display new exponent value
     */
  XmString DisplayString;
  char ExpString[6];

  switch (Num)
    {
    case BTN_0:
    case BTN_1:
    case BTN_2:
    case BTN_3:
    case BTN_4:
    case BTN_5:
    case BTN_6:
    case BTN_7:
    case BTN_8:
    case BTN_9:
      if (Exponent < 99)
	{
	  Exponent = Exponent * 10 + Num;
	}
      break;
    case BTN_NEG:
      Exponent = -Exponent;
      break;
    }
  sprintf(ExpString, "E%04d", Exponent);
  DisplayString = XmStringCreate(ExpString, XmSTRING_DEFAULT_CHARSET);
  XtVaSetValues(CalcWidget[DISPLAY_AREA], XmNlabelString, DisplayString, NULL);
  XmStringFree(DisplayString);
}

void ExitProgram(Widget CBWidget, XtPointer CallbackData, 
		 XtPointer WidgetInfo)
{
  exit(0);
}

void InitCalc()
{
    /*
     * Set calculator initial state, display mode, etc.
     */
  memset(Stack.Data, 0, sizeof Stack.Data);
  Stack.Level = 0;
  CurrentAccum.IAccum = 0;
  Base = 10;
  SaveBase = 10;
  PointSeen = False;
  Scale = 1;
  DisplayMode = BTN_NORM;
  CalcState = GetNumOrUnary;
  UnaryOp[0] = -1;
  PrevOp = BTN_PLUS;
  TrigMode = BTN_DEG;
  TrigConvFactor = 0.01745329251994330;
  UnaryCount = 0;
}

int IntegerMultiply(Int P1, Int P2, Int *Product)
{
    /*
     * Perform a 64x64 bit multiply, checking for overflow.
     */
  int HiBit;
  UNum *Op1P;
  int Op1Sign;
  UNum *Op2P;
  int Op2Sign;
  UShort Result[4];
  UNum TempNum;
  UNum TempResult;
    /*
     * Following code performs a 64x64 bit multiply and detects overflow.
     * This is necessary since C provides no way to detect overflow in
     * a multiplication. In order to do the multiplication, each 64 bit
     * integer is treated as 2 consecutive 32 bit integers and the result
     * is treated as 4 consecutive 32 bit integers:
     *
     *                                   |          |          |
     *                              OP1  |    D1    |    D2    |
     *                                   |          |          |
     *                                   +----------+----------+
     * 
     *                                   |          |          |
     *                              OP2  |    D3    |    D4    |
     *                                   |          |          |
     *                                   +----------+----------+
     *
     *           |           |           |          |          |
     *  RESULT   |    R1     |    R2     |    R3    |    R4    |
     *           |           |           |          |          |
     *           +-----------+-----------+----------+----------+
     *
     * The sequence of multiplies and adds is as follows
     * R3:R4 = D2*D4, overflow impossible
     * R2:R3 = D1*D4 + R3, possible 1-bit overflow into R1
     * R2:R3 = D3*D2 + R2:R3, possible 1 bit overflow into R1
     * R1:R2 = D3*D1 + R2, possible overflow is ignored
     *
     * If either R1 or R2 is non-zero, overflow has occurred.
     *
     * All arithmetic is done unsigned, then the proper sign is set at
     * the end of the calculation.
     *  With reversed byte order architectures, like Intel x86, all array
     * subscripts need to be inverted, since low order bytes appear first
     * in storage. Also, because of big endian/little endian byte ordering,
     * this code cannot make assumptions about ovelaying long long
     * integers over pairs of long integers, each half of a long long
     * integer should be processed separately.
     */
    /*
     * Initialize result, save signs, make both operands unsigned
     */
  memset(Result, 0, sizeof Result);
  Op1Sign = (P1 < 0) ? -1 : 1;
  Op2Sign = (P2 < 0) ? -1 : 1;
  Op1P = (UNum *) &P1;
  Op2P = (UNum *) &P2;
  if (P1 < 0)
    {
      P1 = -P1;
    }
  if (P2 < 0)
    {
      P2 = -P2;
    }
    /*
     * R3:R4 = D2*D4
     */
  TempResult.L = (UInt) Op1P->I[D2] * (UInt) Op2P->I[D4];
  Result[R3] = TempResult.I[D1];
  Result[R4] = TempResult.I[D2];
    /*
     * R2:R3 = D1*D4 + R3, possible 1 bit overflow into R1
     */
  TempResult.L = (UInt) Op1P->I[D1] * (UInt) Op2P->I[D4];
  HiBit = TempResult.L >> UINT_SHIFT;
  TempResult.L = TempResult.L + Result[R3];
  if ((TempResult.L >> UINT_SHIFT) != HiBit)
    {
      Result[R1] = Result[R1] + 1;
    }
  Result[R2] = TempResult.I[D1];
  Result[R3] = TempResult.I[D2];
    /*
     * R2:R3 = D3*D2 + R2:R3, possible 1 bit overflow into R1
     */
  TempResult.L = (UInt) Op1P->I[D2] * (UInt) Op2P->I[D3];
  HiBit = TempResult.L >> UINT_SHIFT;
  TempNum.I[D1] = Result[R2];
  TempNum.I[D2] = Result[R3];
  TempResult.L = TempResult.L + TempNum.L;
  if ((TempResult.L >> UINT_SHIFT) != HiBit)
    {
      Result[R1] = Result[R1] + 1;
    }
  Result[R2] = TempResult.I[D1];
  Result[R3] = TempResult.I[D2];
    /*
     * R1:R2 = D3*D1 + R2, possible overflow is ignored
     */
  TempResult.L = (UInt) Op1P->I[D1] * (UInt) Op2P->I[D3];
  TempResult.L = TempResult.L + Result[R2];
  Result[R1] = TempResult.I[D1];
  Result[R2] = TempResult.I[D2];
  if ((DisplayMode == BTN_FIX) && ((Result[R1] != 0) || (Result[R2] != 0)))
    {
      PopupError(MsgCat, XMSGG_1Set, XMSGG_1OVERFLOW, 0,
		 "Integer arithmetic overflow.");
      ProcessInput(BTN_CLEAR_ALL);
      return -1;
    }
  TempResult.I[D1] = Result[R3];
  TempResult.I[D2] = Result[R4];
  if (Op1Sign == Op2Sign)
    {
      *Product = TempResult.L;
    }
  else
    {
      *Product = -TempResult.L;
    }
  return 0;
}

char *NumToA(Int Num, int Base)
{
    /*
     * Format a number in the current number base for display.
     */
  int i;
  int Mask;
  char Negative;
  char NonZeroSeen;
  Int NumIn;
  static char NumOut[40];
  static char NumStr[16] = "0123456789abcdef";
  int Remainder;
  int Shift;
  UInt UNumIn;

  if (Num == 0)
    {
      return "0";
    }
  NonZeroSeen = False;
  i = sizeof NumOut - 1;;
  NumOut[i--] = '\0';
  NumIn = Num;
  if ((NumIn < 0) && (Base == 10))
    {
        /*
	 * Negative decimal numbers are displayed as absolute values with 
	 * leading minus sign.
	 */
      NumIn = -NumIn;
      Negative = True;
    }
  else
    {
      Negative = False;
    }
  if (NumIn >= 0)
    {
        /*
	 * Extract last digit of the number and use that value as index into
	 * the ascii digit lookup table.
	 */
      while ((NumIn != 0) && (i < sizeof NumOut))
	{
	  Remainder = NumIn % Base;
	  NumIn = NumIn / Base;
	  NumOut[i--] = NumStr[Remainder];
	}
    }
  else
    {
        /*
	 * Processing a negative non-decimal number. Extract the proper number
	 * of trailing bits and use that as index to ascii digit lookup table.
	 */
      UNumIn = NumIn;
      switch (Base)
	{
	case 2:
	  Shift = 1;
	  Mask = 0x01;
	  break;
	case 8:
	  Shift = 3;
	  Mask = 0x07;
	  break;
	case 16:
	  Shift = 4;
	  Mask = 0x0f;
	  break;
	}
      while ((UNumIn != 0) && (i < sizeof NumOut))
	{
	  Remainder = UNumIn & Mask;
	  UNumIn = UNumIn >> Shift;
	  NumOut[i--] = NumStr[Remainder];
	}
    }
  if ((Negative == True) && (Base == 10))
    {
      NumOut[i--] = '-';
    }
  return &NumOut[i + 1];
}

void PopupError(nl_catd Cat, int Set, int MsgNum, int Flags, char * Msg, ...)
{
    /*
     * Retrieve the specified error message from the message catalog, format
     * it, and display it in a popup dialog.
     */
  va_list Args;
  static Widget CancelButton;
  static Widget Dialog;
  char MsgOut[512];
  char *MsgP;
  XmString MsgString;

  if (Dialog == NULL)
    {
        /* The dialog doesn't exist yet. Create it, then hide the help button
	 * since there is no help.
	 */
      Dialog = XmCreateMessageDialog(MainWindow, "dialog", NULL, 0);
      XtVaSetValues(XtParent(Dialog), XmNtitle, "Error", NULL);
      XtVaSetValues(Dialog, XmNdialogType, XmDIALOG_MESSAGE, NULL);
      XtUnmanageChild(XtNameToWidget(Dialog, "Help"));
      CancelButton = XtNameToWidget(Dialog, "Cancel");
      XtVaSetValues(XtNameToWidget(Dialog, "OK"), XmNmarginWidth, 5, 
		    XmNmarginHeight, 5, NULL);
      XtVaSetValues(CancelButton, XmNmarginWidth, 5, XmNmarginHeight, 5, NULL);
      XtAddCallback(Dialog, XmNcancelCallback, ExitProgram, NULL);
    }
    /*
     * Show the cancel button only if cancel is an option.
     */
  if ((Flags & ERR_SHOW_CANCEL) == ERR_SHOW_CANCEL)
    {
      XtManageChild(CancelButton);
    }
  else
    {
      XtUnmanageChild(CancelButton);
    }
    /*
     * Format and display the text.
     */
  va_start(Args, Msg);
  MsgP = catgets(Cat, Set, MsgNum, Msg);
  vsprintf(MsgOut, MsgP, Args);
  va_end(Args);
  MsgString = XmStringCreate(MsgOut, XmSTRING_DEFAULT_CHARSET);
  XtVaSetValues(Dialog, XmNmessageString, MsgString, NULL);
  XmStringFree(MsgString);
  XtManageChild(Dialog);
  XtPopup(XtParent(Dialog), XtGrabNone);
}

void ProcessImmedOp(int Request)
{
    /*
     * Process calculator keys which are immediate operations on number just
     * entered. The working accumulator is updated with the new value. Update
     * the display with the new value.
     */
  int i;
  Int ResultI;
  double ResultF;
  if (ProcessUnaryOp() == -1)
    {
      return;
    }
  switch (Request)
    {
    case BTN_SQUARE:
      CurrentAccum.FAccum = CurrentAccum.FAccum * CurrentAccum.FAccum;
      if (IntegerMultiply(CurrentAccum.IAccum, CurrentAccum.IAccum,
			  &CurrentAccum.IAccum) == -1)
	{
	  return;
	}
      break;
    case BTN_FAC:
      if (CurrentAccum.FAccum != CurrentAccum.IAccum)
	{
	  PopupError(MsgCat, XMSGG_1Set, XMSGG_1FACT_NOT_INT, 0,
		     "Can't compute non-integer factorial.");
	  ProcessInput(BTN_CLEAR_ALL);
	  return;
	}
      else
	if (CurrentAccum.IAccum < 1)
	  {
	    PopupError(MsgCat, XMSGG_1Set, XMSGG_1FACT_RANGE, 0,
		       "Can't compute factorial < 1");
	    ProcessInput(BTN_CLEAR_ALL);
	    return;
	  }
      ResultI = 1;
      ResultF = 1.0;
      for (i = 1; i <= CurrentAccum.IAccum; i++)
	{
	  ResultF = ResultF * i;
	  if (IntegerMultiply(ResultI, (Int) i, &ResultI) == -1)
	    {
	      return;
	    }
	}
      CurrentAccum.FAccum = ResultF;
      CurrentAccum.IAccum = ResultI;
      break;
    }
  UpdateDisplay(&CurrentAccum);
}

void ProcessInput(int Request)
{
    /*
     * Process calculator keystrokes, either from mouse clicks or keyboard
     * entry.
     */
  int OpStatus;

  switch (Request)
    {
    case BTN_0:
    case BTN_1:
    case BTN_2:
    case BTN_3:
    case BTN_4:
    case BTN_5:
    case BTN_6:
    case BTN_7:
    case BTN_8:
    case BTN_9:
    case BTN_A:
    case BTN_B:
    case BTN_C:
    case BTN_D:
    case BTN_E:
    case BTN_F:
    case BTN_DOT:
      switch (CalcState)
	{
	case ExponentMode:
	  AppendExponent(Request);
	  break;
	case ProcessedEqual:
	    /*
	     * = key was entered. Clear all intermediate results
	     */
	  memset(Stack.Data, 0, sizeof Stack.Data);
	  Stack.Level = 0;
	  PrevOp = BTN_PLUS;
	case ProcessedImmedOp:
	case GetNumOrUnary:
	case GetNum:
	case ProcessNum:
	  AppendDigit(Request);
	  CalcState = ProcessNum;
	  UpdateDisplay(&CurrentAccum);
	  break;
	}
      break;
    case BTN_NEG:
      if (CalcState == ExponentMode)
	{
	  AppendExponent(Request);
	  break;
	}
    case BTN_NOT:
    case BTN_SIN:
    case BTN_COS:
    case BTN_TAN:
    case BTN_ASIN:
    case BTN_ACOS:
    case BTN_ATAN:
    case BTN_CEIL:
    case BTN_FLOOR:
    case BTN_SQRT:
    case BTN_LOG:
    case BTN_LN:
    case BTN_POW_E:
    case BTN_POW_10:
    case BTN_INT:
    case BTN_FRAC:
    case BTN_HYP:
        /*
	 * Add prefix unary operation to list of operations to be done at the
	 * next calculation
	 */
      if (((CalcState == ProcessedEqual) || (CalcState == GetNumOrUnary)) && 
	  (UnaryCount < 10))
	{
	  UnaryOp[UnaryCount++] = Request;
	}
      Scale = Base;
      PointSeen = False;
      break;
    case BTN_PLUS:
    case BTN_MINUS:
    case BTN_TIMES:
    case BTN_SLASH:
    case BTN_AND:
    case BTN_OR:
    case BTN_XOR:
    case BTN_EQUAL:
    case BTN_MOD:
    case BTN_POW_X:
    case BTN_INV_POW_X:
    case BTN_LSHIFT:
    case BTN_RSHIFT:
        /*
	 * Peform requested calculation using working accumulator and top
	 * level of data stack. This includes processing any pending prefix
	 * operators.
	 */
      if (CalcState == ExponentMode)
	{
	  CurrentAccum.FAccum = CurrentAccum.FAccum * 
	    pow(10.0, (double) Exponent);
	  CurrentAccum.IAccum = CurrentAccum.FAccum;
	  Exponent = 0;
	  CalcState = ProcessNum;
	}
      if (SaveBase != Base)
	{
	  ResetBase();
	}
      if ((CalcState == ProcessedEqual) || (CalcState == ProcessNum) ||
	  (CalcState == ProcessedImmedOp))
	{
	  OpStatus = ProcessPrevOperator();
	  UpdateDisplay(&Stack.Data[Stack.Level]);
	  if (OpStatus == 0)
	    {
	      PrevOp = Request;
	      if (Request == BTN_EQUAL)
		{
		  CalcState = ProcessedEqual;
		}
	      else
		{
		  CurrentAccum.IAccum = Stack.Data[Stack.Level].IAccum;
		  CurrentAccum.FAccum = Stack.Data[Stack.Level].FAccum;
		  CalcState = GetNumOrUnary;
		}
	    }
	  PointSeen = False;
	}
      break;
    case BTN_SQUARE:
    case BTN_FAC:
      ProcessImmedOp(Request);
      CalcState = ProcessedImmedOp;
      break;
    case BTN_CLEAR_ALL:
      memset(Stack.Data, 0, sizeof Stack.Data);
      Stack.Level = 0;
      PrevOp = BTN_PLUS;
    case BTN_CE:
      CurrentAccum.IAccum = 0;
      CurrentAccum.FAccum = 0.0;
      UnaryOp[0] = -1;
      CalcState = GetNumOrUnary;
      UpdateDisplay(&CurrentAccum);
      PointSeen = False;
      UnaryCount = 0;
      ResetBase();
      break;
    case BTN_DEC_INPUT:
      SetTempBase(10, DecimalButtons);
      break;
    case BTN_HEX_INPUT:
      SetTempBase(16, HexButtons);
      break;
    case BTN_OCT_INPUT:
      SetTempBase(8, OctalButtons);
      break;
    case BTN_BIN_INPUT:
      SetTempBase(2, BinaryButtons);
      break;
    case BTN_EE:
      CalcState = ExponentMode;
      Exponent = 0;
      break;
    case BTN_QUIT:
      exit(0);
    }
}

int ProcessPrevOperator()
{
    /*
     * Perform the requested calculations, including processing any prefix ops
     */
  int Op1Sign;
  int Op2Sign;
  int ResultSign;
  
  if (ProcessUnaryOp() == -1)
    {
      return -1;
    }
  switch (PrevOp)
    {
    case BTN_PLUS:
        /*
	 * Add two numbers, checking for overflow. Overflow occurs if both
	 * input numbers have the same sign and the result sign is different
	 * from the input sign.
	 */
      Op1Sign = (Stack.Data[Stack.Level].IAccum < 0) ? -1 : 1;
      Op2Sign = (CurrentAccum.IAccum < 0) ? -1 : 1;
      Stack.Data[Stack.Level].IAccum = Stack.Data[Stack.Level].IAccum +
	CurrentAccum.IAccum;
      Stack.Data[Stack.Level].FAccum = Stack.Data[Stack.Level].FAccum +
	CurrentAccum.FAccum;
      if ((DisplayMode == BTN_FIX) && (Op1Sign == Op2Sign))
	{
	  ResultSign = (Stack.Data[Stack.Level].IAccum < 0) ? -1 : 1;
	  if (ResultSign != Op1Sign)
	    {
	      PopupError(MsgCat, XMSGG_1Set, XMSGG_1OVERFLOW, 0,
			 "Integer arithmetic overflow.");
	      ProcessInput(BTN_CLEAR_ALL);
	      return -1;
	    }
	}
      break;
    case BTN_MINUS:
        /*
	 * Subtract two numbers, checking for overflow. Overflow occurs if the
	 * inputs have different signs, and the sign of the result is not the
	 * same as the sign of the first input.
	 */
      Op1Sign = (Stack.Data[Stack.Level].IAccum < 0) ? -1 : 1;
      Op2Sign = (CurrentAccum.IAccum < 0) ? -1 : 1;
      Stack.Data[Stack.Level].IAccum = Stack.Data[Stack.Level].IAccum -
	CurrentAccum.IAccum;
      Stack.Data[Stack.Level].FAccum = Stack.Data[Stack.Level].FAccum -
	CurrentAccum.FAccum;
      if ((DisplayMode == BTN_FIX) && (Op1Sign != Op2Sign))
	{
	  ResultSign = (Stack.Data[Stack.Level].IAccum < 0) ? -1 : 1;
	  if (ResultSign != Op1Sign)
	    {
	      PopupError(MsgCat, XMSGG_1Set, XMSGG_1OVERFLOW, 0,
			 "Integer arithmetic overflow.");
	      ProcessInput(BTN_CLEAR_ALL);
	      return -1;
	    }
	}
      break;
    case BTN_TIMES:
      Stack.Data[Stack.Level].FAccum = Stack.Data[Stack.Level].FAccum *
	CurrentAccum.FAccum;
      if (IntegerMultiply(Stack.Data[Stack.Level].IAccum, CurrentAccum.IAccum,
			  &Stack.Data[Stack.Level].IAccum) == -1)
	{
	  return -1;
	}
      break;
    case BTN_SLASH:
      if (CurrentAccum.FAccum == 0.0)
	{
	  PopupError(MsgCat, XMSGG_1Set, XMSGG_1ZERODIVIDE, 0,
		     "Division by zero.");
	  ProcessInput(BTN_CLEAR_ALL);
	  return -1;
	}
      if (CurrentAccum.IAccum == 0)
	{
	  Stack.Data[Stack.Level].IAccum = 0;
	}
      else
	{
	  Stack.Data[Stack.Level].IAccum = Stack.Data[Stack.Level].IAccum /
	    CurrentAccum.IAccum;
	}
      Stack.Data[Stack.Level].FAccum = Stack.Data[Stack.Level].FAccum /
	CurrentAccum.FAccum;
      break;
    case BTN_OR:
      Stack.Data[Stack.Level].IAccum = Stack.Data[Stack.Level].IAccum |
	CurrentAccum.IAccum;
      Stack.Data[Stack.Level].FAccum = Stack.Data[Stack.Level].IAccum;
      break;
    case BTN_AND:
      Stack.Data[Stack.Level].IAccum = Stack.Data[Stack.Level].IAccum &
	CurrentAccum.IAccum;
      Stack.Data[Stack.Level].FAccum = Stack.Data[Stack.Level].IAccum;
      break;
    case BTN_XOR:
      Stack.Data[Stack.Level].IAccum = Stack.Data[Stack.Level].IAccum ^
	CurrentAccum.IAccum;
      Stack.Data[Stack.Level].FAccum = Stack.Data[Stack.Level].IAccum;
      break;
    case BTN_MOD:
      Stack.Data[Stack.Level].IAccum = Stack.Data[Stack.Level].IAccum %
	CurrentAccum.IAccum;
      Stack.Data[Stack.Level].FAccum = fmod(Stack.Data[Stack.Level].FAccum,
					    CurrentAccum.FAccum);
      break;
    case BTN_POW_X:
      Stack.Data[Stack.Level].FAccum = pow(Stack.Data[Stack.Level].FAccum,
					   CurrentAccum.FAccum);
      Stack.Data[Stack.Level].IAccum = Stack.Data[Stack.Level].FAccum;
      break;
    case BTN_INV_POW_X:
      Stack.Data[Stack.Level].FAccum = pow(Stack.Data[Stack.Level].FAccum,
					   -CurrentAccum.FAccum);
      Stack.Data[Stack.Level].IAccum = Stack.Data[Stack.Level].FAccum;
      break;
    case BTN_LSHIFT:
      Stack.Data[Stack.Level].IAccum = Stack.Data[Stack.Level].IAccum <<
	CurrentAccum.IAccum;
      Stack.Data[Stack.Level].FAccum = Stack.Data[Stack.Level].IAccum;
      break;
    case BTN_RSHIFT:
      Stack.Data[Stack.Level].IAccum = Stack.Data[Stack.Level].IAccum >>
	CurrentAccum.IAccum;
      Stack.Data[Stack.Level].FAccum = Stack.Data[Stack.Level].IAccum;
      break;
    case BTN_EQUAL:
      break;
    }
  return 0;
}

int ProcessUnaryOp()
{
    /*
     * Process prefix operators in reverse order from their entry. Each
     * operation updates the value in the working accumulator. Hyperbolic
     * trig funcions (sinh, etc) are a special case, since there are two 
     * operators combined into a single operator, the hyp key and the following
     * trig function key.
     */
  int i;
  double IntResult;

  for (i = UnaryCount - 1; i >= 0; i--)
    {
      switch (UnaryOp[i])
	{
	case BTN_NEG:
	  CurrentAccum.IAccum = -CurrentAccum.IAccum;
	  CurrentAccum.FAccum = -CurrentAccum.FAccum;
	  break;
	case BTN_NOT:
	  CurrentAccum.IAccum = ~CurrentAccum.IAccum;
	  CurrentAccum.FAccum = CurrentAccum.IAccum;
	  break;
	case BTN_COS:
	  if ((i > 0) && (UnaryOp[i - 1] == BTN_HYP))
	    {
	      CurrentAccum.FAccum = cosh(CurrentAccum.FAccum * TrigConvFactor);
	      CurrentAccum.IAccum = CurrentAccum.FAccum;
	      i = i - 1;
	    }
	  else
	    {
	      CurrentAccum.FAccum = cos(CurrentAccum.FAccum * TrigConvFactor);
	      CurrentAccum.IAccum = 0;
	    }
	  break;
	case BTN_SIN:
	  if ((i > 0) && (UnaryOp[i - 1] == BTN_HYP))
	    {
	      CurrentAccum.FAccum = sinh(CurrentAccum.FAccum * TrigConvFactor);
	      CurrentAccum.IAccum = CurrentAccum.FAccum;
	      i = i - 1;
	    }
	  else
	    {
	      CurrentAccum.FAccum = sin(CurrentAccum.FAccum * TrigConvFactor);
	      CurrentAccum.IAccum = 0;
	    }
	  break;
	case BTN_TAN:
	  if ((i > 0) && (UnaryOp[i - 1] == BTN_HYP))
	    {
	      CurrentAccum.FAccum = tanh(CurrentAccum.FAccum * TrigConvFactor);
	      CurrentAccum.IAccum = CurrentAccum.FAccum;
	      i = i - 1;
	    }
	  else
	    {
	      CurrentAccum.FAccum = tan(CurrentAccum.FAccum * TrigConvFactor);
	      CurrentAccum.IAccum = CurrentAccum.FAccum;
	    }
	  break;
	case BTN_ACOS:
	  if ((i > 0) && (UnaryOp[i - 1] == BTN_HYP))
	    {
	      CurrentAccum.FAccum = acosh(CurrentAccum.FAccum * 
					  TrigConvFactor);
	      CurrentAccum.IAccum = CurrentAccum.FAccum;
	      i = i - 1;
	    }
	  else
	    {
	      CurrentAccum.FAccum = acos(CurrentAccum.FAccum) / TrigConvFactor;
	      CurrentAccum.IAccum = CurrentAccum.FAccum;
	    }
	  break;
	case BTN_ASIN:
	  if ((i > 0) && (UnaryOp[i - 1] == BTN_HYP))
	    {
	      CurrentAccum.FAccum = asinh(CurrentAccum.FAccum * 
					  TrigConvFactor);
	      CurrentAccum.IAccum = CurrentAccum.FAccum;
	      i = i - 1;
	    }
	  else
	    {
	      CurrentAccum.FAccum = asin(CurrentAccum.FAccum) / TrigConvFactor;
	      CurrentAccum.IAccum = CurrentAccum.FAccum;
	    }
	  break;
	case BTN_ATAN:
	  if ((i > 0) && (UnaryOp[i - 1] == BTN_HYP))
	    {
	      CurrentAccum.FAccum = atanh(CurrentAccum.FAccum * 
					  TrigConvFactor);
	      CurrentAccum.IAccum = CurrentAccum.FAccum;
	      i = i - 1;
	    }
	  else
	    {
	      CurrentAccum.FAccum = atan(CurrentAccum.FAccum) / TrigConvFactor;
	      CurrentAccum.IAccum = CurrentAccum.FAccum;
	    }
	  break;
	case BTN_CEIL:
	  CurrentAccum.FAccum = ceil(CurrentAccum.FAccum);
	  CurrentAccum.IAccum = CurrentAccum.FAccum;
	  break;
	case BTN_FLOOR:
	  CurrentAccum.FAccum = floor(CurrentAccum.FAccum);
	  CurrentAccum.IAccum = CurrentAccum.FAccum;
	  break;
	case BTN_INT:
	  modf(CurrentAccum.FAccum, &IntResult);
	  CurrentAccum.FAccum = IntResult;
	  CurrentAccum.IAccum = IntResult;
	  break;
	case BTN_FRAC:
	  CurrentAccum.FAccum = modf(CurrentAccum.FAccum, &IntResult);
	  CurrentAccum.IAccum = 0;
	  break;
	case BTN_SQRT:
	  if (CurrentAccum.FAccum < 0)
	    {
	      PopupError(MsgCat, XMSGG_1Set, XMSGG_1NEG_SQRT, 0,
			 "Square root of negative number.");
	      ProcessInput(BTN_CLEAR_ALL);
	      return -1;
	    }
	  else
	    {
	      CurrentAccum.FAccum = sqrt(CurrentAccum.FAccum);
	    }
	  CurrentAccum.IAccum = CurrentAccum.FAccum;
	  break;
	case BTN_LOG:
	  if (CurrentAccum.FAccum < 0)
	    {
	      PopupError(MsgCat, XMSGG_1Set, XMSGG_1NEG_LOG, 0,
			 "Log of negative number.");
	      ProcessInput(BTN_CLEAR_ALL);
	      return -1;
	    }
	  else
	    if (CurrentAccum.FAccum == 0)
	      {
		PopupError(MsgCat, XMSGG_1Set, XMSGG_1LOG_0, 0,
			   "Log of zero.");
		ProcessInput(BTN_CLEAR_ALL);
		return -1;
	      }
	    else
	      {
		CurrentAccum.FAccum = log10(CurrentAccum.FAccum);
	      }
	  CurrentAccum.IAccum = CurrentAccum.FAccum;
	  break;
	case BTN_LN:
	  if (CurrentAccum.FAccum < 0)
	    {
	      PopupError(MsgCat, XMSGG_1Set, XMSGG_1NEG_LOG, 0,
			 "Log of negative number.");
	      ProcessInput(BTN_CLEAR_ALL);
	      return -1;
	    }
	  else
	    if (CurrentAccum.FAccum == 0)
	      {
		PopupError(MsgCat, XMSGG_1Set, XMSGG_1LOG_0, 0,
			   "Log of zero.");
		ProcessInput(BTN_CLEAR_ALL);
		return -1;
	      }
	    else
	      {
		CurrentAccum.FAccum = log(CurrentAccum.FAccum);
	      }
	  CurrentAccum.IAccum = CurrentAccum.FAccum;
	  break;
	case BTN_POW_E:
	  if (CurrentAccum.FAccum < 0)
	    {
	      PopupError(MsgCat, XMSGG_1Set, XMSGG_1NEG_POW, 0,
			 "Power of negative number.");
	      ProcessInput(BTN_CLEAR_ALL);
	      return -1;
	    }
	  else
	    {
	      CurrentAccum.FAccum = exp(CurrentAccum.FAccum);
	    }
	  CurrentAccum.IAccum = CurrentAccum.FAccum;
	  break;
	case BTN_POW_10:
	  if (CurrentAccum.FAccum < 0)
	    {
	      PopupError(MsgCat, XMSGG_1Set, XMSGG_1NEG_POW, 0,
			 "Power of negative number.");
	      ProcessInput(BTN_CLEAR_ALL);
	      return -1;
	    }
	  else
	    {
	      CurrentAccum.FAccum = 
		exp(CurrentAccum.FAccum * 2.30258509299404590);
	    }
	  CurrentAccum.IAccum = CurrentAccum.FAccum;
	  break;
	}
    }
  UnaryOp[0] = -1;
  UnaryCount = 0;
  return 0;
}

void PutMessage(nl_catd Cat, int Set, int Msg, char *DefMsg, ...)
{
    /*
     * Display a message to the invoking terminal window. Used only before the
     * calculator display is initialized.
     */
  va_list Args;
  char *MsgP;

  va_start(Args, DefMsg);
  MsgP = catgets(Cat, Set, Msg, DefMsg);
  vfprintf(stderr, MsgP, Args);
  va_end(Args);
}

void SetupMainWindow(int *ArgCount, char *Args[])
{
    /*
     * Create the main dialog for this program. The dialog definition is in a
     * uid file, so load that file, then setup the necessary callbacks and
     * load the widgets. Once the dialog has been initialized, start the event
     * processing loop.
     */
  MrmHierarchy AppHierarchy;
  String AppResource[] = {"calc"};
  int i;
  int Stat;
  int WindowHeight;

  XtSetLanguageProc(NULL, NULL, NULL);
  MrmInitialize();
  TopLevel = XtVaAppInitialize(&AppContext, "calc", NULL, 0, ArgCount,
			       Args, NULL, NULL);
  XtAppAddActions(AppContext, Actions, XtNumber(Actions));
  XtVaSetValues(TopLevel, 
		XmNmwmDecorations, MWM_DECOR_BORDER | MWM_DECOR_TITLE | 
		  MWM_DECOR_MENU,
		XmNmwmFunctions, MWM_FUNC_MOVE | MWM_FUNC_CLOSE,
		NULL);

  Stat = MrmOpenHierarchyPerDisplay(XtDisplay(TopLevel), 
				      XtNumber(AppResource), AppResource,
				      NULL, &AppHierarchy);
  if (Stat != MrmSUCCESS)
    {
      PutMessage(MsgCat, XMSGG_1Set, XMSGG_1UID_OPEN_ERROR,
		 "Can't open application resource file.\n");
      exit(1);
    }
  Stat = MrmRegisterNames(Callbacks, XtNumber(Callbacks));
  if (Stat != MrmSUCCESS)
    {
      PutMessage(MsgCat, XMSGG_1Set, XMSGG_1CALLBACK_ERROR,
		 "Can't setup application callbacks.\n");
      exit(1);
    }
  Stat = MrmFetchWidget(AppHierarchy, "Main", TopLevel, &MainWindow,
			  &MainWindowClass);
  if (Stat != MrmSUCCESS)
    {
      PutMessage(MsgCat, XMSGG_1Set, XMSGG_1WIDGET_FETCH_ERROR,
		 "Can't get main window widget.\n");
      exit(1);
    }
  XtManageChild(MainWindow);
    /*
     * Set the proper calculator mode buttons to their initial state.
     */
  XmToggleButtonSetState(CalcWidget[BTN_DEC], True, True);
  XmToggleButtonSetState(CalcWidget[BTN_DEG], True, True);
  XmToggleButtonSetState(CalcWidget[BTN_NORM], True, True);
    /*
     * Set up the colors for the calculator buttons. Special resources are
     * defined for defining colors of calculator button groups, rather than
     * assigning colors to buttons individually. Retrieve the color resource
     * values and assign them to the buttons in that button class.
     */
  XtVaGetApplicationResources(TopLevel, &Colors, ColorResources, 
			      XtNumber(ColorResources), NULL);
  for (i = 0; i < (sizeof CtlColorButtons / sizeof(short)); i++)
    {
      XtVaSetValues(CalcWidget[CtlColorButtons[i]], 
		    XmNforeground, Colors.CtlColor, NULL);
    }
  for (i = 0; i < (sizeof LogicColorButtons / sizeof(short)); i++)
    {
      XtVaSetValues(CalcWidget[LogicColorButtons[i]], 
		    XmNforeground, Colors.LogicColor, NULL);
    }
  for (i = 0; i < (sizeof NumColorButtons / sizeof(short)); i++)
    {
      XtVaSetValues(CalcWidget[NumColorButtons[i]], 
		    XmNforeground, Colors.NumColor, NULL);
    }
  for (i = 0; i < (sizeof MathColorButtons / sizeof(short)); i++)
    {
      XtVaSetValues(CalcWidget[MathColorButtons[i]], 
		    XmNforeground, Colors.MathColor, NULL);
    }
  for (i = 0; i < (sizeof ModeColorButtons / sizeof(short)); i++)
    {
      XtVaSetValues(CalcWidget[ModeColorButtons[i]], 
		    XmNforeground, Colors.ModeColor, NULL);
    }
  for (i = 0; i < (sizeof OperColorButtons / sizeof(short)); i++)
    {
      XtVaSetValues(CalcWidget[OperColorButtons[i]], 
		    XmNforeground, Colors.OperColor, NULL);
    }
  XtRealizeWidget(TopLevel);
  XtVaGetValues(CalcWidget[DISPLAY_WINDOW], XtNheight, &WindowHeight, NULL);
  XtVaSetValues(CalcWidget[DISPLAY_WINDOW], XmNpaneMinimum, WindowHeight,
		XmNpaneMaximum, WindowHeight, NULL);
  XtAppMainLoop(AppContext);
}

void RegisterWidget(Widget CBWidget, XtPointer CallbackData, 
		    XtPointer WidgetInfo)
{
  CalcWidget[*(int *) CallbackData] = CBWidget;
}

void ResetBase()
{
    /*
     * Set calculator number base back to it's permanent base, reset the
     * state of the calculator buttons, and update the display area
     */
  int i;
  ButtonState *P;

  switch (SaveBase)
    {
    case 2:
      P = BinaryButtons;
      break;
    case 8:
      P = OctalButtons;
      break;
    case 10:
      P = DecimalButtons;
      break;
    case 16:
      P = HexButtons;
      break;
    }
  Base = SaveBase;
  for (i = 0; i < (sizeof BinaryButtons / sizeof(ButtonState)); i++)
    {
      XtSetSensitive(CalcWidget[P[i].Id], P[i].State);
    }
  for (i = 0; i < sizeof ModeColorButtons / sizeof(short); i++)
    {
      XtSetSensitive(CalcWidget[ModeColorButtons[i]], True);
    }
  UpdateDisplay(&CurrentAccum);
}

void SelectWidget(Widget CBWidget, XtPointer CallbackData, 
		  XtPointer WidgetInfo)
{
    /*
     * Process a button click from a calculator pushbutton widget.
     */
  ProcessInput(*(int *) CallbackData);
}

void SetBase(Widget CBWidget, XtPointer CallbackData, XtPointer WidgetInfo)
{
    /*
     * Set the number base based on the button selected from the base radio
     * button set. Depending on number base, enable the pushbuttons that are
     * valid for that base then update the display window.
     */
  int i;
  ButtonState *P;
  ButtonState *P2;
  XmString TempString;

  if (((XmToggleButtonCallbackStruct *) WidgetInfo)->set == XmSET)
    {
      Base = *(int *) CallbackData;
      TempString = XmStringCreate(NumToA(CurrentAccum.IAccum, Base),
				  XmSTRING_DEFAULT_CHARSET);
      XtVaSetValues(CalcWidget[DISPLAY_AREA], XmNlabelString, TempString, 
		    NULL);
      XmStringFree(TempString);
      switch (Base)
	{
	case 2:
	  P = BinaryButtons;
	  P2 = LogicButtons;
	  XmToggleButtonSetState(CalcWidget[BTN_FIX], True, True);
	  break;
	case 8:
	  P = OctalButtons;
	  P2 = LogicButtons;
	  XmToggleButtonSetState(CalcWidget[BTN_FIX], True, True);
	  break;
	case 10:
	  P = DecimalButtons;
	  P2 = MathButtons;
	  XmToggleButtonSetState(CalcWidget[BTN_NORM], True, True);
	  break;
	case 16:
	  P = HexButtons;
	  P2 = LogicButtons;
	  XmToggleButtonSetState(CalcWidget[BTN_FIX], True, True);
	  break;
	}
      for (i = 0; i < (sizeof BinaryButtons / sizeof(ButtonState)); i++)
	{
	  XtSetSensitive(CalcWidget[P[i].Id], P[i].State);
	}
      for (i = 0; i < (sizeof MathButtons / sizeof(ButtonState)); i++)
	{
	  XtSetSensitive(CalcWidget[P2[i].Id], P2[i].State);
	}
    }
  SaveBase = Base;
  UpdateDisplay(&Stack.Data[Stack.Level]);
}

void SetTempBase(int NewBase, ButtonState *ButtonP)
{
    /*
     * Set a temporary number base for the number currenly being input. Once
     * an operator is selected, the number base reverts back to it's permanent
     * value. Enable the proper pushbuttons for the number base, and update
     * the display.
     */
  int i;

  SaveBase = Base;
  Base = NewBase;
  for (i = 0; i < (sizeof BinaryButtons / sizeof(ButtonState)); i++)
    {
      XtSetSensitive(CalcWidget[ButtonP[i].Id], ButtonP[i].State);
    }
  for (i = 0; i < sizeof ModeColorButtons / sizeof(short); i++)
    {
      XtSetSensitive(CalcWidget[ModeColorButtons[i]], False);
    }
  UpdateDisplay(&CurrentAccum);
}

void SetDisplayMode(Widget CBWidget, XtPointer CallbackData,
		    XtPointer WidgetInfo)
{
  if (((XmToggleButtonCallbackStruct *) WidgetInfo)->set == XmSET)
    {
      switch (*(int *) CallbackData)
	{
	case BTN_SCI:
	  XtSetSensitive(CalcWidget[BTN_EE], True);
	  break;
	case BTN_FIX:
	  XtSetSensitive(CalcWidget[BTN_EE], False);
	  break;
	case BTN_NORM:
	  XtSetSensitive(CalcWidget[BTN_EE], True);
	  break;
	}
      DisplayMode = *(int *) CallbackData;
    }
  UpdateDisplay(&Stack.Data[Stack.Level]);
}

void SetTrigMode(Widget CBWidget, XtPointer CallbackData,
		 XtPointer WidgetInfo)
{
    /*
     * Set the proper input conversion factor, depending on which trig mode
     * radio button (degrees, radians, or grads) is selected.
     */
  if (((XmToggleButtonCallbackStruct *) WidgetInfo)->set == XmSET)
    {
      switch (*(int *) CallbackData)
	{
	case BTN_DEG:
	  TrigConvFactor = 0.01745329251994330;
	  break;
	case BTN_RAD:
	  TrigConvFactor = 1.0;
	  break;
	case BTN_GRA:
	  TrigConvFactor = 0.01570796326794897;
	  break;
	}
      TrigMode = *(int *) CallbackData;
   }
}

void UpdateDisplay(Accum *Acc)
{
    /*
     * Update the value in the calculator display window depending on current
     * display mode.
     */
  char NumOut[40];
  XmString TempString;

  if ((DisplayMode != BTN_FIX) && 
      ((isinf(Acc->FAccum) != 0) || (isnan(Acc->FAccum) != 0)))
    {
      PopupError(MsgCat, XMSGG_1Set, XMSGG_1FLOAT_OVERFLOW, 0,
		 "Floating point overflow.");
      ProcessInput(BTN_CLEAR_ALL);
    }
  switch (DisplayMode)
    {
    case BTN_FIX:
      TempString = XmStringCreate(NumToA(Acc->IAccum, Base), 
				  XmSTRING_DEFAULT_CHARSET);
      break;
    case BTN_SCI:
      sprintf(NumOut, "%17.15E", Acc->FAccum);
      TempString = XmStringCreate(NumOut, XmSTRING_DEFAULT_CHARSET);
      break;
    case BTN_NORM:
      sprintf(NumOut, "%27.15g", Acc->FAccum);
      TempString = XmStringCreate(NumOut, XmSTRING_DEFAULT_CHARSET);
      break;
    }
  XtVaSetValues(CalcWidget[DISPLAY_AREA], XmNlabelString, TempString, NULL);
  XmStringFree(TempString);
}

void KeyStroke(Widget SrcWidget, XEvent *EventCode, String Input,
	      Cardinal Count)
{
    /*
     * Process keyboard input from the calculator. For selected pushbuttons,
     * equivalent keystrokes are accepted. Map those keystrokes to the
     * proper request code then perform the requested action if it is valid
     * in the current number base.
     */
  KeySym KeyID;
  char KeyString[4];

  XLookupString(&(EventCode->xkey), KeyString, sizeof KeyString, &KeyID,
		NULL);
  switch (KeyID)
    {
    case XK_0:
    case XK_KP_0:
      ProcessInput(BTN_0);
      break;
    case XK_1:
    case XK_KP_1:
      ProcessInput(BTN_1);
      break;
    case XK_2:
    case XK_KP_2:
      if (Base > 2)
	{
	  ProcessInput(BTN_2);
	}
      break;
    case XK_3:
    case XK_KP_3:
      if (Base > 2)
	{
	  ProcessInput(BTN_3);
	}
      break;
    case XK_4:
    case XK_KP_4:
      if (Base > 2)
	{
	  ProcessInput(BTN_4);
	}
      break;
    case XK_5:
    case XK_KP_5:
      if (Base > 2)
	{
	  ProcessInput(BTN_5);
	}
      break;
    case XK_6:
    case XK_KP_6:
      if (Base > 2)
	{
	  ProcessInput(BTN_6);
	}
      break;
    case XK_7:
    case XK_KP_7:
      if (Base > 2)
	{
	  ProcessInput(BTN_7);
	}
      break;
    case XK_8:
    case XK_KP_8:
      if (Base > 8)
	{
	  ProcessInput(BTN_8);
	}
      break;
    case XK_9:
    case XK_KP_9:
      if (Base > 8)
	{
	  ProcessInput(BTN_9);
	}
      break;
    case XK_A:
    case XK_a:
      if (Base > 10)
	{
	  ProcessInput(BTN_A);
	}
      break;
    case XK_B:
    case XK_b:
      if (Base > 10)
	{
	  ProcessInput(BTN_B);
	}
      break;
    case XK_C:
    case XK_c:
      if (Base > 10)
	{
	  ProcessInput(BTN_C);
	}
      break;
    case XK_D:
    case XK_d:
      if (Base > 10)
	{
	  ProcessInput(BTN_D);
	}
      break;
    case XK_E:
    case XK_e:
      if (Base > 10)
	{
	  ProcessInput(BTN_E);
	}
      break;
    case XK_F:
    case XK_f:
      if (Base > 10)
	{
	  ProcessInput(BTN_F);
	}
      break;
    case XK_K:
    case XK_k:
      ProcessInput(BTN_CLEAR_ALL);
      break;
    case XK_Q:
    case XK_q:
      ProcessInput(BTN_QUIT);
      break;
    case XK_plus:
    case XK_KP_Add:
      ProcessInput(BTN_PLUS);
      break;
    case XK_minus:
    case XK_KP_Subtract:
      ProcessInput(BTN_MINUS);
      break;
    case XK_asterisk:
    case XK_KP_Multiply:
      ProcessInput(BTN_TIMES);
      break;
    case XK_slash:
    case XK_KP_Divide:
      ProcessInput(BTN_SLASH);
      break;
    case XK_period:
    case XK_KP_Decimal:
      if (DisplayMode != BTN_FIX)
	{
	  ProcessInput(BTN_DOT);
	}
      break;
    case XK_equal:
      ProcessInput(BTN_EQUAL);
      break;
    case XK_ampersand:
      if (Base != 10)
	{
	  ProcessInput(BTN_AND);
	}
      break;
    case XK_exclam:
      if (Base != 10)
	{
	  ProcessInput(BTN_NOT);
	}
      break;
    case XK_less:
      if (Base != 10)
	{
	  ProcessInput(BTN_LSHIFT);
	}
      break;
    case XK_greater:
      if (Base != 10)
	{
	  ProcessInput(BTN_RSHIFT);
	}
      break;
    case XK_asciicircum:
      if (Base != 10)
	{
	  ProcessInput(BTN_XOR);
	}
      break;
    case XK_bar:
      if (Base != 10)
	{
	  ProcessInput(BTN_OR);
	}
      break;
    }
}
