/*

Copyright (C) 1998  Paul Wilkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
/* lcd.c   by Paul Wilkins 3/22/97 */

#include <stdio.h>
#include <Xm/Xm.h>
#include <Xm/DrawingA.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>

#include "lcd.h"
#include "mode.h"
#include "number.h"
#include "stack.h"
#include "process_input.h"
#include "editor.h"
#include "error.h"

/* the ideal starting values */
#define LCD_WIDTH 400
#define LCD_HEIGHT 200

/* the amount of spacing we want around the display */
#define BORDER 2 

Widget lcdDA = 0;
char **lcdText = NULL;
GC lcdGC = 0;
GC lcdHighlightGC = 0;

/* the last area highlighted */
int dataSelected = 0;
int hiX0, hiX1, hiY1;
int hiY0, hiX2, hiY2;

int fontW = 0;
int fontH = 0;
int fontD = 0;
int lcdWidth = 0;
int lcdHeight = 0;
int lcdDisplayMode = LONG_DISPLAY;

/* some callbacks */
void lcdExposeCB(Widget, XtPointer, XtPointer);
void lcdResizeCB(Widget, XtPointer, XtPointer);
void lcdInputCB(Widget, XtPointer, XtPointer);
void lcdKeyPressEvnt(Widget, XtPointer, XEvent *, char *);

/* local functions */
void calcStackLCD();
void drawStackLCD();

/* this stuff for cut & paste */
#define START_SELECTING 0x1
#define NONE_SELECTED 0x2
#define DATA_SELECTED 0x4
void lcdButtonEvnt(Widget, XtPointer, XEvent *, char *);
void lcdMotionEvnt(Widget, XtPointer, XEvent *, char *);
void lcdPasteCB(Widget, XtPointer, Atom *, Atom *, XtPointer, unsigned long *, int *);
void lcdGetSelection(XEvent *evnt);
void lcdStartHighlight(XEvent *evnt);
void lcdStopHighlight(XEvent *evnt);
void lcdContinueHighlight(int x, int y);




Widget setupLCD(Widget parent){

   /* get the drawing area widget */
   lcdDA = XtVaCreateManagedWidget("lcdDA", 
      xmDrawingAreaWidgetClass, parent, 
      XmNwidth, LCD_WIDTH, 
      XmNheight, LCD_HEIGHT, 
      NULL);

   /* add the callbacks */
   XtAddCallback(lcdDA, XmNexposeCallback, lcdExposeCB, NULL);
   XtAddCallback(lcdDA, XmNresizeCallback, lcdResizeCB, NULL);

   /* Annoying... Motif "steals" our <enter> keypress events, so
    * we can't use the drawing area's input callback.  Luckly
    * we can still use XtAddEventHandler to do the same thing.
    * XtAddCallback(lcdDA, XmNinputCallback, lcdInputCB, NULL);
    */
   XtAddEventHandler(lcdDA, KeyPressMask, False, lcdKeyPressEvnt, NULL);

   /* this stuff for cut & paste */
   XtAddEventHandler(lcdDA, ButtonPressMask, False, lcdButtonEvnt, NULL);
   XtAddEventHandler(lcdDA, ButtonReleaseMask, False, lcdButtonEvnt, NULL);
   XtAddEventHandler(lcdDA, ButtonMotionMask, False, lcdMotionEvnt, NULL);

   return lcdDA;
}

void clearLCD(){

   /* clear the stack */
   clearStack();

   /* reset any error string */
   resetError();

   /* clear the entry buffer */
   cancelEditor();

   /* redraw the window */
   calcStackLCD();
   drawStackLCD();
}

/* Recalculate the contents of the lcd display. 
 * note: we don't redraw the lcd here, that is done by drawStackLCD().
 */
void calcStackLCD(){
   int i, j;
   int strLen, pLen, curPos;
   char *c, *p;
   char *txt;     /* the number */
   char label[16];  /* the stack number label */
   int height;
   int indx;
   int row;       /* the row to start print the current number */
   int txtPos;    /* which col do we start printing in */
   int lines;     /* the number of lines this number has */
   int top;       /* which row is the top for drawing the stack */
   int bottom;    /* which row is the current bottom for drawing the stack */
   int labelLen;  /* the stack number label length */
   Number *num;

   if(isEditingEditor()){
      height = lcdHeight - 1;
   } else {
      height = lcdHeight;
   }

   if(height <= 0) return;

   /* clear the old display */
   for(i=0; i<lcdHeight; i++){
      for(j=0; j<lcdWidth; j++){
         lcdText[i][j] = ' ';
      }
   }

   /* check for an error */
   if(isError()){
      top = 1;
      c = getStringError();
      strLen = strlen(c);
      if(strLen > lcdWidth) strLen = lcdWidth;
      strncpy(lcdText[0], c, strLen);
   } else {
      top = 0;
   }
   bottom = height;

   /* print all of the numbers */
   for(indx=0; indx<stackLen(); indx++){

      /* get the number to print */
      if(NULL == (num = getStackEle(indx))) break;

      /* get the text of the number */
      if(indx == 0 || lcdDisplayMode == LONG_DISPLAY){
	 if(NULL == (txt = printNumber(num))) break;
      } else {
	 if(NULL == (txt = printNumberShort(num))) break;
      }
      
      /* how many lines is this number? */
      lines = 1;
      for(p=txt; *p!='\0'; p++) if(*p=='\n'){ lines++; *p='\0'; }

      /* find the starting line of the number */
      row = bottom - lines;
      if(row < top) row = top;

      /* print the stack number label */
      sprintf(label, "%d: ", indx+1);
      labelLen = strlen(label);
      if(labelLen > lcdWidth) labelLen = lcdWidth;
      strncpy(lcdText[row], label, labelLen);

      /* print the number */
      p = txt;         
      for(j=0, i=row; i<bottom && j<lines; i++, j++){
	 pLen = strlen(p);
	 txtPos = lcdWidth - pLen;
	 if(txtPos < labelLen) txtPos = labelLen;
         if(txtPos > lcdWidth) break;

	 strncpy(&(lcdText[i][txtPos]), p, lcdWidth-txtPos);
	 p = p + pLen + 1;
      }

      /* free the mem used to print the number */
      free(txt);

      bottom = bottom - j;

      if(bottom == top) break;

   }

   /* fill in the rest of the indexes */
   for(i=indx, j=bottom; j>top; i++, j--){
      sprintf(label, "%d: ", i+1);
      labelLen = strlen(label);
      if(labelLen > lcdWidth) labelLen = lcdWidth;
      strncpy(lcdText[j-1], label, labelLen);
   }

   if(isEditingEditor()){

      txt = getLineEditor();
      curPos = cursorPosEditor();

      if(curPos > lcdWidth){
         txt += (curPos - lcdWidth);
      }
      strLen = strlen(txt);
      if(strLen > lcdWidth) strLen = lcdWidth;

      strncpy(lcdText[lcdHeight-1], txt, strLen);
   }

}

/* Redraw the contents of the lcd display. 
 * note: start BORDER pixels down and to the right.
 * TODO: we could be smart and only redraw what has changed. 
 * TODO: does this work correctly with varriable sized fonts?
 */
void drawStackLCD(){
   int i;
   int curPos;
   int strt, stop;
   GC drawgc;

   /* draw the stack */
   for(i=0; i<lcdHeight; i++){

      /* draw the first unhighlighted section of the line */
      strt = 0;
      if(dataSelected == DATA_SELECTED && i>=hiY1 && i<=hiY2){
         stop = hiX1;
         if(stop > lcdWidth) stop = lcdWidth;
      } else {
         stop = lcdWidth;
      }
      if(stop-strt){
	 XDrawImageString(XtDisplay(lcdDA), XtWindow(lcdDA), lcdGC, 
	    strt*fontW + BORDER, (i+1)*fontH - fontD + BORDER, 
	    &lcdText[i][strt], stop-strt);
      }

      /* draw the highlighted section of the line */
      if(dataSelected == DATA_SELECTED && i>=hiY1 && i<=hiY2){
	 strt = hiX1;
         if(strt > lcdWidth) strt = lcdWidth;
	 stop = hiX2+1;
         if(stop > lcdWidth) stop = lcdWidth;
         if(stop-strt){
	    XDrawImageString(XtDisplay(lcdDA), XtWindow(lcdDA), lcdHighlightGC, 
	       strt*fontW + BORDER, (i+1)*fontH - fontD + BORDER, 
	       &lcdText[i][strt], stop-strt);
         }
      }

      /* draw the last unhighlighted section of the line */
      stop = lcdWidth;
      if(dataSelected == DATA_SELECTED && i>=hiY1 && i<=hiY2){
         strt = hiX2+1;
         if(strt > lcdWidth) strt = lcdWidth;
      } else {
         strt = lcdWidth;
      }
      if(stop-strt){
	 XDrawImageString(XtDisplay(lcdDA), XtWindow(lcdDA), lcdGC, 
	    strt*fontW + BORDER, (i+1)*fontH - fontD + BORDER, 
	    &lcdText[i][strt], stop-strt);
      }
   }

   /* draw the cursor */
   if(isEditingEditor()){
      curPos = cursorPosEditor();
      if(curPos > lcdWidth){
         curPos = lcdWidth;
      }
      if(dataSelected == DATA_SELECTED &&
	 curPos>=hiX1 && curPos<=hiX2 && 
         lcdHeight-1>=hiY1 && lcdHeight-1<=hiY2)
      {
         drawgc = lcdHighlightGC;
      } else {
	 drawgc = lcdGC;
      }

      /* hack: draw a white line to erase any old cursor
         in position 0. XDrawImageString dosen't cover this
         up like it does with an old cursor in any other position */
      if(!(dataSelected == DATA_SELECTED && 
         0==hiX1 && lcdHeight-1>=hiY1 && lcdHeight-1<=hiY2))
      {
	 XDrawLine(XtDisplay(lcdDA), XtWindow(lcdDA), lcdHighlightGC, 
	    BORDER - 1, (lcdHeight-1) * fontH + BORDER, 
	    BORDER - 1, lcdHeight * fontH - 1 + BORDER);
      }

      /* draw the cursor */
      XDrawLine(XtDisplay(lcdDA), XtWindow(lcdDA), drawgc, 
	 fontW * curPos - 1 + BORDER, (lcdHeight-1) * fontH + BORDER, 
	 fontW * curPos - 1 + BORDER, lcdHeight * fontH - 1 + BORDER);
   }
}



/* redraw what we need to */ 
void redrawLCD(){

   calcStackLCD();
   drawStackLCD();

}


void lcdInitFont(){

   /* TODO: maybe make the font a user selectable thing */
   char *fontname = "12x24";
   XFontStruct *fontInfo;
   unsigned long valuemask;
   XGCValues values;
   Pixel fg, bg;
   Display *display;

   display = XtDisplay(lcdDA);

   /* get the foreground and background colors */
   XtVaGetValues(lcdDA, 
      XmNforeground, &fg,
      XmNbackground, &bg,
      NULL);

   /* Create default Graphics Context */
   valuemask = GCForeground | GCBackground;
   values.foreground = fg;
   values.background = bg;
   lcdGC = XCreateGC(display, XtWindow(lcdDA), valuemask, &values);

   /* highlight swaps foreground and background */
   valuemask = GCForeground | GCBackground;
   values.foreground = bg;
   values.background = fg;
   lcdHighlightGC = XCreateGC(display, XtWindow(lcdDA), valuemask, &values);

   /* Access font */
   /* TODO: this introduces a memory leak. but do we care? */
   if(NULL == (fontInfo = XLoadQueryFont(display, fontname))){
      fprintf(stderr, "Cannot open %s font\n", fontname); exit(0);
   }

   XSetFont(display, lcdGC, fontInfo->fid);
   XSetFont(display, lcdHighlightGC, fontInfo->fid);

   /* globals we use all over the place */
   fontH = fontInfo->max_bounds.ascent + fontInfo->max_bounds.descent;
   fontW = fontInfo->max_bounds.width;
   fontD = fontInfo->max_bounds.descent;
}


void lcdResize(){
   int i;
   char *buf;
   Dimension width, height;
   XtVaGetValues(lcdDA, XtNwidth, &width, XtNheight, &height, NULL);
 
   lcdWidth = (width - 2*BORDER) / fontW;
   if(lcdWidth < 0) lcdWidth = 0;

   lcdHeight = (height - 2*BORDER) / fontH;
   if(lcdHeight < 0) lcdHeight = 0;
 
   /* free the old mem */
   if(lcdText){
      free(lcdText[0]);
      free(lcdText);
   }

   /* get the new mem. note: add 1 bec size could otherwize be 0. */
   if(NULL == (lcdText=(char**)malloc((1+lcdHeight)*sizeof(char *)))){
      perror("malloc");
      exit(0);
   }
   if(NULL == (buf=(char*)malloc(1+lcdHeight*lcdWidth*sizeof(char)))){
      perror("malloc");
      exit(0);
   }
   lcdText[0] = buf;  /* in case lcdHeight == 0 */
   for(i=0; i<lcdHeight; i++){
      lcdText[i] = &buf[lcdWidth * i];
   }
 
   /* we don't get an expose if the window shrinks */
   XClearWindow(XtDisplay(lcdDA), XtWindow(lcdDA));
   calcStackLCD();
   drawStackLCD();
}



void setShortLCD(int val){

   if(val == LONG_DISPLAY) lcdDisplayMode = LONG_DISPLAY;
   else if(val == SHORT_DISPLAY) lcdDisplayMode = SHORT_DISPLAY;
   else printf("setShortLCD invalid mode\n");

}


void lcdExposeCB(Widget w, XtPointer clientData, XtPointer callData){
   static int initalized = 0;

   if(initalized == 0){
      lcdInitFont();
      lcdResize();
      initalized = 1;
   }

   XClearWindow(XtDisplay(lcdDA), XtWindow(lcdDA));

   drawStackLCD();
}


void lcdResizeCB(Widget w, XtPointer clientData, XtPointer callData){
   lcdResize();
}


void lcdInputCB(Widget w, XtPointer clientData, XtPointer callData){
   XmAnyCallbackStruct *cb = (XmAnyCallbackStruct *)callData;

   /* get the key that was pressed */
   if(cb->event->type == KeyPress){
   
      processInput((XKeyEvent *)(cb->event), 'd');
   }
}


void lcdKeyPressEvnt(
   Widget w,
   XtPointer client_data,
   XEvent *evnt,
   Boolean *continue_to_dispatch)
{

   /* get the key that was pressed */
   if(evnt->type == KeyPress){
   
      processInput((XKeyEvent *)(evnt), 'd');
   }
}

void lcdButtonEvnt(
   Widget w,
   XtPointer client_data,
   XEvent *evnt,
   Boolean *continue_to_dispatch)
{

   /* get the button that was pressed */
   if(evnt->type == ButtonPress){
 
      switch(evnt->xbutton.button){
         case 1:
            lcdStartHighlight(evnt);
            break;
         case 2:
            break;
         case 3:
            break;
      }
   } else if(evnt->type == ButtonRelease){

      switch(evnt->xbutton.button){
         case 1:       /* stop highlighting an area */
            lcdStopHighlight(evnt);
            break;
         case 2:       /* paste data */
	    lcdGetSelection(evnt);
            break;
         case 3:       /* might be used to continue highlighting */
            if(dataSelected == DATA_SELECTED){
	       lcdContinueHighlight(evnt->xbutton.x, evnt->xbutton.y);
	       lcdStopHighlight(evnt);
            }
            break;
      }
   }
}

void lcdMotionEvnt(
   Widget w,
   XtPointer client_data,
   XEvent *evnt,
   Boolean *continue_to_dispatch)
{
   /* get the button that was pressed */
   if(evnt->type == MotionNotify){

      if(evnt->xmotion.state & Button1Mask){

         /* highlighting with button 1 */
	 lcdContinueHighlight(
	    evnt->xmotion.x,
	    evnt->xmotion.y);

      } else if((evnt->xmotion.state & Button3Mask) && 
         dataSelected == DATA_SELECTED){

         /* continuing a highlight with button 3 */
	 lcdContinueHighlight(
	    evnt->xmotion.x,
	    evnt->xmotion.y);
      }
   }
}


Boolean convertSelection(
   Widget w,
   Atom *selection, 
   Atom *target, 
   Atom *type,
   XtPointer *value,
   unsigned long *length,
   int *format)
{
   int i;
   int width, height;
   int len;
   char *data;
   
   /* no highlight */
   if(dataSelected != DATA_SELECTED) return False;

   if(*target == XA_STRING){
      width = hiX2 - hiX1 + 1;
      height = hiY2 - hiY1 + 1;
      len = (width + 1) * height;

      *type = XA_STRING;
      data = (char *)XtMalloc(len * sizeof(char));
      *length = len - 1;
      *format = 8;

      for(i=0; i<height; i++){
         strncpy(data+(i*(width+1)), &(lcdText[i+hiY1][hiX1]), width);
         *(data+(i*(width+1))+width) = '\n';
      }

      *value = (XtPointer)data;
      return True;
   }

   return False;
}

void loseSelection(
    Widget w,
    Atom *selection)
{
   /* if there is a current (old) selection, then clear it */
   if(dataSelected == DATA_SELECTED){
      dataSelected = NONE_SELECTED;
      drawStackLCD();
   }
}


void lcdStartHighlight(XEvent *evnt){
   int xx, yy;

   /* if there is a current (old) selection, then clear it */
   if(dataSelected == DATA_SELECTED){
      dataSelected = NONE_SELECTED;
      drawStackLCD();

      /* ... and tell Xt */
      XtDisownSelection(lcdDA, XA_PRIMARY, evnt->xbutton.time);
   }

   xx = (evnt->xbutton.x - BORDER) / fontW;
   if(xx < 0) xx = 0;
   if(xx >= lcdWidth) xx = lcdWidth - 1;

   yy = (evnt->xbutton.y - BORDER) / fontH;
   if(yy < 0) yy = 0;
   if(yy >= lcdHeight) yy = lcdHeight - 1;

   hiX0 = hiX1 = hiX2 = xx;
   hiY0 = hiY1 = hiY2 = yy;

   dataSelected = START_SELECTING;

}

void lcdStopHighlight(XEvent *evnt){
   int reCalc, curPos;
   int xx, yy;

   /* if the user started to select something, but didn't actually
      select anything, then reset stuff */
   if(dataSelected == START_SELECTING){
      dataSelected = NONE_SELECTED;

      /* user may have been positioning the edit cursor */
      if(isEditingEditor()){
	 xx = (evnt->xbutton.x - BORDER) / fontW;
	 if(xx < 0) xx = 0;
	 if(xx >= lcdWidth) xx = lcdWidth - 1;
       
	 yy = (evnt->xbutton.y  - BORDER) / fontH;
         if(yy == lcdHeight-1){


            curPos = cursorPosEditor();
	    if(curPos > lcdWidth){
               reCalc = 1;
	       xx += (curPos - lcdWidth);
	    } else {
               reCalc = 0;
            }
            setCursorPosEditor(xx);

            /* need to recalc since cursor may have moved out of the
               visable part of the editor */
            if(reCalc) calcStackLCD();
         }

      }

      /* clear any old highlights */
      drawStackLCD();
   }

   /* if data is now selected */
   if(dataSelected == DATA_SELECTED){

      /* regester ourself as the selection owner */
      if(False == XtOwnSelection(
	 lcdDA,
	 XA_PRIMARY,
	 evnt->xbutton.time,
	 convertSelection,
	 loseSelection,
         NULL))
      {
         fprintf(stderr, "failed to gain selection\n");

	 dataSelected = NONE_SELECTED;
	 drawStackLCD();
      }
   }
}



void lcdContinueHighlight(int x, int y){
   int xx, yy;
   int lastx1, lastx2, lasty1, lasty2;

   xx = (x - BORDER) / fontW;
   if(xx < 0) xx = 0;
   if(xx >= lcdWidth) xx = lcdWidth - 1;

   yy = (y - BORDER) / fontH;
   if(yy < 0) yy = 0;
   if(yy >= lcdHeight) yy = lcdHeight - 1;

   lastx1 = hiX1;
   lastx2 = hiX2;
   lasty1 = hiY1;
   lasty2 = hiY2;

   if(xx < hiX0){
      hiX1 = xx;
      hiX2 = hiX0 + 0;
   } else {
      hiX1 = hiX0;
      hiX2 = xx + 0;
   }

   if(yy < hiY0){
      hiY1 = yy;
      hiY2 = hiY0 + 0;
   } else {
      hiY1 = hiY0;
      hiY2 = yy + 0;
   }

   /* if the highlight area changed */
   if(lastx1 != hiX1 || lastx2 != hiX2 || lasty1 != hiY1 || lasty2 != hiY2){

      dataSelected = DATA_SELECTED;

      /* repaint the display */
      drawStackLCD();
   }

}


void lcdGetSelection(XEvent *evnt){
   /* ask for data */
   XtGetSelectionValue(lcdDA,
      XA_PRIMARY,
      XA_STRING,
      lcdPasteCB,
      (XtPointer)(evnt->xbutton.time),
      evnt->xbutton.time);
}

void lcdPasteCB(
   Widget w,
   XtPointer client_data,
   Atom *selection,
   Atom *type,
   XtPointer value,
   unsigned long *length,
   int *format)
{
   int i;
   char *str;
   Atom CLIPBOARD;

   /* check for valid data */
   if(*type != XA_STRING || *length == 0){

      /* TODO: do we really want to try the clipboard? */

      if(*selection != XA_PRIMARY){
         /* fprintf(stderr, "clipboard paste failed\n"); */
         return;
      }

      /* selection failed.  try the clipboard */
      CLIPBOARD = XInternAtom(XtDisplay(lcdDA), "CLIPBOARD", True);
      if(CLIPBOARD == None){
         fprintf(stderr, "Selecton failed, and no CLIPBOARD atom exists\n");
         return;
      } 
      XtGetSelectionValue(lcdDA,
	 CLIPBOARD,
	 XA_STRING,
	 lcdPasteCB,
	 NULL,
	 (Time)client_data);

      return;
   }

   str = (char*)value;

   for(i=0; i<*length; i++){

      /* TODO maybe do something like isalpha() here? */
      processInput(NULL, str[i]);
   }

   XtFree((char*)value);
}
