/*
 * Here we collect some simple statistics about the performance of
 * the virtual modem. These are only for use when the modem is not
 * running as a forked daemon. The initialisation is used to preserve
 * a channel to the output and this is used to print the statistic
 * reports.
 */

#include <stdio.h>
#include "virtmodem.h"

/* These are the spinning wheels used to indicate packet flow */

#define WHEEL		"-\\|/"	/* actually 4 chars due to quoting */
#define WHEEL_LEN	4

static FILE *report = NULL;
static struct pk_stats {
    char *name;
    long pkts;
    long min;
    long max;
    long bytes;
} s[2];

/*
 * stats_call indicates a new call has been made. It prints the
 * called host if one is given and initialises the stats structures.
 */

void
stats_call(char *host)
{
    if(host && report) {
	fprintf(report,"Call made to %s\n",host);
	fflush(report);
    }
    s[0].pkts=0;
    s[0].name="Incoming";
    s[1].pkts=0;
    s[1].name="Outgoing";
}

/*
 * Initialise the statsistics by duplicating stderr and setting
 * the numbers of packets to zero using the stats_call function.
 */

void
stats_init(void)
{
    int newfd;
    newfd = dup(2);
    if(newfd<0)
	return;
    report=fdopen(newfd,"w");
    stats_call(NULL);
}

/*
 * Finish by reporting the statistics for incoming and outgoing packets.
 * The statistics are then reset to prevent duplicate printing.
 */

void
stats_end(void)
{
    int i;
    if(!report)
	return;
    for(i=0;i<2;i++) {
	if(s[i].pkts>0) {
	    fprintf(report,"\r%s: %ld pkts",s[i].name,s[i].pkts);
	    fprintf(report," [min=%ld,max=%ld,av=%ld]\n",
		s[i].min,s[i].max,s[i].bytes/s[i].pkts);
	s[i].pkts=0;
	}
    }
    fflush(report);
}

/*
 * Add a packet in a particular direction. 0 is incoming, 1 is
 * outgoing. Zero length packets are not recorded.
 */

void
add_pkt(int dirn, long size)
{
    if(!report || size<=0)
	return;

    if(s[dirn].pkts==0) {
	s[dirn].min=s[dirn].max=s[dirn].bytes=size;
	s[dirn].pkts=1;
	return;
    }

    s[dirn].pkts++;
    s[dirn].bytes+=size;
    if(size>s[dirn].max)
	s[dirn].max=size;
    if(size<s[dirn].min)
	s[dirn].min=size;
    fprintf(report,"\rIn %c    Out %c",
	WHEEL[s[0].pkts%WHEEL_LEN],
	WHEEL[s[1].pkts%WHEEL_LEN]);
    fflush(report);
}
