// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <sys/ioctl.h>
#include <linux/telephony.h>
#include <trunkmap.h>
#include <runmap.h>
#include <vocab.h>

#ifdef	IXJ_DRIVER
#include <linux/ixjuser.h>
#endif

#define PH_MAKE_EVENT(arg1, arg2)       (arg1+arg2)

#define	UM_SERVER_EVENTS		0x8000
#define	UM_EVENT_ENTER_STATE		PH_MAKE_EVENT(UM_SERVER_EVENTS, 0)
#define	UM_EVENT_INITIALIZE_PORT	PH_MAKE_EVENT(UM_SERVER_EVENTS, 1)
#define UM_EVENT_MAKE_BUSY		PH_MAKE_EVENT(UM_SERVER_EVENTS, 2)
#define UM_EVENT_MAKE_IDLE		PH_MAKE_EVENT(UM_SERVER_EVENTS, 3)
#define	UM_EVENT_STOP_STATE		PH_MAKE_EVENT(UM_SERVER_EVENTS, 4)
#define	UM_EVENT_EXIT_SHELL		PH_MAKE_EVENT(UM_SERVER_EVENTS, 5)
#define UM_EVENT_RING_LINE		PH_MAKE_EVENT(UM_SERVER_EVENTS, 6)
#define	UM_EVENT_START_LINE		PH_MAKE_EVENT(UM_SERVER_EVENTS, 7)
#define UM_EVENT_ERROR_ABORT		PH_MAKE_EVENT(UM_SERVER_EVENTS, 8)
#define UM_EVENT_EXIT_MODULE		PH_MAKE_EVENT(UM_SERVER_EVENTS, 9)

#define	PH_EVENT_TIMER_EXPIRED		PH_MAKE_EVENT(0, 0)
#define	PH_EVENT_INPUT_PENDING		PH_MAKE_EVENT(0, 1)
#define	PH_EVENT_SYSTEM_NOTIFY		PH_MAKE_EVENT(0, 2)
#define PH_EVENT_DTMF_DIGIT		PH_MAKE_EVENT(0, 3)
#define	PH_EVENT_LINE_RINGING		PH_MAKE_EVENT(0, 4)
#define	PH_EVENT_CALLER_ID		PH_MAKE_EVENT(0, 5)
#define	PH_EVENT_LINE_WINKING		PH_MAKE_EVENT(0, 6)
#define PH_EVENT_OFF_HOOK		PH_MAKE_EVENT(0, 7)
#define PH_EVENT_ON_HOOK		PH_MAKE_EVENT(0, 8)
#define PH_EVENT_SERVICE_ENDING		PH_MAKE_EVENT(0, 9)

class Trunk;
class Session;

typedef	struct
{
	unsigned long event, p1, p2;
}	EventRecord;

typedef enum
{
	generic_driver,
	ixj_driver
}	driver_t;

typedef	void (Trunk::*handler_t)(EventRecord *er);

class Service
{
protected:
	Trunk *trunk;
	int device;
	driver_t driver;

	Service(Trunk *trk);
	void endService(void);
};

class Player : private Thread, private Service
{
private:
	int fd;
	long offset;	
	char pathname[256];
	int frame;

	Player(Trunk *trk, char *path, long pos = 0l);
	~Player();

	void Initial(void);
	void Run(void);

	friend class Trunk;
};

class WordPlayer : private Thread, private Service
{
private:
	WordPlayer(Trunk *trk, WORD *word);

	void Run(void);

	friend class Trunk;
};

class Recorder : private Thread, private Service
{
private:
	int fd;
	char pathname[256];
	int frame;

	Recorder(Trunk *trk, char *path);
	~Recorder();

	void Initial(void);
	void Run(void);

	friend class Trunk;
};

class Dialer : private Thread, private Service
{
private:
	char *dial;

	Dialer(Trunk *trunk, char *str);
	~Dialer();

	void Run(void);

	friend class Trunk;
};

class Trunk : public Trunkmap, public Mutex
{
private:
	int  device;
	struct timeval porttimer;
	Session *session;
	int cap_count;
	struct phone_capability *cap_list;
	driver_t driver;
	Thread *service;

	void resetAudio(void);
	void resetThreads(void);

protected:
	char *runmap;
	
	void (Trunk::*handler)(EventRecord *er);
	void setHandler(handler_t h);
	void setTimer(long timeout);
	void endTimer(void);
	void setDTMF(void);
	void endDTMF(void);
	void scrStop(void);
	void scrHangup(void);
	void exitModule(char *errmsg, int id);

	void scrFlash(void)
		{setHandler((handler_t)&Trunk::FlashOn);};

	void scrModule(void)
		{setHandler((handler_t)&Trunk::Module);};
	
	void scrSay(void)
		{setHandler((handler_t)&Trunk::Idle);};

	void scrPlay(void)
		{setHandler((handler_t)&Trunk::Play);};

	void scrExWait(void)
		{setHandler((handler_t)&Trunk::ExWait);};

	void scrSleep(void) 
		{setHandler((handler_t)&Trunk::TrkSleep);};

	void scrWait(void)
		{setHandler((handler_t)&Trunk::TrkWait);};

	void scrAnswer(void)
		{setHandler((handler_t)&Trunk::Answer);};

	void scrBusyout(void)
		{setHandler((handler_t)&Trunk::Busy);};

	void scrDial(void)
		{setHandler((handler_t)&Trunk::Dial);}

	void scrCollect(void)
		{setHandler((handler_t)&Trunk::Collect);};

	void scrRecord(void)
		{setHandler((handler_t)&Trunk::Record);};

	void scrStep(void)
		{return scrNext();};

	void Busy(EventRecord *er);
	void Idle(EventRecord *er);
	void Answer(EventRecord *er);
	void Default(EventRecord *er);

	void PostMessage(EventRecord *er);
	void TrkSleep(EventRecord *er);
	void TrkWait(EventRecord *er);
	void ExWait(EventRecord *er);
	void Module(EventRecord *er);
	void Say(EventRecord *er);
	void Play(EventRecord *er);
	void Collect(EventRecord *er);
	void Record(EventRecord *er);
	void Dial(EventRecord *er);
	void FlashOn(EventRecord *er);
	void FlashOff(EventRecord *er);

public:
	Trunk(Runmap *map, char *scr, int fd, Session *session);
	virtual ~Trunk();

	long getTimer(void);
	void getEvent(void);
	struct phone_capability *getCapability(phone_cap id, int sub = -1);

	void setExiting(void);

	inline int getDevice(void)
		{return device;};

	inline driver_t getDriver(void)
		{return driver;};

	friend void SendMessage(Trunk *trk, EventRecord *er);
};

extern Trunk **trunks;



