// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of ACS as noted here.
//
// This exception is that permission is hereby granted to link ACS with
// the Pika MonteCarlo static libraries to produce a executable image
// without requiring MonteCarlo itself to be supplied in source form so
// long as each source file so linked contains this exclusion.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name ACS.  If you copy code from other releases into a copy of
// ACS, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to ACS, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include <config.h>
#include <sys/stat.h>
#include <cc++/thread.h>
#include "pika.h"
#include "audio.h"
#include "server.h"

Audio::Audio(TResourceHandle dsp, char *fn, long pos = 0) :
Semaphore(), Thread((Semaphore *)this, atoi(getkeyfirst(keythread, "audio")))
{
	char *ext = strrchr(fn, '/');
	char *p;
	int sampling;

	buffers = atoi(getkeyfirst(keyaudio, "buffers"));
	frames = atoi(getkeyfirst(keyaudio, "frames"));
	fd = -1;
	
	hDsp = dsp;
	buffer = NULL;
	offset = pos;
	idle = false;

	//  subdir/xxx uses current prefix also...

	if(strchr(fn, '/'))
		pathname[0] = 0;
	else
	{
		strcpy(pathname, getkeylast(keypaths, "prompts"));
		strcat(pathname, "/");
	}

	strcat(pathname, fn);
	
	if(ext)
		ext = strrchr(ext, '.');
	else
		ext = strrchr(fn, '.');

	p = getkeyfirst(keyaudio, "sampling");
	if(!p)
		p = "8";

	switch(*p)
	{
	case 4:
		sampling = AU_SAMPLING_4khz;
		break;
	case 6:
		sampling = AU_SAMPLING_6khz;
		break;
	case 1:
		sampling = AU_SAMPLING_11khz;
		break;
	default:
		sampling = AU_SAMPLING_8khz;
		break;
	}

	p = getkeyfirst(keyaudio, "encoding");
	if(!p)
		p = "ulaw";

	switch(*p)
	{
	case 'l':
	case 'L':
		setaulong(aufile.encoding, AU_ENCODING_PCM16);
		break;
	case 'p':
	case 'P':
	case '8':
		setaulong(aufile.encoding, AU_ENCODING_PCM8);
		break;
	case '3':
		setaulong(aufile.encoding, AU_ENCODING_G723);
		break;
	case 'A':
	case 'a':
		setaulong(aufile.encoding, AU_ENCODING_ALAW);
		break;
	case '4':
		setaulong(aufile.encoding, AU_ENCODING_G721);
		break;
	case 'U':
	case 'u':
		setaulong(aufile.encoding, AU_ENCODING_MULAW);
		break;
	case 'g':
	case 'G':
		if(!strcmp(p + 1, "721"))
		{
			setaulong(aufile.encoding, AU_ENCODING_G721);
			break;
		}
		if(!strcmp(p + 1, "723"))
		{
			setaulong(aufile.encoding, AU_ENCODING_G723);
			break;
		}
	default:
		setaulong(aufile.encoding, AU_ENCODING_MULAW);
	}

	setaulong(aufile.magic, AU_FILE_MAGIC);
	setaulong(aufile.hdrsize, 0l);
	setaulong(aufile.channels, 01l);
	setaulong(aufile.datasize, AU_SIZE_UNKNOWN);
	setaulong(aufile.sampling, sampling);

	if(!ext)
	{
		ext = ".ul";
		strcat(pathname, ".ul");
	}

	if(!stricmp(ext, ".a24") || !stricmp(ext, ".24k") || !stricmp(ext, ".vox"))
	{
		setaulong(aufile.encoding, AU_ENCODING_G721);
		setaulong(aufile.sampling, AU_SAMPLING_6khz);
	}

	if(!stricmp(ext, ".a32") || !stricmp(ext, ".32k"))
	{
		setaulong(aufile.encoding, AU_ENCODING_G721);
		setaulong(aufile.sampling, AU_SAMPLING_8khz);
	}

	if(!stricmp(ext, ".adpcm") || !stricmp(ext, ".g721"))
	{
		setaulong(aufile.encoding, AU_ENCODING_G721);
		setaulong(aufile.sampling, sampling);
	}

	if(!stricmp(ext, ".g723"))
	{
		setaulong(aufile.encoding, AU_ENCODING_G723);
		setaulong(aufile.sampling, sampling);
	}

	if(!stricmp(ext, ".64k"))
	{
		setaulong(aufile.encoding, AU_ENCODING_PCM8);
		setaulong(aufile.sampling, AU_SAMPLING_8khz);
	}

	if(!stricmp(ext, ".pcm"))
	{
		setaulong(aufile.encoding, AU_ENCODING_PCM8);
		setaulong(aufile.sampling, sampling);
	}

	if(!stricmp(ext, ".pcm16"))
	{
		setaulong(aufile.encoding, AU_ENCODING_PCM16);
		setaulong(aufile.sampling, sampling);
	}

	if(!stricmp(ext, ".ul") || !stricmp(ext, ".64k"))
	{
		setaulong(aufile.encoding, AU_ENCODING_MULAW);
		setaulong(aufile.sampling, AU_SAMPLING_8khz);
	}

	if(!stricmp(ext, ".al") || !stricmp(ext, ".wve"))
	{
		setaulong(aufile.encoding, AU_ENCODING_ALAW);
		setaulong(aufile.sampling, AU_SAMPLING_8khz);
	}

	if(!stricmp(ext, ".au"))
		setaulong(aufile.hdrsize, sizeof(AUFILE));

	ctrl = (TBufferHeader *)new TBufferHeader[frames];
}

void	Audio::setFormat(int format, int sampling)
{
	if(buffer)
		return;
	
	filler = PK_M_LAW_SILENCE;

	switch(format)
	{
	case AU_ENCODING_PCM8:
		format = PK_PCM_8_BIT;
		break;
	case AU_ENCODING_PCM16:
		format = PK_PCM_16_BIT;
		break;
	case AU_ENCODING_G723:
		format = PK_ADPCM_3_BIT;
		break;
	case AU_ENCODING_G721:
		format = PK_ADPCM_4_BIT;
		break;
	case AU_ENCODING_ALAW:
		format = PK_COMPOUND_A_LAW;
		filler = PK_A_LAW_SILENCE;
		break;
	default:
		format = PK_COMPOUND_MU_LAW;
		filler = PK_M_LAW_SILENCE;
	}

	switch(sampling)
	{
	case AU_SAMPLING_4khz:
		sampling = PK_SAMPLING_RATE_4_KHz;
		break;
	case AU_SAMPLING_6khz:
		sampling = PK_SAMPLING_RATE_6_KHz;
		break;
	case AU_SAMPLING_11khz:
		sampling = PK_SAMPLING_RATE_11_KHz;
		break;
	default:
		sampling = PK_SAMPLING_RATE_8_KHz;
	}

	PK_AUDIO_SetFormat(hDsp, format | sampling);

	bufsize = PK_AUDIO_BufferSize(hDsp);
	buffer = (char *)new char[bufsize * buffers * frames];
}

Audio::~Audio()
{
	PK_AUDIO_Reset(hDsp);

	if(fd > -1)
		close(fd);

	if(ctrl)
		delete[] ctrl;

	if(buffer)
		delete buffer;
}

int Audio::Fill(int frame, int size, unsigned flag)
{
	int bufs;
	int mod;
	char *buf;

	if(size > bufsize * buffers)
		size = bufsize * buffers;

	bufs = size / bufsize;
	mod = size % bufsize;
	if(mod || !bufs)
	{
		buf = getFrameData(frame) + size;
		while(mod++ < bufsize)
		{
			*(buf++) = filler;
			++size;
		}
	}

	ctrl[frame].lpData = getFrameData(frame);
	ctrl[frame].dwBufferLength = size;
	ctrl[frame].dwBytesRecorded = size;
	ctrl[frame].dwLoops = 0;
	ctrl[frame].dwFlags = flag;
	ctrl[frame].lpNext = NULL;
	ctrl[frame].reserved = 0;
	return size;
}

int Audio::Empty(int frame, unsigned flag)
{
	ctrl[frame].lpData = getFrameData(frame);
	ctrl[frame].dwBufferLength = getFrameSize();
	ctrl[frame].dwBytesRecorded = 0;
	ctrl[frame].dwLoops = 0;
	ctrl[frame].dwFlags = flag;
	ctrl[frame].lpNext = NULL;
	ctrl[frame].reserved = 0;
	PK_AUDIO_InputAddBuffer(hDsp, &ctrl[frame]);
	return getFrameSize();
}

