// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of ACS as noted here.
//
// This exception is that permission is hereby granted to link ACS with
// the Pika MonteCarlo static libraries to produce a executable image
// without requiring MonteCarlo itself to be supplied in source form so
// long as each source file so linked contains this exclusion.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name ACS.  If you copy code from other releases into a copy of
// ACS, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to ACS, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include <config.h>
#include <getopt.h>
#include <lockfile.h>
#include <sched.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <sys/resource.h>
#include <scheduler.h>
#include <realtime.h>
#include <shell.h>
#include <dso.h>
#include <script.h>
#include <pwd.h>
#include <trunkmap.h>
#include <module.h>
#include <cc++/file.h>
#include <vocab.h>
#include "pika.h"
#include "server.h"

keydata_t *keypaths, *keytimers, *keyldap, *keyscript, *keyaudio, *keythread;
Thread *scheduler;
Shell *sh;
int stepping = 0;
bool trace = false;

typedef	struct
{
	char *key;
	char *data;
}	KEYDEFS;

static KEYDEFS	defthreads[] =
	{{"scheduling", "0"},
	 {"audio", "0"},
	 {"gateways", "1"},
	 {NULL, NULL}};

static KEYDEFS	defpaths[] =
	{{"pidfile", "/var/lock/umd.pid"},
	 {"runfile", "/var/run/umd"},
	 {"ctrlfile", "/var/run/umdctrl"},
	 {"datafiles", "/var/ACS"},
	 {"schedule", "/etc/umd.sched"},
	 {"prompts", "/usr/share/aaprompts"},
	 {"scripts", "/usr/share/aascripts"},
	 {"libexec", "/usr/libexec/tgi"},
	 {"logpath", "/var/log/acs"},
	 {NULL, NULL}};

static KEYDEFS defscript[] =
	{{"default", "default"},
	 {NULL, NULL}};

static KEYDEFS defldap[] =
	{{"context", "o=DEFAULT, c=US"},
	 {"servers", "localhost"},
	 {NULL, NULL}};

static KEYDEFS defaudio[] =
	{{"buffers", "16"},
	 {"frames", "2"},
	 {"min", "16"},
	 {"trim", "1"},
	 {"timer", "60"},
	 {"inputgain", "10"},
	 {"outputgain", "10"},
	 {NULL, NULL}};

static void loadkeys(keydata_t *keys, KEYDEFS *defs)
{
	while(defs->key)
	{
		if(!getkeyfirst(keys, defs->key))
			setkeyvalue(keys, defs->key, defs->data);
		++defs;
	}
}

static RETSIGTYPE final(int signo)
{
	char	*pidfile = getkeylast(keypaths, "pidfile");

	signal(SIGSEGV, SIG_DFL);
	unlockmem();

#ifdef	HAVE_MODULES
	stopModules();
#endif
	stopDriver();
#ifdef	HAVE_MODULES
	unload();
#endif
	if(scheduler)
	{
		delete scheduler;
		scheduler = NULL;
	}

	if(sh)
	{
		delete sh;
		sh = NULL;
	}

	endScript();
	unloadVocab();

	if(pidfile)
		remove(pidfile);

	if(signo)
		syslog(LOG_INFO, "exiting; reason %d", signo);
	else
		syslog(LOG_INFO, "exiting");
 
	closelog();
	remove(getkeyfirst(keypaths, "ctrlfile"));
	exit(signo);
}

void initial(int argc, char **argv)
{
	static bool daemon = false;
	static bool usage = false;
	static bool test = false;
	static int pri = 0, pid;
	static int lines = 0;
	static char *scr = NULL;
	struct passwd *pwd;
	char *ctrlfile;
	int uid, pool;
	char *p;
	char pstr[11];
	sigset_t sigs;
	char cwd[256];
	char env[256];
	char *cwp;
	int stack;
	
	static struct option long_options[] =
	{
		{"background", 0, 0, 'D'},
		{"foreground", 0, 0, 'F'},
		{"daemon", 0, 0, 'D'},
		{"priority", 1, 0, 'p'},
		{"lines", 1, 0, 'l'},
		{"test", 0, 0, 't'},
		{"help", 0, 0, 'h'},
		{"debug", 0, 0, 'x'},
		{"stepping", 0, 0, 's'},
	        {"trace", 0, 0, 'T'},
		{0, 0, 0, 0}
	};

	char	*cp, *p1, *p2;
	int 	opt, opt_index;

	sigemptyset(&sigs);
	sigaddset(&sigs, SIGTERM);
	sigaddset(&sigs, SIGQUIT);
	sigaddset(&sigs, SIGINT);
	pthread_sigmask(SIG_BLOCK, &sigs, NULL);

	while(EOF != (opt = getopt_long(argc, argv, "DTFhtp:l:x:s:", long_options, &opt_index)))
		switch(opt)
		{
		case 'x':
			_debug = atoi(optarg);
			break;
		case 't':
			if(!_debug)
				_debug = 5;
			test = true;
			break;
		case 'l':
			lines = atoi(optarg);
			break;
		case 's':
			stepping = atoi(optarg);
			break;
		case 'p':
			pri = atoi(optarg);
			break;
		case 'D':
			trace = false;
			daemon = true;
			break;
		case 'T':
			trace = true;
		case 'F':
			daemon = false;
			break;
		case 'h':
		default:
			usage = true;
		}

	if(optind < argc)
		scr = argv[optind++];

	if(usage || optind < argc)
	{
		printf( "use: umd [-options] [script-override]\n"
			"The following options are available:\n\n"
			"-T, --trace             trace runtime state changes\n"
			"-t, --test              run in test mode from source tree\n"
			"-l, --lines=n           enable only up to 'n' lines\n"
			"-p, --priority=n        default process priority\n"
			"-F, --foreground        execute in foreground\n"
			"-D, --daemon            execute in daemon mode\n"
			"-s, --stepping=n        stepping interval\n"
		);
		    
		exit(-1);
	}

	if(test)
	{
		getcwd(cwd, sizeof(cwd));
		cwp = cwd + strlen(cwd);
		if(!strchr(argv[0], '/') || argv[0][0] == '.')
			cwp = strrchr(cwd, '/');

		strcpy(cwp, "/etc/");
		setenv("CONFIG_KEYDATA", cwd, 1);
	}

	keypaths = getkeydata("/umd/paths");
	keythread = getkeydata("/umd/threads");
	keyldap = getkeydata("/umd/ldap");
	keyscript = getkeydata("/umd/script");
	keyaudio = getkeydata("/umd/audio");

	openlog("umd", LOG_PERROR | LOG_NDELAY, LOG_DAEMON);

	if(!keypaths || !keythread || !keyldap || !keyscript || !keyaudio)
		throw(keypaths);

	loadkeys(keypaths, defpaths);
	loadkeys(keyldap, defldap);
	loadkeys(keythread, defthreads);
	loadkeys(keyscript, defscript);
	loadkeys(keyaudio, defaudio);

	if(test)
	{
		strcpy(cwp, "/etc/umd.sched");
		setkeyvalue(keypaths, "schedule", cwd);
		strcpy(cwp, "/aascripts");
		setkeyvalue(keypaths, "scripts", cwd);
		strcpy(cwp, "/aaprompts");
		setkeyvalue(keypaths, "prompts", cwd);
		strcpy(cwp, "/utils");
		setkeyvalue(keypaths, "libexec", cwd);

		p = getkeyfirst(keypaths, "lockfiles");
		if(p)
		{
			p = strtok(p, " \t\n");
			while(p)
			{
				while((pid = lockfile(p, "umd")))
				{
					if(pid < 0)
						throw(pid);
					sleep(2);
				}
				p = strtok(NULL, " \t\n");
			}
		}
	}

	File pidfile(getkeylast(keypaths, "pidfile"),
		FILE_OPEN_WRITEONLY, 0644);

	sprintf(pstr, "%010d", getpid());
	write(pidfile, pstr);

	if(!pri)
	{
		cp = getkeylast(keythread, "pri");
		if(!cp)
			cp = getkeylast(keythread, "priority");

		if(cp)
			pri = atoi(cp);
	}

	if(!stepping)
	{
		cp = getkeylast(keythread, "stepping");
		if(cp)
			stepping = atoi(cp);
	}

	endkeydata(NULL);
	pwd = getpwnam("mail");
	if(!pwd)
		throw(pwd);

	uid = pwd->pw_uid;
	setgid(pwd->pw_gid);	

	p = getkeylast(keypaths, "datafiles");
	mkdir(p, 0750);
	chdir(p);
	mkdir("tmp", 0770);

	if(!test)
	{
		p = getkeylast(keypaths, "prompts");
		chmod(p, 0775);
		chown(p, 0, pwd->pw_gid);

		p = getkeylast(keypaths, "scripts");
		chmod(p, 0700);
	}

	endpwent();

	if(daemon)
	{
		closelog();
		close(0);
		close(1);
		close(2);
		openlog("umd", LOG_NDELAY, LOG_DAEMON);
		pdetach();
	}

	setenv("SERVER_PLATFORM", "pika", 1);
	setenv("SERVER_LIBEXEC", getkeylast(keypaths, "libexec"), 1);
	setenv("SERVER_SOFTWARE", "ACS", 1);	
	setenv("SERVER_VERSION", "1.0", 1);
	setenv("SERVER_PROMPTS", getkeylast(keypaths, "prompts"), 1);
	setenv("SERVER_RUNTIME", getkeylast(keypaths, "datafiles"), 1);

	ctrlfile = getkeyfirst(keypaths, "ctrlfile");
	remove(ctrlfile);
	mkfifo(ctrlfile, 0600);
	pool = atoi(getkeyfirst(keythread, "gateways"));
	sh = new Shell(ctrlfile, uid, pool);

	loadVocab(keypaths);

#ifdef	HAVE_MODULES
	load();
#endif

	if(pri)
	{
		pri = getpriority(PRIO_PROCESS, 0) - pri -1;
		setpriority(PRIO_PROCESS, 0, pri);
	}

	p = getkeyfirst(keythread, "stack");
	if(p)
		stack = atoi(p);
	else
		stack = 0;

	p = getkeyfirst(keythread, "scheduler");
	if(p)
		pri = atoi(p);
	else
		pri = 0;	

	scheduler = new Scheduler(getkeylast(keypaths, "schedule"), pri);
	new Script(getkeylast(keypaths, "scripts"), keyscript);

	if(!scheduler)
		throw(scheduler);

	initDriver(scr, lines);
	scheduler->Start();
#ifdef	HAVE_MODULES
	startModules();
#endif	

	signal(SIGTERM, final);
	signal(SIGQUIT, final);
	signal(SIGINT, final);
	signal(SIGABRT, final);
	pthread_sigmask(SIG_UNBLOCK, &sigs, NULL);
	syslog(LOG_INFO, "startup");
	if(stack)
	{
		signal(SIGSEGV, final);
		if(lockmem(stack))
			final(SIGSEGV);
	}	
}

int main(int argc, char **argv)
{
        int     fd;
        int     bpos;
	char	buf[128];
	char	*args[32];
	int	len;
	char	*ctrlfile;
	Sessionmap *session;

	try
	{
		initial(argc, argv);
	}
	catch(keydata_t *p)
	{
		syslog(LOG_CRIT, "umd.conf: missing or incomplete");
		exit(-1);
	}	
	catch(Scheduler *sc)
	{
		syslog(LOG_CRIT, "umd.sched: missing or incomplete");
		exit(-1);
	}
	catch(Script *scr)
	{
		syslog(LOG_CRIT, "script files missing");
		exit(-1);
	}
	catch(Vocab *vcb)
	{
		syslog(LOG_CRIT, "failed vocabulary loading");
		exit(-1);
	}
	catch(File *fp)
	{
		syslog(LOG_CRIT, "initialization file failure");
		exit(-1);
	}
	catch(int card)
	{
		syslog(LOG_CRIT, "pika cards missing or failed");
		exit(-1);
	}
	catch(...)
	{
		syslog(LOG_CRIT, "initialization failure"); 
		exit(-1);
	}

	ctrlfile = getkeyfirst(keypaths, "ctrlfile");
	fd = open(ctrlfile, O_RDWR);
	if(fd < 0)
	{
		syslog(LOG_CRIT, "%s: failed", ctrlfile);
		final(-1);
	}
	debug(4, "waiting on fifo <%s>", ctrlfile);

	session = getSessionHandler();
	for(;;)
	{
                debug(6, "waiting for fifo input");
                bpos = 0;
                while(bpos < sizeof(buf) - 1)
		{
                        len = read(fd, buf + bpos, 1);
			if(len < 1)
			        continue;
			if(buf[bpos] == '\n')
                                break;
                        ++bpos;
		}
                buf[bpos]=0;
		debug(4, "command received <%s>", buf);

		argc = 0;
		args[argc] = strtok(buf, " \t\n");
		while(args[argc] && argc < 31)
			args[++argc] = strtok(NULL, " \t\n");

		len = argc;
		while(len < 32)
			args[len++] = NULL;

		if(!stricmp(args[0], "down"))
			break;

		if(!stricmp(args[0], "compile"))
			new Script(getkeylast(keypaths, "scripts"), keyscript);

		if(!stricmp(args[0], "exit"))
			session->exitTGI(atoi(args[1]), atoi(args[2]));
	
		if(!stricmp(args[0], "wait"))
			session->setTGI(atoi(args[1]), atoi(args[2]));

		if(!stricmp(args[0], "busy"))
		{
			len = 1;
			while(--argc)
				session->setBusy(atoi(args[len++]));
		}

		if(!stricmp(args[0], "idle"))
		{
			len = 1;
			while(--argc)
				session->setIdle(atoi(args[len++]));
		}

		if(!stricmp(args[0], "start"))
			session->callTrunk(atoi(args[1]), args[2]);

		if(!stricmp(args[0], "set"))
			session->setVariable(atoi(args[1]), args[2], args[3]);

		if(!stricmp(args[0], "put"))
			session->putSymbol(args[1], args[2]);

		if(!stricmp(args[0], "ring"))
			session->ringTrunk(atoi(args[1]));

#ifdef	HAVE_MODULES
		fifoModule(argc, args);
#endif
	}		
	final(0);
}

