// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of ACS as noted here.
//
// This exception is that permission is hereby granted to link ACS with
// the Pika MonteCarlo static libraries to produce a executable image
// without requiring MonteCarlo itself to be supplied in source form so
// long as each source file so linked contains this exclusion.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name ACS.  If you copy code from other releases into a copy of
// ACS, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to ACS, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include <config.h>
#include <debug.h>
#include <stdlib.h>
#include <ctype.h>
#include <sym.h>
#include <cc++/thread.h>
#include <cc++/file.h>
#include "trunkmap.h"
#include "script.h"
#include "shell.h"
//#include "portmap.h"
#include "module.h"
#include "database.h"
#include "vocab.h"

char	Trunkmap::dtmfkeys[] = "0123456789*#ABCD";
Trunkmap *Trunkmap::firstmap = NULL;
Trunkmap *Trunkmap::lastmap = NULL;
int Trunkmap::active = 0;

Trunkmap::Trunkmap(Runmap *map, char *scr)
{
	runmap = map;
	dtmf = listflag = offhook = idle = trap = false;
	timer = 0;
	rings = 0;
	digits = 0;
	digit[0] = 0;
	cid[0] = 0;
	dnid[0] = 0;
	ani[0] = 0;
	volume = 0x5050;
//	portmap = NULL;
	script = NULL;
	mailbox = NULL;
	strcpy(lang, "UsEngM");
	vocab = getVocabulary(lang);

//	if(getPortmaps())
//		portmap = new Trunkmap*[getPortmaps()];

	if(!firstmap)
	{
		firstmap = lastmap = this;
		nextmap = NULL;
	}
	else
	{
		lastmap->nextmap = this;
		lastmap = this;
		nextmap = NULL;
	}

	yield();
	clrSchedule();
	setCurrent("Starting...");
	if(!scr)
		setSchedule(SCHED_RESET, "default");
	else
		setSchedule(SCHED_OVERRIDE, scr);

	id = ++active;
}

Trunkmap::~Trunkmap()
{
	if(script)
		detach(script);

//	if(portmap)
//		delete[] portmap;

	setCurrent("DOWN");
	--active;
}

void Trunkmap::setcid(void)
{
	struct tm *dt;

	dt = localtime(&ringtime);
	sprintf(cid, "time=%02d:%02d:%02d&date=%04d%02d%02d",
		dt->tm_hour, dt->tm_min, dt->tm_sec,
		dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
}

void Trunkmap::setCurrent(char *str)
{
	yield();
	memset(runmap->current, 0, 32);
	strncpy(runmap->current, str, 31);	
}

void Trunkmap::setSchedule(int sc, char *scr)
{
	switch(sc)
	{
	case SCHED_DAILY:
	case SCHED_RESET:
		if(sched == SCHED_OVERRIDE)
			return;
	}

	yield();
	memset(runmap->schedule, 0, 32);
	strncpy(runmap->schedule, scr, 31);
	sched = sc;
}

void Trunkmap::scrErrorStep(char *errmsg)
{
	strncpy(error, errmsg, 16);
	error[16] = 0;

	debug(1, "%s(%d): error handler", errmsg, id);

	if(menu->ERR == menu->first || (cmd[stack]->mask & ERROR_MASK))
		scrStep();
	else
	{
		cmd[stack] = menu->ERR;
		scrExecute();
	}
}

void Trunkmap::scrExists(void)
{
	char *path = getOption(NULL);
	struct stat ino;

	if(!path)
		return scrErrorStep("missing-arg");

	if(!stat(path, &ino))
	{
		scrGoto();
		return;
	}

	scrStep();
}

void Trunkmap::scrErase(void)
{
	char *target = getOption(NULL);

	if(target)
	{
		if(*target != '/')
		{
			if(remove(target))
				return scrErrorStep("remove-failed");
		}
		else
			return scrErrorStep("invalid-path");
	}
	else
		return scrErrorStep("missing-arg");
	scrStep();
}

void Trunkmap::scrMove(void)
{
	char *source = getOption(NULL);
	char *target = getOption(NULL);

	if(source && target)
	{
		if(*source != '/' && *target != '/')
		{
			if(!link(source, target))
			{
				if(remove(source))
					return scrErrorStep("remove-failed");
			}
			else
				return scrErrorStep("move-failed");
		}
		else
			return scrErrorStep("invalid-path");
	}
	else
		return scrErrorStep("missing-arg");

	scrStep();
}

void Trunkmap::scrLink(void)
{
	char *src = getOption(NULL);
	char *target;

	while(NULL != (target = getOption(NULL)))
	{
		if(*src != '/' && *target != '/')
		{
			if(link(src, target))
				return scrErrorStep("link-failed");
		}
		else
			return scrErrorStep("invalid-path");
	}
	scrStep();
}

void Trunkmap::scrIf(void)
{
	int l1, l2;
	char *op1 = getOption(NULL);
	char *op = getOption(NULL);
	char *op2 = getOption(NULL);

	if(!op1 || !op || !op2)
		return scrErrorStep("missing-arg");

	if(!stricmp(op, "="))
	{
		if(atol(op1) == atol(op2))
			return scrGoto();
	}

	if(!stricmp(op, "<>"))
	{
		if(atol(op1) != atol(op2))
			return scrGoto();
	}

	if(!stricmp(op, "=="))
	{
		if(!stricmp(op1, op2))
			return scrGoto();
	}

	if(!stricmp(op, "!="))
	{
		if(stricmp(op1, op2))
			return scrGoto();
	}

	if(!stricmp(op, "$"))
	{
		if(strstr(op2, op1))
			return scrGoto();
	}

	if(!stricmp(op, "$<") || !stricmp(op, "$+"))
	{
		if(!strnicmp(op1, op2, strlen(op1)))
			return scrGoto();
	}

	if(!stricmp(op, "$>") || !stricmp(op, "$-"))
	{
		l1 = strlen(op1);
		l2 = strlen(op2);
		if(l1 <= l2)
			if(!strnicmp(op1, op2 + l2 - l1, l1))
				return scrGoto();
	}

	if(!stricmp(op, ">"))
	{
		if(atol(op1) > atol(op2))
			return scrGoto();
	}

	if(!stricmp(op, "<"))
	{
		if(atol(op1) < atol(op2))
			return scrGoto();
	}

	if(!stricmp(op, "<=") || !stricmp(op, "=<"))
	{
		if(atol(op1) <= atol(op2))
			return scrGoto();
	}

	if(!stricmp(op, ">=") || !stricmp(op, "=>"))
	{
		if(atol(op1) >= atol(op2))
			return scrGoto();
	}

	scrStep();
}

void Trunkmap::scrPrefix(void)
{
	char *target = getOption(NULL);
	char *value;

	if(!target)
		return scrErrorStep("missing-arg");

	while(NULL != (value = getOption(NULL)))
	{
		if(!strnicmp(value, target, strlen(value)))
			return scrGoto();
		getOption(NULL);
	}
	scrStep();
}

void Trunkmap::scrTrace(void)
{
	char *opt = getOption("on");
	if(!stricmp(opt, "off"))
		listflag = false;
	else
		listflag = true;
	scrStep();
}

void Trunkmap::scrSelect(void)
{
	char *target = getOption(NULL);
	char *value;

	if(!target)
		return scrErrorStep("missing-arg");

	while(NULL != (value = getOption(NULL)))
	{
		if(!stricmp(value, target))
			return scrGoto();
		getOption(NULL);	// skip to next entry
	}		
	scrStep();
}

void Trunkmap::scrPop(void)
{
	if(stack)
	{
		cmd[stack - 1] = cmd[stack];
		--stack;
	}
	else
		return scrErrorStep("stack-empty");

	scrStep();
}

void Trunkmap::scrReturn(void)
{
	if(stack)
		--stack;
	else
		return scrErrorStep("stack-empty");
	scrStep();
}

void Trunkmap::scrCall(void)
{
	if(stack < (MAX_STACK - 1))
	{
		cmd[stack + 1]	= cmd[stack];
		++stack;
	}
	else
		return scrErrorStep("stack-full");

	return scrGoto();
}

void Trunkmap::scrTrap(void)
{
	char *opt;

	if(trap)
		return scrGoto();

	opt = getOption(NULL);
	if(opt)
		menu->TRAP = cmd[stack];
	else
		menu->TRAP = NULL;
}

bool Trunkmap::exeTrap(void)
{
	if(trap)
		return true;

	if(!offhook || !menu->TRAP)
		return false;

	trap = true;
	debug(2, "trap(%d): invoking trap script", id);
	stack = 0;
	cmd[stack] = menu->TRAP;
	menu->TRAP = NULL;
	scrExecute();
}

void Trunkmap::scrGoto(void)
{
	char buf[32];
	Menu *start = NULL;
	int dig;

	char *opt = getOption(getSchedule(buf));

	if(!*opt)
		return scrErrorStep("missing-arg");

	if(*opt == '#')
	{
		dig = opt[1] - '0';
		if(dig > -1 && dig < 10)
		{
			cmd[stack] = menu->L[dig];
			debug(3, "branching %s(%d)", opt, id);
			scrExecute();
			return;
		}
		else
			return scrErrorStep("invalid-arg");
	}

	if(*opt == '^')
	{
		for(dig = 0; dig < 16; ++dig)
			if(opt[1] == dtmfkeys[dig])
				break;
	
		if(dig == 16)
		{
			dig = -1;	
			switch(opt[1])
			{
			case 't':
			case 'T':
				cmd[stack] = menu->TO;
				debug(3, "branching timeout(%d)", id);
				scrExecute();
				return;
			case 'e':
			case 'E':
				cmd[stack] = menu->ERR;
				debug(3, "branching error(%d)", id);
				scrExecute();
				return;
			}
		}
		if(dig >= 0)
		{
			debug(3, "branching digit-%d(%d)", dig, id);
			cmd[stack] = menu->D[dig];			
			scrExecute();
			return;
		}
		else
			return scrErrorStep("invalid-arg");
	}

	start = lookup(script, opt);
	if(!start)
	{
		syslog(LOG_WARNING, "%s: cannot start", opt);
		return scrErrorStep("missing-script");
	}
	menu = start;
	menu->TRAP = NULL;
	debug(2, "starting %s(%d)", opt, id);
	offset = 0l;
	cmd[stack] = menu->first;
	setCurrent(menu->name);
	scrExecute();
}

void Trunkmap::scrAttach(char *scr)
{
	char schedule[33];

	if(!script)
		script = attach();

	if(scr)
		strcpy(schedule, scr);
	else
		getSchedule(schedule);

	menu = lookup(script, schedule);
	menu->TRAP = NULL;
	if(!menu)
	{
		syslog(LOG_WARNING, "%s: cannot start", schedule);
		return;
	}

	debug(1, "scheduling %s(%d)", schedule, id);
	offset = 0l;
	stack = 0;
	trap = listflag = false;
	time(&starttime);
	cmd[stack] = menu->first;
	setCurrent(menu->name);
	scrExecute();
}

void Trunkmap::scrDetach(void)
{
	if(script)
	{
		detach(script);
		script = NULL;
	}
};


void Trunkmap::scrExecute(void)
{
	argc = 0;
	temps = 0;
	if(listflag)
	{
		syslog(LOG_DEBUG, "%03d %s(%d) %s, mask=%08x",
			cmd[stack]->line,
			menu->name,
			id,
			cmd[stack]->cmd,
			cmd[stack]->mask);
	}
	(this->*(cmd[stack]->scrHandler))();
}

void Trunkmap::scrTimeout(void)
{
	if(menu->TO != menu->first)
		cmd[stack] = menu->TO;
	else
		cmd[stack] = cmd[stack]->next;
	scrStop();
}

void Trunkmap::scrError(char *err)
{
	strcpy(error, err);
	if(menu->ERR != menu->first && !(cmd[stack]->mask & ERROR_MASK))
		cmd[stack] = menu->ERR;
	else
		cmd[stack] = cmd[stack]->next;
	scrStop();
}

void Trunkmap::scrDigit(int trap)
{
	if(menu->D[trap] != menu->first && !cmd[stack]->mask)
	{
		digits = 1;
		digit[0] = dtmfkeys[trap];
		digit[1] = 0;
		cmd[stack] = menu->D[trap];
		scrStop();
		return;
	}	
	if(digits > 32)
		return;

	digit[digits++] = dtmfkeys[trap];
	digit[digits] = 0;
}

void Trunkmap::scrLog(void)
{
	char buf[256];
	char *opt;

	sprintf(buf, "%s(%02d):", menu->name, id);
	while(NULL != (opt = getOption(NULL)))
	{
		strcat(buf, " ");
		strcat(buf, opt);
	}
	syslog(LOG_INFO, "%s", buf);
	scrStep();
}

void Trunkmap::scrTrim(void)
{
	char *opt = getOption(NULL);
	int l1;
	int c;

	if(!opt)
		return scrErrorStep("missing-arg");

	l1 = strlen(opt);
	if(*opt == '-')
	{
		++opt;
		--l1;
		if(l1 <= digits)
		{
			if(!strcmp(opt, digit + digits - l1))
			{
				digits -= l1;
				digit[digits] = 0;
			}
		}
		scrStep();
		return;
	}
	
	if(!strnicmp(digit, opt, l1))
	{
		for(c = l1; c <= digits; ++c)
			digit[c - l1] = digit[c];

		digits -= l1;
	}
	scrStep();
}

void Trunkmap::scrAdd(void)
{
	char *opt = getOption(NULL);
	int l1;
	int c;

	if(!opt)
		return scrErrorStep("missing-arg");

	if(*opt == '-')
	{
		++opt;
		strncat(digit, opt, 32 - digits);
		digit[32] = 0;
		digits = strlen(digit);
		scrStep();
		return;
	}
	
	l1 = strlen(opt);
	if(l1 + digits > 32)
	{
		scrStep();
		return;
	}
	
	for(c = digits; c >= 0; --c)
		digit[c + l1] = digit[c];
		
	strncpy(digit, opt, l1);
	digits += l1;
	scrStep();
}

void Trunkmap::scrDrop(void)
{
	if(!mailbox)
		return scrErrorStep("no-mailbox");

	mailbox->Drop();
	scrStep();
}

void Trunkmap::scrSave(void)
{
	if(!mailbox)
		return scrErrorStep("no-mailbox");
	
	mailbox->Save();
	scrStep();
}

void Trunkmap::scrSkip(void)
{
	if(!mailbox)
		return scrErrorStep("no-mailbox");

	mailbox->Skip(getOption("next"));
	scrStep();
}

void Trunkmap::scrMessages(void)
{
	if(!mailbox)
		return scrErrorStep("no-mailbox");

	mailbox->setMessages(getOption("new"));
	scrStep();
}

void Trunkmap::scrClear(void)
{
	char *opt = getOption(NULL);
	int count = 0;
	int pos = 0;

	if(opt)
	{
		if(*opt == '-')
			count = -atoi(++opt);
		else
			count = atoi(opt);
	}

	if(count > 0 && count < digits)
	{
		digit[digits] = 0;
		for(;;)
		{
			digit[pos] = digit[pos + count];
			if(!digit[pos])
				break;
			++pos;
		}
		digits = strlen(digit);
	}
	else if(count < 0 && -count < digits)
	{
		digits += count;
		digit[digits] = 0;	
	}
	else
	{
		digits = 0;
		digit[0] = 0;
	}
	scrStep();
}

void Trunkmap::scrTGI(bool wait, bool detach)
{
	char buf[128];
	char *p;
	char *q;
	int cnt = 0;
	int args = 0;

	strcpy(buf, getOption(NULL));
	
	for(;;)
	{
		q = cmd[stack]->argv[argc];

		if(NULL == (p = getOption(NULL)))
			break;

		if(!(cnt++))
			strcat(buf, " query=");
		else
			strcat(buf, "&");
		
		if(*q == '%' || *q == '@')
			++q;
		else
		{
			sprintf(buf, "value%d", ++args);
			q = buf;	
		}
		strcat(buf, q);
		strcat(buf, "=");		
		strcat(buf, p);
	}

	if(digits > 0)
	{
		digit[digits] = 0;
		strcat(buf, " digits=");
		strcat(buf, digit);
	}

	if(cid[0])
	{
		strcat(buf, " cid=");
		strcat(buf, cid);
	}

	if(ani[0])
	{
		strcat(buf, " ani=");
		strcat(buf, ani);
	}

	if(dnid[0])
	{
		strcat(buf, " dnid=");
		strcat(buf, dnid);
	}

	waitpid = 0;
	exstatus = -1;

	if(detach)
		execute(0, buf);
	else
		execute(id, buf);

	if(wait)
	{
		argc = 0;
		scrExWait();
	}
	else
		scrStep();
}

void Trunkmap::scrLibexec(void)
{
	char *timeout = getOption(NULL);
	if(timeout)
	{
		if(!stricmp(timeout, "trap"))
		{
			menu->TRAP = NULL;
			scrTGI(false, true);
		}
		else if(!stricmp(timeout, "detach"))
		{
			scrTGI(false, true);
		}
		else if(!stricmp(timeout, "start"))
		{
			scrTGI(false, false);
		}
		else if(!stricmp(timeout, "wait") || !stricmp(timeout, "exit"))
		{
			scrExWait();
		}
		else if(!stricmp(timeout, "kill"))
		{
			if(waitpid)
			{
				kill(waitpid, SIGINT);
				waitpid = 0;
			}
			scrStep();
		}
		else
			scrTGI(true, false);
	}
	else
		scrErrorStep("missing-arg");
}

void Trunkmap::scrList(void)
{
	char env[256];
	char *target = getOption(NULL);
	char *e;

	if(!target)
		return scrErrorStep("missing-arg");
	
	env[0] = 0;
	while(NULL != (e = getOption(NULL)))
	{
		if(env[0])
			strcat(env, ":");
		strcat(env, e);
	}
	setsym(target, env, id);
	item = 0;
	scrStep();
}

void Trunkmap::scrLastItem(void)
{
	char *target = getOption(NULL);
	char *list = getOption(NULL);
	char *cp, *last;

	if(!target)
		return scrErrorStep("missing-arg");

	last = list;
	item = 0;
	if(!list)
	{
		list = target;
		target = NULL;
	}
	while(NULL != (cp = strchr(last, ':')))
	{
		last = ++cp;
		++item;
	}
	if(!target)
		return scrStep();

	setsym(target, last, id);
	scrStep();
}

void Trunkmap::scrFirstItem(void)
{
	char *target = getOption(NULL);
	char *list = getOption(NULL);
	char env[256];
	int i = 0;

	if(!target)
		return scrErrorStep("missing-arg");

	item = 0;
	if(!list)
		return scrStep();

	while(*list && *list != ':')
		env[i++] = *(list++);

	env[i] = 0;
	setsym(target, env, id);
	scrStep();
}	

void Trunkmap::scrPrevItem(void)
{
	char *target = getOption(NULL);
	char *list = getOption(NULL);
	char env[256];
	int i = 1;
	char *cp;

	if(!target)
		return scrErrorStep("missing-arg");

	if(!item)
		return scrErrorStep("start-of-list");

	if(!list)
	{
		list = target;
		target = NULL;
	}

	cp = list;
	while(i++ < item)
	{
		cp = strchr(cp, ':');
		if(!cp)
			return scrErrorStep("end-of-list");
		++cp;
	}
	i = 0;
	--item;
	if(!target)
		return scrStep();

	while(*cp && *cp != ':')
		env[i++] = *(cp++);
	env[i] = 0;
	setsym(target, env, id);
}

void Trunkmap::scrNextItem(void)
{
	char *target = getOption(NULL);
	char *list = getOption(NULL);
	char env[256];
	int i = 0;
	char *cp;

	if(!target)
		return scrErrorStep("missing-arg");

	if(!list)
	{
		list = target;
		target = NULL;
	}

	cp = list;
	while(i++ < item)
	{
		cp = strchr(cp, ':');
		if(!cp)
			return scrErrorStep("end-of-list");
		++cp;
	}
	i = 0;
	++item;
	if(!target)
		return scrStep();

	while(*cp && *cp != ':')
		env[i++] = *(cp++);
	env[i] = 0;
	setsym(target, env, id);
}

	
void Trunkmap::scrSet(void)
{
	char env[256];
	char *target = getOption(NULL);
	int var;
	char *e;
	int vol;

	if(!target)
		return scrErrorStep("missing-arg");

	if(mailbox)
	{
		if(!stricmp(target, "password"))
			return mailbox->setPassword(getOption(""));
	
		if(!stricmp(target, "order"))
			return mailbox->setOrder(getOption(""));
	}

	if(!strnicmp(target, "var", 3))
	{
		var = atoi(target + 3) % 10;
		vars[var][0] = 0;
		while(NULL != (target = getOption(NULL)))
			strcat(vars[var], target);
	}
	else if(!strnicmp(target, "lang", 4))
	{
		strcpy(lang, getOption("UsEngM"));
		vocab = getVocabulary(lang);
		if(!vocab)
			debug(1, "%s(%d): no vocabulary found", lang, id);
		return;
	}
	else if(!strnicmp(target, "temp", 4))
	{
		var = atoi(target + 4) % 4;
		temp[var][0] = 0;
		while(NULL != (target = getOption(NULL)))
			strcat(vars[var], target);
	}
	else if(!strnicmp(target, "digit", 5))
	{
		strncpy(digit, getOption(""), 32);
		digit[32] = 0;
		digits = strlen(digit);
	}
	else if(!strnicmp(target, "record", 6))
	{
		vol = atoi(getOption("80"));
		if(vol < 0)
			vol = 0;
		if(vol > 99)
			vol = 99;
		volume = volume & 0x00ff | (vol * 256);
	}
	else if(!strnicmp(target, "play", 4))
	{
		vol = atoi(getOption("80"));
		if(vol < 0)
			vol = 0;
		if(vol > 99)
			vol = 99; 
		volume = volume & 0xff00 | atoi(getOption("80"));
	}
	else
	{		
		env[0] = 0;
		while(NULL != (e = getOption(NULL)))
			strcat(env, e);
		setsym(target, env, id);
	} 
	scrStep();
}

void Trunkmap::scrDefine(void)
{
	char env[256];
	char *target = getOption(NULL);
	char *e;

	if(!target)
		return scrErrorStep("missing-arg");
	
	env[0] = 0;
	while(NULL != (e = getOption(NULL)))
		strcat(env, e);

	setsym(target, env, 0);
	scrStep();
}

void Trunkmap::scrInit(void)
{
	char env[256];
	char *target = getOption(NULL);
	char *e;

	if(!target)
		return scrErrorStep("missing-arg");

	target = getsym(target, id);
	if(target)
	{
		scrStep();
		return;
	}

	env[0] = 0;
	while(NULL != (e = getOption(NULL)))
		strcat(env, e);
	
	setsym(target, env, id);
	scrStep();
}

void Trunkmap::scrNext(void)
{
	cmd[stack] = cmd[stack]->next;
	scrExecute();
}

#ifdef	HAVE_MODULES

void Trunkmap::scrMailbox(void)
{
	MBIModule *mbi = (MBIModule *)cmd[stack]->module;
	char *err;

	if(mailbox)
	{
		delete mailbox;
		mailbox = NULL;
	}

	if(!mbi)
		return scrErrorStep("no-mbi");

	err = mbi->Mailbox(this);
	if(err)
		return scrErrorStep(err);
	if(mbi->getTimeout())
		scrModule();
	else
		scrStep();
}

void Trunkmap::scrExtension(void)
{
	MBIModule *mbi = (MBIModule *)cmd[stack]->module;
	char *err;

	if(mailbox)
	{
		delete mailbox;
		mailbox = NULL;
	}

	if(!mbi)
		return scrErrorStep("no-mbi");

	err = mbi->Extension(this);
	if(err)
		return scrErrorStep(err);
	if(mbi->getTimeout())
		scrModule();
	else
		scrStep();
}

void Trunkmap::scrCommit(void)
{
	MBIModule *mbi = (MBIModule *)cmd[stack]->module;
	char *err;

	if(!mbi)
		return scrErrorStep("no-mbi");

	err = mbi->Commit(this);
	if(err)
		return scrErrorStep(err);
	if(mbi->getTimeout())
		scrModule();
	else
		scrStep();
}

void Trunkmap::scrDeliver(void)
{
	MBIModule *mbi = (MBIModule *)cmd[stack]->module;
	char *err;

	if(!mbi)
		return scrErrorStep("no-mbi");

	err = mbi->Deliver(this);
	if(err)
		return scrErrorStep(err);
	if(mbi->getTimeout())
		scrModule();
	else
		scrStep();
}

void Trunkmap::scrInsert(void)
{
	DBIModule *dbi = (DBIModule *)cmd[stack]->module;
	char *err;

	if(!dbi)
		return scrErrorStep("no-dbi");

	err = dbi->Insert(this);
	if(err)
		return scrErrorStep(err);
	if(dbi->getTimeout())
		scrModule();
	else
		scrStep();
}

void Trunkmap::scrUpdate(void)
{
	DBIModule *dbi = (DBIModule *)cmd[stack]->module;
	char *err;

	if(!dbi)
		return scrErrorStep("no-dbi");

	err = dbi->Update(this);
	if(err)
		return scrErrorStep(err);
	if(dbi->getTimeout())
		scrModule();
	else
		scrStep();
}

void Trunkmap::scrDelete(void)
{
	DBIModule *dbi = (DBIModule *)cmd[stack]->module;
	char *err;

	if(!dbi)
		return scrErrorStep("no-dbi");

	err = dbi->Delete(this);
	if(err)
		return scrErrorStep(err);
	if(dbi->getTimeout())
		scrModule();
	else
		scrStep();
}

void Trunkmap::scrSearch(void)
{
	DBIModule *dbi = (DBIModule *)cmd[stack]->module;
	char *err;

	if(!dbi)
		return scrErrorStep("no-dbi");

	err = dbi->Search(this);
	if(err)
		return scrErrorStep(err);
	if(dbi->getTimeout())
		scrModule();
	else
		scrStep();
}

#endif

void Trunkmap::setVariable(char *sym, char *value)
{
	int var;
	char env[256];

	if(!strnicmp(sym, "var", 3))
	{
		var = atoi(sym + 3);
		if(var >= -1 && var < 10)
		{
			strncpy(vars[var], value, 32);
			temp[var][33] = 0;
		}
		return;
	}	

	if(!strnicmp(sym, "temp", 4))
	{
		var = atoi(sym + 4);
		if(var >= -1 && var < 4)
		{
			strncpy(temp[var], value, 32);
			temp[var][33] = 0;
		}
		return;
	}

	if(!stricmp(sym, "digits"))
	{
		var = strlen(value);
		if(var > 32)
			var = 32;

		if(var)
			strncpy(digit, value, var);
		digit[var] = 0;
		digits = var;
		return;
	}
	
	setsym(sym, value, id);
}
		
void Trunkmap::setPid(int pid)
{
	if(exstatus != -1)
		return;

	waitpid = pid;
}

char *Trunkmap::getVariable(char *var)
{
	char env[40];

	if(!strnicmp(var, "var", 3))
		return vars[atoi(var + 4) % 10];

	if(!strnicmp(var, "digit", 5)) 
		return digit;

	if(!stricmp(var, "script"))
		return menu->name;

	return getsym(var, id);
}

time_t Trunkmap::getIdleTime(void)
{
	time_t now;

	if(idle)
	{
		time(&now);
		return now - idletime;
	}
	else
		return 0;
}	

void Trunkmap::sayWord(char *word)
{
	char *buf = wordbuffer;
	char *dec;
	char *tail;

	if(!word)
		return;

	tail = word + strlen(word) - 1;

	wordcount = wordlast = 0;

	if(!stricmp(word, "<spell>"))
	{
		wordmask |= WORD_SPELL;
		return;
	}

	if(!stricmp(word, "</spell>"))
	{
		wordmask &= ~WORD_SPELL;
		return;
	}

	if(!stricmp(word, "<order>"))
	{
		wordmask |= WORD_ORDER;
		return;
	}

	if(!stricmp(word, "</order>"))
	{
		wordmask &= ~WORD_ORDER;
		return;
	}

	if(strchr(word, '/') || strchr(word, ':'))
	{
		wordlist[wordcount++] = word;
		return;
	}

	if(wordmask & WORD_SPELL)
	{
		while(*word && buf < wordbuffer + 60)
		{
			if(*word == '.')
				wordlist[wordcount++] = "DOT";
			if(isalnum(*word))
			{
				buf[0] = toupper(*word);
				buf[1] = 0;
				wordlist[wordcount++] = buf;
				buf += 2;
			}
			++word;
		}
		return;
	}

	if(isdigit(*word))
	{
		dec = strchr(word, '.');
		if(wordmask & WORD_ORDER)
		{
			sayNumber(atol(word), NUMBER_ORDER);
			return;
		}
		sayNumber(atol(word), NUMBER_NORMAL);
		if(dec)
		{
			wordlist[wordcount++] = "POINT";
			while(isdigit(*(++dec)))
			{
				buf[0] = *dec;
				buf[1] = 0;
				wordlist[wordcount++] = buf;
				buf += 2;
			}
		}
		if(*tail == '%')
			wordlist[wordcount++] = "PERCENT";
		return;
	}					

	if(strchr(word, '.'))
		if(*tail != '.')
		{
			wordlist[wordcount++] = word;
			return;
		}

	while(*word && buf < wordbuffer + 60)
	{
		if(isalnum(*word))
			*(buf++) = toupper(*word);
		++word;
	}
	*buf = 0;
	if(buf > wordbuffer)
		wordlist[wordcount++] = wordbuffer;
}

void Trunkmap::newWord(void)
{
	wordlast = wordcount = wordmask = 0;
}

char *Trunkmap::getWord(void)
{
	if(wordlast >= wordcount)
		return NULL;

	return wordlist[wordlast++];
}

void Trunkmap::sayNumber(long nbr, int mode)
{
	static char *norm[] =
		{"0", "1", "2", "3", "4",
		 "5", "6", "7", "8", "9",
		 "10", "11", "12", "13", "14",
		 "15", "16", "17", "18", "19"};

	static char *ords[] =
		{"0TH", "1ST", "2ND", "3RD", "4TH",
		 "5TH", "6TH", "7TH", "8TH", "9TH",
		 "10TH", "11TH", "12TH", "13TH", "14TH",
		 "15TH", "16TH", "17TH", "18TH", "19TH"};

	static char *tens[] =
		{"0",  "10", "20", "30", "40",
		 "50", "60", "70", "80", "90"};

	static char *tenths[] =
		{"0th", "10TH", "20TH", "30TH", "40TH",
		"50TH", "60TH", "70TH", "80TH", "90TH"};

	char **nums = norm;
	long power = 1000000;
	long hi, lo;

	if(mode == NUMBER_ORDER)
		nums = ords;

	if(!nbr)
	{
		wordlist[wordcount++] = nums[0];
		return;
	}

	nums = norm;

	while(power && nbr)
	{
		if(nbr < power)
		{
			power /= 1000;
			continue;
		}

		hi = nbr / power;
		nbr %= power;
		if(!nbr && mode == NUMBER_ORDER)
			nums = ords;

		if(hi > 99)
		{
			wordlist[wordcount++] = norm[hi / 100];
			hi %= 100;
			if(!hi && !nbr && mode == NUMBER_ORDER)
				wordlist[wordcount++] = "100TH";
			else 
				wordlist[wordcount++] = "100";
		}
		if(hi < 20)
		{
			if(hi)
				wordlist[wordcount++] = nums[hi];
		}
		else
		{
			lo = hi % 10;
			if(!lo && !nbr && mode == NUMBER_ORDER)
				wordlist[wordcount++] = tenths[hi / 10];
			else
				wordlist[wordcount++] = tens[hi / 10];
			if(lo)
				wordlist[wordcount++] = nums[lo];
		}
		if(power == 1000000)
		{
			if(mode == NUMBER_ORDER && !nbr)
				wordlist[wordcount++] = "1000000TH";
			else
				wordlist[wordcount++] = "1000000";
		}
		if(power == 1000)
		{
			if(mode == NUMBER_ORDER && !nbr)
				wordlist[wordcount++] = "1000TH";
			else
				wordlist[wordcount++] = "1000";
		}
		power /= 1000;
	}
}			

char *Trunkmap::getOption(char *def)
{
	char env[40];
	int var;
	char *option = cmd[stack]->argv[argc];
	temps % 4;
	time_t now;
	struct tm *dt;
	char *cp;

	if(!option)
		return def;

	++argc;

	if(*option == '@')
	{
		option = getsym(++option, 0);	
		if(!option)
			option = "";

		return option;
	}

	if(*option != '%')
		return option;

	if(!stricmp(option, "%empty"))
		return "";

	if(!strnicmp(option, "%temp", 5))
	{
		sprintf(temp[temps], "tmp/l%02sx%08lx%s",
			id, starttime & 0xffffffff, option + 5);
		return temp[temps++];
	}

	if(!strnicmp(option, "%var", 4))
	{
		var = atoi(option + 4) % 10;
		return vars[var];
	}

	if(!stricmp(option, "%language"))
		return lang;

	if(!stricmp(option, "%cid"))
		return cid;

	if(!stricmp(option, "%ani"))
		return ani;

	if(!stricmp(option, "%dnid"))
		return dnid;

	if(!stricmp(option, "%digits"))
		return digit;

	if(!stricmp(option, "%extension"))
	{
		strcpy(temp[temps], getExtension());
		return temp[temps++];
	}

	if(!stricmp(option, "%script"))
		return menu->name;

	if(!stricmp(option, "%count"))
	{
		sprintf(temp[temps], "%d", digits);
		return temp[temps++];
	}

	if(!stricmp(option, "%rings"))
	{
		sprintf(temp[temps], "%d", rings);
		return temp[temps++];
	}

	if(!stricmp(option, "%item"))
	{
		sprintf(temp[temps], "%d", item);
		return temp[temps++];
	}

	if(!stricmp(option, "%error"))
		return error;

	if(!stricmp(option, "%offset"))
	{
		sprintf(temp[temps], "%ld", offset);
		return temp[temps++];
	}

	if(!stricmp(option, "%date"))
	{
		time(&now);
		dt = localtime(&now);
		sprintf(temp[temps], "%04d%02d%02d",
			dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
		return temp[temps++];
	}

	if(!stricmp(option, "%logdate"))
	{
		time(&now);
		dt = localtime(&now);
		sprintf(temp[temps], "%02d/%02d/%04d",
			dt->tm_mon + 1, dt->tm_mday, dt->tm_year);
		return temp[temps++];
	}

	if(!stricmp(option, "%time"))
	{
		time(&now);
		dt = localtime(&now);
		sprintf(temp[temps], "%02d%02d%02d",
			dt->tm_hour, dt->tm_min, dt->tm_sec);
		return temp[temps++];
	}

	if(!stricmp(option, "%logtime"))
	{
		time(&now);
		dt = localtime(&now);
		sprintf(temp[temps], "%02d:%02d:%02d",
			dt->tm_hour, dt->tm_min, dt->tm_sec);
		return temp[temps];
	}


	if(!stricmp(option, "%runtime"))
	{
		time(&now);
		sprintf(temp[temps], "%d", now - starttime);
		return temp[temps++];
	}	

	if(!stricmp(option, "%status"))
	{
		sprintf(temp[temps], "%d", exstatus);
		return temp[temps++];
	}

	if(!stricmp(option, "%version"))
	{
		strcpy(temp[temps], VERSION);
		return temp[temps++];
	}

	if(!stricmp(option, "%release"))
	{
		strcpy(temp[temps], VERSION);
		cp = strchr(temp[temps], '.');
		if(cp)
			*cp = 0;
		return temp[temps++];
	}

	if(!stricmp(option, "%id"))
	{
		sprintf(temp[temps], "%d", id);
		return temp[temps++];
	}

	if(!strnicmp(option, "*:", 2))
	{
		sprintf(wordbuffer, "%s:%s", lang, option + 2);
		return wordbuffer;
	}

	if(mailbox)
	{
		if(!stricmp(option, "%message"))
			return mailbox->getMessage();

		if(!stricmp(option, "%password"))
			return mailbox->getPassword();

		if(!stricmp(option, "%term"))
			return mailbox->getTerm();

		if(!stricmp(option, "%user"))
			return mailbox->getUser();

		if(!stricmp(option, "%#record"))
			return mailbox->getRecordSize();

		if(!stricmp(option, "%mailbox"))
			return mailbox->getMailbox();

		if(!stricmp(option, "%extension"))
			return mailbox->getExtension();

		if(!stricmp(option, "%#new"))
			return mailbox->getNewCount();

		if(!stricmp(option, "%#old"))
			return mailbox->getOldCount();

		if(!stricmp(option, "%#saved"))
			return mailbox->getSavedCount();

	}
	option = getsym(++option, id);
	if(!option)
		option = "";

	return option;
}

