/*
 *  Flasher Source File
 *
 *  GNU Copyright (C) 2003 Gaspar Sinai <gsinai@yudit.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "flasher.h"

#include <getopt.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

using namespace FLASHER;

static void usage(const char* message);

static char* THIS="flasher";
static char* PROGRAM_VERSION="0.1";

/*!
 * \brief Flasher Main Program.
 */
int
main (int argc, char* argv[])
{
  // String
  const char* port="/dev/ttyS1";
  const char* id="00:00:00:00:00:00:00";
  const char* command = "mcu-version";
  const char* output_file = "flash.out";
  const char* data_format = "hex";
  bool quiet=false;
  bool ignore_error=false;

  // Converted from string.
  //Serial::BaudRate speed = Serial::BPS_57600; 
  Serial::BaudRate speed = Serial::BPS_9600; 
  unsigned long from=0xfe000;
  unsigned long   to=0xfffff;

  bool address_specified = false;

  char dummy[2];
  while (true)
  {
    int c;
    int this_option_optind = optind ? optind : 1;
    int option_index = 0;

    static struct option long_options[] =
    {
      {"id", 1, 0, 'i'},
      {"port", 1, 0, 'p'},
      {"baud", 1, 0, 'b'},
      {"ignore", 0, 0, 'e'},
      {"format", 1, 0, 'd'},
      {"quiet", 0, 0, 'q'},
      {"from", 1, 0, 'f'},
      {"to", 1, 0, 't'},
      {"command", 1, 0, 'c'},
      {"out", 1, 0, 'o'},
      {"version", 0, 0, 'v'},
      {0, 0, 0, 0}
    };
    c = getopt_long (argc, argv, "i:p:b:d:qef:t:c:o:v", long_options, &option_index);
    if (c == -1) break;
    if (c < 20)
    {
      c = long_options[option_index].val;
    }
    switch (c)
    {
    case 'c':
      command = optarg;
      break;
    case 'b':
      if (strcmp (optarg, "9600")==0)
      {
        speed = Serial::BPS_9600;
      }
      else if (strcmp (optarg, "19200")==0)
      {
        speed = Serial::BPS_19200;
      }
      else if (strcmp (optarg, "38400")==0)
      {
        speed = Serial::BPS_38400;
      }
      else if (strcmp (optarg, "57600")==0)
      {
        speed = Serial::BPS_57600;
      }
      else if (strcmp (optarg, "115200")==0)
      {
        speed = Serial::BPS_115200;
      }
      else
      {
        fprintf (stderr, "%s: bad baud rate (%s).\n", THIS, optarg);
        fprintf (stderr, "%s: use 9600,19200,38400 or 57600.\n", THIS, optarg);
        return 1;
      }
      break;
    case 'd':
      data_format = optarg;
      break;
    case 'e':
      ignore_error = true;
      break;
    case 'f':
      if (sscanf (optarg, "%lx%1s", &from, dummy) != 1)
      {
        fprintf (stderr, "%s: bad hexademical value (-f %s).\n", THIS, optarg);
        return 1;
      }
      address_specified = true;
      break;
    case 'i':
      id = optarg;
      break;
    case 'o':
      output_file = optarg;
      break;
    case 'p':
      port = optarg;
      break;
    case 'q':
      quiet = true;
      break;
    case 't':
      if (sscanf (optarg, "%lx%1s", &to, dummy) != 1)
      {
        fprintf (stderr, "%s: bad hexademical value (-t %s).\n", THIS, optarg);
        return 1;
      }
      address_specified = true;
      break;
    case 'v':
      fprintf (stderr, "%s: version %s\n", THIS, PROGRAM_VERSION);
      fprintf (stderr, "GNU Copyright (C) 2003 Gaspar Sinai <gsinai@yudit.org>.\n"); 
      return 0;
      break;
    default:
      usage (0);
      return 1;
    }
  }


  if (strcmp (command, "write-flash") != 0 
    && strcmp (command, "write-lock") != 0
    && strcmp (command, "execute-ram") != 0
    && optind < argc)
  {
    usage (0);
    return 1;
  }

  if (address_specified && strcmp (command, "read-flash") != 0
    && strcmp (command, "read-boot") !=0)
  {
     fprintf (stderr, "%s: Only read-flash and read-boot commands can take address range.\n", THIS);
     return 1;
  }

  // Quick Command Check.
  if (strcmp (command, "mcu-version")!=0
    && strcmp (command, "write-flash")!=0
    && strcmp (command, "read-flash")!=0
    && strcmp (command, "write-lock")!=0
    && strcmp (command, "execute-ram")!=0
    && strcmp (command, "read-lock")!=0
    && strcmp (command, "read-boot")!=0)
  {
    usage (0);
    return 1;
  }
  try
  {
    MemoryModel model(0x100000);
    if (strcmp (command, "write-flash") == 0 
      || strcmp (command, "execute-ram") == 0
      || strcmp (command, "write-lock") == 0)
    {
      if (optind < argc)
      {
        for (int i=optind; i<argc; i++)
        {
          if (!quiet) fprintf (stdout, "%s: reading %s\n", THIS, argv[i]);
          model.read (argv[i], data_format);
        }
      }
      else
      {
        if (isatty(0))
        {
          fprintf (stderr, "Enter data directly. Press ^D to end.\n");
        }
        model.read (0, data_format);
      }
    }
    Serial serial(port);
    if (!serial.setBaudRate(speed))
    {
      fprintf (stderr, "%s: Can not set baud rate.\n", THIS);
    }
    M16CFlasher flasher (serial, !quiet);
    if (!flasher.checkID(id))
    {
      fprintf (stderr, "%s: Bad ID - %s.\n", THIS, id);
      return 0;
    }
    if (strcmp (command, "mcu-version")==0)
    {
      fprintf (stdout, "%s: mcu-version='%s'\n", THIS, flasher.readVersion());
      return 0;
    }
    else if (strcmp (command, "read-flash")==0)
    {
      MemoryModel* model = flasher.read (from, to+1, ignore_error);
      if (!quiet) fprintf (stdout, "%s: writing file: %s.\n", THIS, output_file);
      model->write (output_file, data_format);
      delete model;
      return 0;
    }
    else if (strcmp (command, "read-lock")==0)
    {
      MemoryModel* model = flasher.readLock (ignore_error);
      if (!quiet) fprintf (stdout, "%s: writing file: %s.\n", THIS, output_file);
      model->write (output_file, data_format);
      delete model;
      return 0;
    }
    else if (strcmp (command, "read-boot")==0)
    {
      MemoryModel* model = flasher.readBoot (from, to+1, ignore_error);
      if (!quiet) fprintf (stdout, "%s: writing file: %s.\n", THIS, output_file);
      model->write (output_file, data_format);
      delete model;
      return 0;
    }
    else if (strcmp (command, "execute-ram")==0)
    {
      flasher.execute (model);
      return 0;
    }
    else if (strcmp (command, "write-lock")==0)
    {
      flasher.writeLock (model);
      return 0;
    }
    else if (strcmp (command, "write-flash")==0)
    {
      flasher.write (model);
      return 0;
    }
    fprintf (stderr, "%s: unkown command: %s\n", command);
    return 1;
  }
  catch (Exception e)
  {
    fprintf (stderr, "%s: error - %s\n", THIS, e.toString());
  }
  return 0;
}

static void usage (const char* message)
{
  fprintf (stderr, "usage %s [options] input-file1 [input-file2..]\n", THIS);
  fprintf (stderr, "options:\n");
  fprintf (stderr, " -q,--quiet                quiet-mode.\n");
  fprintf (stderr, " -v,--version              print program version.\n");
  fprintf (stderr, " -e,--ignore               ignore read errors.\n");
  fprintf (stderr, " -b,--baud     baud-rate   (57600)\n");
  fprintf (stderr, " -p,--port     port-name   (/dev/ttyS1)\n");
  fprintf (stderr, " -i,--id       id-string   (00:00:00:00:00:00:00)\n");
  fprintf (stderr, " -f,--from     address     (0xFE000)\n");
  fprintf (stderr, " -t,--to       address     (0xFFFFF)\n");
  fprintf (stderr, " -d,--format   data-format (hex)\n");
  fprintf (stderr, " -c,--command  command     (version)\n");
  fprintf (stderr, " -o,--out      output-file (flash.out)\n");
  fprintf (stderr, "commands:\n");
  fprintf (stderr, " mcu-version - print MPU version info and exit.\n");
  fprintf (stderr, " read-flash  - read memory\n");
  fprintf (stderr, " write-flash - write flash memory\n");
  fprintf (stderr, " execute-ram - upload a program into RAM and execute it.\n");
  fprintf (stderr, " write-lock  - write lock bit\n");
  fprintf (stderr, " read-boot   - read boot ROM\n");
  fprintf (stderr, " read-lock   - read lock bit\n");
}
