/*
 *  Flasher Source File
 *
 *  GNU Copyright (C) 2003 Gaspar Sinai <gsinai@yudit.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "m16c_flasher.h"
#include "simple_stream.h"

// Funny eh?
#include <stdio.h>


#define FLASHER_CODE_PROCESSING_FUNCTION      0xf5
#define FLASHER_CLEAR_STATUS_REGISTER         0x50
#define FLASHER_READ_STATUS_REGISTER          0x70
#define FLASHER_READ_LOCK_BIT_STATUS          0x71
#define FLASHER_BLOCK_ERASE                   0x20
#define FLASHER_LOCK_BIT_PROGRAM              0x77
#define FLASHER_LOCK_BIT_ENABLE               0x7a
#define FLASHER_LOCK_BIT_DISABLE              0x75
#define FLASHER_DOWNLOAD_FUNCTION             0xfa
#define FLASHER_VERSION_DATA_OUTPUT_FUNCTION  0xfb
#define FLASHER_READ_CHECK_DATA               0xfd
#define FLASHER_BOOT_ROM_AREA_OUTPUT_FUNCTION 0xfc
#define FLASHER_PAGE_READ                     0xff
#define FLASHER_PAGE_PROGRAM                  0x41

#define FLASHER_BLOCK_ERASE_VERIFY            0xd0

namespace FLASHER
{
  Logger logger ("M16CFlasher", stdout);

  /*!
   * \brief initialize the block model.
   */
  M16CBlockModel::M16CBlockModel(void)
  {
    /* 16 k */
    block[0].begin = 0x0fc000; block[0].end = 0x0fffff; block[0].flag=0;
    /* 8 k */
    block[1].begin = 0x0fa000; block[1].end = 0x0fbfff; block[1].flag=0;
    /* 8 k */
    block[2].begin = 0x0f8000; block[2].end = 0x0f9fff; block[2].flag=0;
    /* 32 k */
    block[3].begin = 0x0f0000; block[3].end = 0x0f7fff; block[3].flag=0;
    /* 64 k */
    block[4].begin = 0x0e0000; block[4].end = 0x0effff; block[4].flag=0;
    /* 64 k */
    block[5].begin = 0x0d0000; block[5].end = 0x0dffff; block[5].flag=0;
    /* 64 k */
    block[6].begin = 0x0c0000; block[6].end = 0x0cffff; block[6].flag=0;
  }

  /*!
   * \brief Destructor.
   */
  M16CBlockModel::~M16CBlockModel()
  {
  }

  /*!
   * \brief Get a reference to the block.
   * \param address is the address to which a block must be found.
   */
  M16CBlock& 
  M16CBlockModel::mapBlock(unsigned long address)
  {
    for (unsigned long i=0; i<7; i++)
    {
      if (block[i].begin<=address && address<=block[i].end)
      {
        return *(&block[i]);
      }
    }
    THROW ("Can not find block for address: " << StringStream(address,16));
  }

  /*!
   * \brief Get a reference to the block.
   * \param address is the index.
   */
  M16CBlock& 
  M16CBlockModel::getBlock(unsigned int index)
  {
    return block[index];
  }

  /**
   * \return the number of blocks.
   */
  unsigned int 
  M16CBlockModel::size() const
  {
    return 7;
  }
  

  /*!
   * \brief Construct a new M16CFlasher.
   * \param port is a port that should 
   *  exist throughout this object's lifetime.
   * \param verbose is tru if all messages are printed.
   */ 
  M16CFlasher::M16CFlasher (Serial& port, bool verbose)
    : m_port (port)
  {
    m_verbose = verbose;
    m_port.wait (20);
    if (m_verbose)
    {
      logger.log ((StringStream)"Initializing Link " << port.getName() << ".");
    }
    Serial::BaudRate baud = m_port.getBaudRate();

    unsigned char uclock = getBaudCommand(m_port.getBaudRate());
    unsigned char clock = 0;
    try {
      sendCommand (uclock);
      clock = receiveByte ();
    } 
    catch (const Exception& e)
    {
    }
    if (clock != uclock && baud != Serial::BPS_9600)
    {
      clock = changeBaudRate (Serial::BPS_9600, baud);
    } 
    if (clock != uclock && baud != Serial::BPS_19200)
    {
      clock = changeBaudRate (Serial::BPS_19200, baud);
    } 
    if (clock != uclock && baud != Serial::BPS_38400)
    {
      clock = changeBaudRate (Serial::BPS_38400, baud);
    } 
    if (clock != uclock && baud != Serial::BPS_57600)
    {
      clock = changeBaudRate (Serial::BPS_57600, baud);
    } 
    if (clock != uclock && baud != Serial::BPS_115200)
    {
      clock = changeBaudRate (Serial::BPS_115200, baud);
    } 

    if (clock != uclock)
    {
      THROW ("Can not initialize at " << (unsigned int) baud << "bps.");
    }
    if (m_verbose)
    {
      logger.log ((StringStream)"Link Initialized at " << (unsigned int) baud << "bps.");
      logger.log ((StringStream)"MCU Version Info: " << readVersion());
    }
  }

  /*!
   * \brief Destructor for M16CFlasher.
   */
  M16CFlasher::~M16CFlasher()
  {
  }

  /*!
   * \brief Check the id. 
   * \param id is the id in the format of
   *  id1:id2:id3:id4:id5:id6:id7
   * where ids are 8-bit bytes in hexadecimal format.
   */
  bool
  M16CFlasher::checkID (const char* id)
  {
    unsigned int iid[7];
    char dummy[2];
    if (sscanf (id, "%2x:%2x:%2x:%2x:%2x:%2x:%2x%1s",
      &iid[0], &iid[1], &iid[2], &iid[3], 
      &iid[4], &iid[5], &iid[6], dummy) != 7)
    {
      THROW ("Bad id string: " << id);
    }

    sendCommand (FLASHER_CODE_PROCESSING_FUNCTION);

    // m16c20 - 0x0fffdf
    // 380    - 0x00ffd4
    // r8c    - 0x00ffdf
    sendAddress (0x0fffdf); // ID address

    sendByte (0x7); // ID size
    for (unsigned int i=0; i<7; i++)
    {
      sendByte ((unsigned char) iid[i]);
    }

    // If we can clear the status it worked.
    sendCommand (FLASHER_CLEAR_STATUS_REGISTER);
    readStatus();
    if (m_status[1].bits.bit2 == 0 || m_status[1].bits.bit3 == 0)
    {
      return false;
    }
    if (m_verbose)
    {
      logger.log ((StringStream)"Checking ID: " << id << " OK." );
    }
    return true;
  }

  /*!
   * \brief Read the chips's memory and return it.
   * \param from is the starting address.
   * \param to is the non-inclusive ending address.
   * \param ie is true if we are supposed to ignore read-errors.
   */
  MemoryModel*
  M16CFlasher::read (unsigned long from, unsigned long until, bool ie)
  {
    MemoryModel* model = new MemoryModel(until+0x100);
    if (model ==0)
    {
      THROW ("Not Enough Memory");
    }
    unsigned long alignedFrom = (from/0x100) * 0x100;
    if (m_verbose)
    {
      logger.log ((StringStream)"Reading pages from: " 
         << StringStream (alignedFrom, 16));
    }
    for (unsigned long i=alignedFrom;  i<until; i+=0x100)
    {
      sendCommand (FLASHER_PAGE_READ);
      sendByte ((i>>8)&0xff);
      sendByte ((i>>16)&0xff);
      for (unsigned long j=0; j<0x100; j++)
      {
        unsigned char c = 0;
        try
        {
          c = receiveByte();
        }
        catch (const Exception& e)
        {
          fprintf (stderr, "M16CFlasher: Error reading from: %06X.\n", i+j);
          if (i+j>=until)
          {
            fprintf (stderr, "M16CFlasher: Data not needed. Ignoring error.\n");
            break;
          }
          if (ie)
          {
            fprintf (stderr, "M16CFlasher: Ignoring error.\n");
            break;
          }
          throw;
        }
        if (i+j>=from && i+j<until) model->set (i+j, c);
      }
      if (m_verbose)
      {
        logger.log ((StringStream)".", false);
        logger.flush();
      }
    }
    if (m_verbose) logger.log ((StringStream)"Done.\n", false);
    return model;
  }

  /*!
   * \brief Read the chips's boot-memory and return it.
   * \param from is the starting address.
   * \param to is the non-inclusive ending address.
   * \param ie is true if we are supposed to ignore read-errors.
   */
  MemoryModel*
  M16CFlasher::readBoot (unsigned long from, unsigned long until, bool ie)
  {
    MemoryModel* model = new MemoryModel(until+0x100);
    if (model ==0)
    {
      THROW ("Not Enough Memory");
    }
    unsigned long alignedFrom = (from/0x100) * 0x100;
    if (m_verbose)
    {
      logger.log ((StringStream)"Reading pages from: " 
         << StringStream (alignedFrom, 16));
    }
    for (unsigned long i=alignedFrom;  i<until; i+=0x100)
    {
      sendCommand (FLASHER_BOOT_ROM_AREA_OUTPUT_FUNCTION);
      sendByte ((i>>8)&0xff);
      sendByte ((i>>16)&0xff);
      for (unsigned long j=0; j<0x100; j++)
      {
        unsigned char c = 0;
        try
        {
          c = receiveByte();
        }
        catch (const Exception& e)
        {
          fprintf (stderr, "\nM16CFlasher: error reading from: %06X\n", i+j);
          if (ie)
          {
            fprintf (stderr, "M16CFlasher: ignoring error");
            break;
          }
          throw;
        }
        if (i+j>=from && i+j<until) model->set (i+j, c);
      }
      if (m_verbose)
      {
        logger.log ((StringStream)".", false);
        logger.flush();
      }
    }
    if (m_verbose) logger.log ((StringStream)"Done.\n", false);
    return model;
  }
  /*!
   * \brief Read the lock status.
   * \param ie is true if we are supposed to ignore read-errors.
   * \return a memory model with lock bits.
   */
  MemoryModel*
  M16CFlasher::readLock (bool ie)
  {
    M16CBlockModel bm;
    MemoryModel* model = new MemoryModel(bm.getBlock(0).begin+1);
    if (model ==0)
    {
      THROW ("Not Enough Memory");
    }
    if (m_verbose)
    {
      logger.log("ReadLock "); 
    }
    for (unsigned int i=0; i<bm.size(); i++)
    {
      M16CBlock& block = bm.getBlock(i);
      sendCommand (FLASHER_READ_LOCK_BIT_STATUS);
      sendByte ((block.begin>>8)&0xff);
      sendByte ((block.begin>>16)&0xff);
      unsigned char c = 0;
      try
      {
        c = receiveByte();
      }
      catch (const Exception& e)
      {
        fprintf (stderr, "\nM16CFlasher: error reading block: %d\n", i);
        if (ie)
        {
          fprintf (stderr, "M16CFlasher: ignoring error");
          break;
        }
        throw;
      }
      model->set (block.begin, c);
      if (m_verbose)
      {
        logger.log (StringStream("block[") << StringStream (i) << "]=" 
            << StringStream ((unsigned long)c, 16)
            << " ("
            << StringStream ((unsigned long)block.begin, 16)
            << ".."
            << StringStream ((unsigned long)block.end, 16)
            << ")"
            );
      }
    }
    if (m_verbose) logger.log ((StringStream)"Done.");
    return model;
  } 

  /*!
   * \brief Load the memory-model into the RAM and execute it.
   *        The program is expected to be re-locatable, as we don't 
   *        know where the program will be loaded. Most likely, it will
   *        be loaded to 0x400 RAM area.
   * \param model is the memory model.
   */
  void
  M16CFlasher::execute (const MemoryModel& model) 
  {
    unsigned int checksum = 0;
    unsigned long i;
    unsigned long from = 0;
    unsigned long until = 0x100000;
    for (i=from; i<until; i++)
    {
      if (model.isSet (i))
      {
        from = i;
        break;
      }
    }
    for (i=until; i>from; i--)
    {
      if (model.isSet (i)) break;
      until = i;
    }
    unsigned long size=until-from;
    if (m_verbose)
    {
      logger.log (StringStream("uploading and executing program from: 0x")
        << StringStream (from, 16)
       << " size: 0x" << StringStream (size, 16));
    }
    if (from == until)
    {
      THROW ("Nothing to execute.");
    }
    for (i=from; i<until; i++)
    {
      checksum += model.get (i);
    }
    sendCommand (FLASHER_CLEAR_STATUS_REGISTER);
    sendCommand (FLASHER_DOWNLOAD_FUNCTION);
    sendByte (size & 0xff);
    sendByte ((size>>8) & 0xff);
    sendByte (checksum & 0xff);

    logger.flush();
    for (i=from; i<until; i++)
    {
      sendByte (model.get (i));
      if (m_verbose && (i-from)%0x100==0)
      {
        logger.log ((StringStream)".", false);
        logger.flush();
      }
    }
    if (m_verbose) logger.log ((StringStream)"Done.\n", false);
/*
    readStatus();
    if (m_status[1].bits.bit4 == 0)
    {
      THROW ("CheckSum error");
    }
*/
  }

  /*!
   * \brief Lock bit program. The 6th bit (0x40) should be set for lock.
   * \param model is the memory model.
   */
  void
  M16CFlasher::writeLock (const MemoryModel& model) 
  {
    M16CBlockModel bm;
    M16CBlockModel bset;
    unsigned long i;
    if (m_verbose)
    {
      logger.log((StringStream)"WriteLock"); 
    }
    bool any = false;
    for (i=0x0c0000; i<=0x0fffff; i++)
    {
      if (!model.isSet (i)) continue;
      M16CBlock& b = bm.mapBlock (i);
      M16CBlock& bflag = bset.mapBlock (i);
      if (bflag.flag==0)
      {
        bflag.flag = 1;
        b.flag = model.get (i);
        any = true;
      }
      else
      {
        if (b.flag != model.get (i))
        {
          THROW ("Block has inconsistent/clashing values at address " 
            << StringStream (i, 16));
        }
      }
    }
    if (!any)
    {
      if (m_verbose)
      {
        logger.log("Nothing to do."); 
      }
      return;
    }
    sendCommand (FLASHER_LOCK_BIT_DISABLE);
    for (i=0; i<bm.size(); i++)
    {
      M16CBlock& block = bm.getBlock(i);
      M16CBlock& bflag = bset.getBlock(i);
      if (bflag.flag==0) continue;
      sendCommand (FLASHER_LOCK_BIT_PROGRAM);
      sendByte ((block.begin>>8)&0xff);
      sendByte ((block.begin>>16)&0xff);
      sendByte (block.flag);
      if (m_verbose)
      {
        logger.log (StringStream("block[") << StringStream (i) << "]=" 
            << StringStream ((unsigned long)block.flag, 16)
            << " ("
            << StringStream ((unsigned long)bflag.begin, 16)
            << ".."
            << StringStream ((unsigned long)bflag.end, 16)
            << ")");
      }
    }
    sendCommand (FLASHER_LOCK_BIT_ENABLE);
    if (m_verbose) logger.log ((StringStream)"Done");
  }
  /*!
   * \brief write pages.
   * \param model is the memory model.
   */
  void
  M16CFlasher::write (const MemoryModel& model) 
  {
    M16CBlockModel bset;
    unsigned long i;
    if (m_verbose)
    {
      logger.log((StringStream)"Write Pages"); 
    }
    bool any = false;
    for (i=0x0c0000; i<=0x0fffff; i++)
    {
      if (!model.isSet (i)) continue;
      M16CBlock& bflag = bset.mapBlock (i);
      if (bflag.flag==0)
      {
        bflag.flag = 1;
        any = true;
      }
    }
    if (!any)
    {
      if (m_verbose)
      {
        logger.log("Nothing to do."); 
      }
      return;
    }
    /* erase blocks. */
    sendCommand (FLASHER_CLEAR_STATUS_REGISTER);
    readStatus();
    //fprintf (stderr, "Status: %02x %02x\n", m_status[0].byte, m_status[1].byte);

    sendCommand (FLASHER_LOCK_BIT_DISABLE);
    for (i=0; i<bset.size(); i++)
    {
      M16CBlock& bflag = bset.getBlock(i);
      if (bflag.flag==0) continue;

      sendCommand (FLASHER_BLOCK_ERASE);
      sendByte ((bflag.begin>>8)&0xff);
      sendByte ((bflag.begin>>16)&0xff);
      sendByte (FLASHER_BLOCK_ERASE_VERIFY);
      if (m_verbose)
      {
        logger.log (StringStream("Erasing block[") 
            << StringStream (i) << "] " 
            << " ("
            << StringStream ((unsigned long)bflag.begin, 16)
            << ".."
            << StringStream ((unsigned long)bflag.end, 16)
            << ")");
        if (i==0)
        {
          logger.log (StringStream("Reset ID temporarily to: ")
             << StringStream("ff:ff:ff:ff:ff:ff:ff")
             << StringStream("."));
        }
      }
    }
    readStatus();
    // M16C6N_ds.pdf Page 257.
    if (m_status[0].bits.bit5 == 1)
    {
      sendCommand (FLASHER_CLEAR_STATUS_REGISTER);
      THROW ("Erase Block Failed.");
    }
    if (m_verbose)
    {
      logger.log ((StringStream)"Blocks Erased.");
    }
    // Init checksum.
    sendCommand (FLASHER_READ_CHECK_DATA);
    unsigned short c0 = (unsigned short) receiveByte (); 
    unsigned short c1 = (unsigned short) receiveByte (); 

    unsigned short checksum = 0;

    //sendCommand (FLASHER_LOCK_BIT_DISABLE);
    //m_port.consume (100);
    for (i=0x0c0000; i<=0x0fffff; i += 0x100)
    {
      // Check if block is set
      unsigned long j;
      for (j=0; j<0x100; j++)
      {
        if (model.isSet (i+j)) break;
      }
      if (j==0x100) continue;
      // Program it.
      if (m_verbose)
      {
        logger.log ((StringStream)"Writing: " 
          << StringStream(i, 16)
          << ".");
      }
      if (i==0xfff00 && m_verbose)
      {
        char  p[32];
        sprintf (p, "%02x:%02x:%02x:%02x:%02x:%02x:%02x", 
          model.get(0xfffdf), 
          model.get(0xfffe3), 
          model.get(0xfffeb), 
          model.get(0xfffef), 
          model.get(0xffff3), 
          model.get(0xffff7), 
          model.get(0xffffb)); 
        logger.log (StringStream("WARNING: New ID will be: ")
           << StringStream(p)
           << StringStream("."));
      }

      sendCommand (FLASHER_PAGE_PROGRAM);
      sendCommand ((i>>8)&0xff);
      sendCommand ((i>>16)&0xff);

      //checksum += (FLASHER_PAGE_PROGRAM);
      //checksum += ((i>>8)&0xff);
      //checksum += ((i>>16)&0xff);
      //checksum += ((i>>8)&0xffff);
      for (j=0; j<0x100; j++)
      {
        unsigned char data = model.get(i+j); 
        sendByte (data);
        checksum += (unsigned short) data;
      } 
    }
    //sendCommand (FLASHER_LOCK_BIT_ENABLE);
    readStatus();

    // M16C6N_ds.pdf Page 257.
    if (m_status[0].bits.bit4 == 1 || m_status[0].bits.bit3 == 1)
    {
      sendCommand (FLASHER_CLEAR_STATUS_REGISTER);
      THROW (StringStream("Write Block Failed"));
    }
    sendCommand (FLASHER_READ_CHECK_DATA);
    c0 = (unsigned short) receiveByte (); 
    c1 = (unsigned short) receiveByte (); 
    // It is a CRC and not a checksum.
    // Unfortunatelly I could not find any info on what kind of CRC
    // calculation m16c is using.
    // FIXME
#if 0
    if ((c1 << 8) + c0 != checksum)
    {
      fprintf (stderr, "%04X != %04X\n", (c1 << 8) + c0, checksum);
      THROW (StringStream("Write Block Failed - Checksum Mismatch"));
    } 
#endif
  }

  /*!
   * \brief change the baud rate of the socket and the chip.
   * \param from is the baud rate the is assumed to be on the chip.
   * \param to is the required baud rate.
   * \return the command value that corresponds to to on success, 
   *  zero on failure.
   */
  unsigned char
  M16CFlasher::changeBaudRate (Serial::BaudRate from, Serial::BaudRate to)
  {
    unsigned char uclock = getBaudCommand (to);
    if (m_verbose)
    {
      logger.log ((StringStream)"Trying to change Baud Rate "
        << "from " << (unsigned int) from << "bps "
        << "to " << (unsigned int) to << "bps...");
    }
    m_port.setBaudRate (from);
    m_port.wait (40);
    for (unsigned int i=0; i<0x20; i++)
    {
      m_port.wait (20);
      sendByte (0);
    }

    // Set us to new baud rate.
    sendCommand (uclock);
    m_port.consume (40);

    m_port.setBaudRate (to);
    m_port.consume (40);

    for (unsigned int i=0; i<0x20; i++)
    {
      m_port.wait (20);
      sendByte (0);
    }
    m_port.consume (20);
    sendCommand (uclock);
    unsigned char clock = 0;
    try {
      clock = receiveByte ();
    }
    catch (const Exception& e)
    {
    }
    return clock;
  }

  /*!
   * \brief Send a command to the port.
   * \param command is the command code.
   */
  void
  M16CFlasher::sendCommand (unsigned char command)
  {
    m_port.consume (20);
    m_port.write (command);
  }

  /*!
   * \brief Send a byte to the port.
   * \param byte is the byte to send
   */
  void
  M16CFlasher::sendByte (unsigned char byte)
  {
    //m_port.wait (20);
    m_port.write (byte);
  }

  /*!
   * \brief write a full address 3x8 bits.
   */
  void
  M16CFlasher::sendAddress (unsigned long address)
  {
    unsigned char low = (address & 0xff);
    unsigned char middle = ((address >> 8) & 0xff);
    unsigned char high = ((address >> 16) & 0xff);
    sendByte (low);
    sendByte (middle);
    sendByte (high);
  }

  /*!
   * \brief write a short address 2x8 bits.
   */
  void
  M16CFlasher::sendShortAddress (unsigned short address)
  {
    unsigned char middle = ((address) & 0xff);
    unsigned char high = ((address>>8) & 0xff);
    sendByte (middle);
    sendByte (high);
  }

  /*!
   * \brief Read the status registers.
   */
  void
  M16CFlasher::readStatus()
  {
    sendCommand (FLASHER_READ_STATUS_REGISTER);
    m_status[0].byte = receiveByte();
    m_status[1].byte = receiveByte();
  }

  /*!
   * \return the version string. 
   */
  const char*
  M16CFlasher::readVersion()
  {
    sendCommand (FLASHER_VERSION_DATA_OUTPUT_FUNCTION);
    for (unsigned int i=0; i<8; i++)
    {
      m_version[i] = receiveByte();
    }
    m_version[8] = 0;
    return (const char*) m_version;
  }

  /*!
   * \brief Reveieve a byte from the port.
   * \return the received byte
   */
  unsigned char
  M16CFlasher::receiveByte ()
  {
    return m_port.read ();
  }

  /*!
   * \brief Extract the baud rate from Serial and 
   *        convert it to M16C command string.
   * \param baud is the baud rate.
   * \return The command string that can be sent to MCU.
   */
  unsigned char
  M16CFlasher::getBaudCommand(Serial::BaudRate baud) const
  {
    unsigned char ret = 0xb0;
    switch (baud)
    {
    case    Serial::BPS_9600: ret = 0xb0; break;
    case   Serial::BPS_19200: ret = 0xb1; break;
    case   Serial::BPS_38400: ret = 0xb2; break;
    case   Serial::BPS_57600: ret = 0xb3; break;
    case  Serial::BPS_115200: ret = 0xb4; break;
    }
    return ret;
  }
}

