/*
 *  Flasher Source File
 *
 *  GNU Copyright (C) 2003 Gaspar Sinai <gsinai@yudit.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "memory_model.h"

#include "exception.h"

#include <string.h>

// Set the page size to 4096 bytes.
#define FLASHER_PAGE_SIZE 0x1000

namespace FLASHER
{
  /*!
   * \brief Construct a continuous page.
   * \param page_size is the size of the page.
   */
  Page::Page (unsigned long page_size) : m_bitmap (page_size)
  {
    m_data = new unsigned char[page_size];
    if (m_data==0) THROW ("Not Enough Memory");
    m_size = page_size;
  }
 
  /*!
   * \brief Destruct a page.
   */
  Page::~Page()
  {
    delete [] m_data;
  }

  /*!
   * \brief Set the page at location.
   * \param at is the page location.
   * \param vle is the value to set.
   */
  void
  Page::set (unsigned long at, unsigned char vle)
  {
    if (at >= m_size) THROW ("Array Bound");
    m_data[at] = vle;
    m_bitmap.set (at);
  }

  /*! 
   * \brief Get the value from a page location.
   * \param at is the page location.
   * \param unset_vle is returned if the page location has not
   *    been set via the set method.
   */
  unsigned char
  Page::get (unsigned long at, unsigned char unset_vle) const
  {
    if (at >= m_size) THROW ("Array Bound");
    if (!isSet(at)) return unset_vle;
    return m_data[at];
  }

  /*!
   * \brief Check if a page location has been set.
   * \param at is the page location.
   * \return true if the page location has been set via the set method.
   */
  bool
  Page::isSet (unsigned long at) const
  {
    if (at >= m_size) THROW ("Array Bound");
    return m_bitmap.get(at);
  }
  
  /*! 
   * \brief create a new MemoryModel.
   * \param size is the memory size limit, exclusive.
   */
  MemoryModel::MemoryModel (unsigned long size)
  {
    unsigned long array_size = (size+FLASHER_PAGE_SIZE-1)/FLASHER_PAGE_SIZE;

    m_pages = new Page*[array_size];
    if (m_pages==0) THROW ("Not Enough Memory");
    memset (m_pages, 0, array_size * sizeof (Page*));
    m_size = size;
  }

  /*!
   * \brief Destroy this memory model.
   */
  MemoryModel::~MemoryModel()
  {
    unsigned long array_size = (m_size+FLASHER_PAGE_SIZE-1)/FLASHER_PAGE_SIZE;
    for (unsigned long i=0; i<array_size; i++)
    {
      if (m_pages[i]) delete m_pages[i];
    }
  }

  /*!
   * \brief Set the memory at location.
   * \param at is the memory location.
   * \param vle is the value to set.
   */
  void
  MemoryModel::set (unsigned long at, unsigned char vle)
  {
    if (at >= m_size) THROW ("Array Bound");
    unsigned long page = at/FLASHER_PAGE_SIZE;
    if (m_pages[page]==0)
    {
      m_pages[page] = new Page(FLASHER_PAGE_SIZE);
      if (m_pages[page]==0) THROW ("Not Enough Memory");
    }
    unsigned long location = at%FLASHER_PAGE_SIZE;
    m_pages[page]->set (location, vle);
  }

  /*! 
   * \brief Get the value from a memory location.
   * \param at is the memory location.
   * \param unset_vle is returned if the memory location has not
   *    been set via the set method.
   */
  unsigned char
  MemoryModel::get (unsigned long at, unsigned char unset_vle) const
  {
    if (at >= m_size) THROW ("Array Bound");
    unsigned long page = at/FLASHER_PAGE_SIZE;
    if (m_pages[page]==0) return unset_vle; 
    unsigned long location = at%FLASHER_PAGE_SIZE;
    return m_pages[page]->get(location, unset_vle);
  }

  /*!
   * \brief Check if a memory location has been set.
   * \param at is the memory location.
   * \return true if the memory location has been set via the set method.
   */
  bool
  MemoryModel::isSet (unsigned long at) const
  {
    if (at >= m_size) return false;
    unsigned long page = at/FLASHER_PAGE_SIZE;
    if (m_pages[page]==0) return false; 
    unsigned long location = at%FLASHER_PAGE_SIZE;
    return m_pages[page]->isSet (location);
  }

  /*!
   * \brief Write the memory model into a file.
   * \param fileName is the file to write.
   * \param format is the file format. Currently
   *   recognized formats:
   *     \li hex
   * \throw exception on error.
   */
  void
  MemoryModel::write (const char* fileName, const char* format)
  {
    FILE* out = fopen (fileName, "w+");
    if (out==0)
    {
      THROW ("Can not open file for writing - " << fileName);
    }
    if (strcasecmp (format,"hex")==0)
    {
      unsigned long i=0;
      while (i<m_size)
      {
        if (!isSet (i))
        {
          i++;
          continue;
        }
        if (fprintf (out, "$A0%06X,", i) < 8)
        {
          THROW ("Error while writing - " <<  fileName);
        }
        unsigned long j=0;
        while (j<0x10 && i+j<m_size)
        {
          bool st = false;
          for (unsigned long k=j; k<0x10; k++)
          {
             st |= isSet (i+k);
             if (st) break;
          }
          if (!st) break;
          if (fprintf (out, " %02X", get(i+j)) < 3)
          {
            THROW ("Error while writing - " <<  fileName);
          }
          j++;
        }
        if (fprintf (out, "\n") < 1)
        {
          THROW ("Error while writing - " <<  fileName);
        }
        i+=j;
      }
      if (fclose (out)!=0)
      {
        THROW ("Error while writing - " <<  fileName);
      }
    }
    else 
    {
     THROW ("Unkwon file format - " <<  format);
    }
  }
  /*!
   * \brief Read the memory model from a file.
   * \param fn is the file to read. Null means tsdin should be used.
   * \param format is the file format. Currently
   *   recognized formats:
   *     \li hex
   * \throw exception on error.
   */
  void
  MemoryModel::read (const char* fn, const char* format)
  {
    FILE* in = (fn==0) ? stdin : fopen (fn, "r");
    const char* fileName = (fn==0) ? "stdin" : fn;
    if (in==0)
    {
      THROW ("Can not open file for reading - " << fileName);
    }
    if (strcasecmp (format,"hex")==0)
    {
      int next;
      unsigned long address=0;
      unsigned int data;
      unsigned int line=0;
      while ((next=fgetc(in))!=EOF)
      {
        if (next == '\n') line;
        if (next <= ' ') continue;
        if (next == '$')
        {
          next=fgetc(in);
          if (next != 'A')
          {
            THROW ("Address should start with $A0 in " <<  fileName << " line:" << line);
          }
          next=fgetc(in);
// Too many checks.
#if 0
          if (next != '0')
          {
            THROW ("Error while reading address - " <<  fileName << " line:" << line);
          }
#endif
          if (fscanf (in, "%lx,", &address)==0)
          {
            THROW ("Address in not hexadecimal in " <<  fileName << " line:" << line);
          }
          continue;
        }
        ungetc (next, in);
        if (fscanf (in, "%x", &data)==0)
        {
          THROW ("Error when reading data: " <<  fileName << " line:" << line);
        }
        if (data > 0xff)
        {
          THROW ("Data is not hexadecimal " <<  fileName << " line: " << line << " data: " << StringStream (data, 16));
        }
        set (address, (unsigned char)data);
//SGC
//fprintf (stderr, "%lx: %02x\n", address, data);
        address++;
      }
      fclose (in);
    }
    else 
    {
     THROW ("Unkwon file format - " <<  format);
    }
  }
}

