/*
 *  Flasher Source File
 *
 *  GNU Copyright (C) 2003 Gaspar Sinai <gsinai@yudit.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "serial.h"

#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <assert.h>

#ifdef USE_WINAPI
#include <time.h>
#else 
#include <signal.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/time.h>
#endif

// Missing from Sun's headers.
#ifdef __sun__
#ifndef __svr4__
extern "C"
{
int select (int width, fd_set* readfds, fd_set* writefds,
            fd_set* exceptfds, struct timeval* timeout);
}
#endif
#endif



namespace FLASHER
{
  /*!
   * \brief Create a Serial Port class.
   * \param port is opened with 9600 baud 8N1 parameters.
   * \exception FLASHER::Exception is thrown on any error.
   */
  Serial::Serial (const char* port)
  {
    const char* l_port =  (port == 0) ? "/dev/ttyS1" : port; 
    m_port = StringStream (l_port);

    m_fd = open (l_port, O_RDWR | O_NOCTTY | O_NDELAY);
#ifdef USE_WINAPI
    if (m_fd == 0)
#else
    if (m_fd == -1)
#endif
    {
      THROW ("Can not open: " << l_port);
    }

    // Blocking read.
    fcntl(m_fd, F_SETFL, 0);
    setBaudRate (BPS_9600);
  }
 
  /*!
   * \brief Set the baud rate.
   * \param baud is the baud rate.
   * \return true on succes, false otherwise. 
   */
  bool
  Serial::setBaudRate (BaudRate baud)
  {
    speed_t speed = B9600;
    switch (baud)
    {
    case    BPS_9600:
      speed = B9600;
      break;
    case   BPS_19200:
      speed = B19200;
      break;
    case   BPS_38400:
      speed = B38400;
      break;
    case   BPS_57600:
      speed = B57600;
      break;
    case  BPS_115200:
      speed = B115200;
      break;
    }

    termios tios;
    tcgetattr(m_fd, &tios);

    // 8 bit, no parity, 1 stop bit 
    tios.c_cflag = (CS8 | CLOCAL | CREAD);
    tios.c_iflag = (IGNBRK | IGNPAR);
    tios.c_lflag = 0; 
    tios.c_oflag = 0; 

    tios.c_cc[VMIN]  = 1;  // wait for 1 character
    tios.c_cc[VTIME] = 0;  // turn off timer

    cfsetispeed(&tios, speed);
    cfsetospeed(&tios, speed);


#ifdef __linux__
    /* set line discipline - for linux only */
    tios.c_line      = N_TTY;
#endif

    bool ret = (tcsetattr (m_fd, TCSADRAIN, &tios)==0);
    if (ret) m_baud = baud;
    return ret;
  }

  /*!
   * \return the current badu rate.
   */
  Serial::BaudRate
  Serial::getBaudRate () const
  {
    return m_baud;
  }

  /*!
   * \brief Desctruct this object.
   */
  Serial::~Serial()
  {
    close (m_fd);
  }

  /*!
   * \brief Read a single byte.
   * \param timout is the timeout in milliseconds, after which an
   *   Exception is thrown. If this timeout is zero, this routine
   *   will block until a byte is read.
   * \exception FLASHER::Exception is thrown on timeout.
   */
  unsigned char
  Serial::read (unsigned int timeout)
  {
    if (timeout == 0)
    {
      unsigned char buff;
      if (::read(m_fd, (char*) &buff, 1) != 1)
      {
        THROW ("Read Failed on " << m_port);
      }
      return buff;
    }

    struct timeval t;
    t.tv_usec = (timeout % 1000)*1000;
    t.tv_sec = timeout/1000;

    fd_set ro;
    FD_ZERO (&ro);
    FD_SET (m_fd, &ro); 

    int s = select (m_fd+1, &ro, 0, 0, &t);
    if (!FD_ISSET (m_fd, &ro))
    {
      THROW ("Read Timeout after " << timeout << " milliseconds on port " << m_port);
    }
    unsigned char buff;
    if (::read(m_fd, (char*) &buff, 1) != 1)
    {
      THROW ("Read Failed on " << m_port);
    }
    return buff;
  }

  /*!
   * \brief Write a single byte.
   * \exception FLASHER::Exception is thrown on any error.
   */
  void
  Serial::write  (unsigned char byte)
  {

    if (::write (m_fd, (const char*)&byte, 1)!=1)
    {
      THROW ("Can not write: " << m_port);
    }
  }

  /*!
   * \brief Consume all input.
   * \param timeout is the time in milliseconds
   *  after which we assume that no bytes will 
   *  be sent.
   */
  void
  Serial::consume  (unsigned int  timeout)
  {
    bool found = true;
    while (found)
    {
      found = false;
      try {
        read (timeout);
        found = true;
      }
      catch (const Exception& e)
      {
      }
    }
  }

  /*!
   * \brief wait
   * \param ms is the wait time in milliseconds.
   */
  void
  Serial::wait (unsigned int ms)
  {
    struct timeval t;
    t.tv_usec = (ms % 1000)*1000;
    t.tv_sec = ms/1000;
    select (0, 0, 0, 0, &t);
  }

  /*!
   * \return the name of the port.
   */
  const char*
  Serial::getName() const
  {
    return (const char*) m_port;
  }
}
