/*
 *  Flasher Source File
 *
 *  GNU Copyright (C) 2003 Gaspar Sinai <gsinai@yudit.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "simple_stream.h"
#include "exception.h"

#include <stdio.h>
#include <string.h>

namespace FLASHER
{
  /*!
   * \brief Construct an empty stream.
   */
  StringStream::StringStream(void)
  {
    m_size = 0;
    m_buffer = new char[1];
    if (m_buffer==0) THROW ("Memory Full.");
    m_buffer[m_size] = 0;
  }

  /*!
   * \brief Construct a stream string from.
   * \param str is a null-terminated string.
   */
  StringStream::StringStream(const char* str)
  {
    m_size = strlen (str);
    m_buffer = new char[m_size+1];
    if (m_buffer==0) THROW ("Memory Full.");
    memcpy (m_buffer, str, m_size);
    m_buffer[m_size] = 0;
  }

  /*!
   * \brief Construct a stream string from.
   * \param d will be represented as a decimal number. 
   */
  StringStream::StringStream (long d)
  {
    m_buffer = new char[64];
    sprintf (m_buffer, "%ld", d);
    m_buffer[63] = 0;
    m_size = strlen (m_buffer);
  }

  /*!
   * \brief Create a string representation with radix.
   * \param d is the number
   * \param radix is the radix.
   */
  StringStream::StringStream (unsigned long d, unsigned int radix)
  {
    if (radix > 10 + ('Z' - 'A'))
    {
      THROW ("Bad Radix: " << radix);
    }
    // Calculate the length.
    unsigned int length = 0;
    unsigned long vle=d;
    do 
    {
      length++;
    } while (vle /= radix);

    m_size = length;
    m_buffer = new char[m_size+1];
    if (m_buffer==0) THROW ("Memory Full.");
    m_buffer[length+1] = 0;

    // Second iteration puts the actual values there.
    vle=d;
    do
    {
      unsigned int v0 = vle % radix;
      v0 =  (v0 < 10) ? ('0' + v0) : 'A' + (v0-10);
      m_buffer[--length] = (char) v0;
    } while (vle /= radix); 
  }

  /*!
   * \brief Destroy this string.
   */
  StringStream::~StringStream()
  {
    delete m_buffer;
  }

  /*!
   * \brief Copy a new StringStream.
   * \param str is the string to copy.
   */
  StringStream::StringStream (const StringStream& str)
  {
    m_size = str.m_size;
    m_buffer = new char[m_size+1];
    if (m_buffer==0) THROW ("Memory Full.");
    memcpy (m_buffer, str.m_buffer, m_size+1);
  }

  /*!
   * \brief Assign a new StringStream.
   * \param str is the string to assign.
   */
  StringStream&
  StringStream::operator=(const StringStream& str)
  {
    if (this == &str) return *this;

    delete [] m_buffer;
    m_size = str.m_size;
    m_buffer = new char[m_size+1];
    if (m_buffer==0) THROW ("Memory Full.");
    memcpy (m_buffer, str.m_buffer, m_size+1);
    return *this;
  }

  /*!
   * \brief Append a stream string to this stream string.
   * \param str is the string to append.
   */
  StringStream&
  StringStream::operator << (const StringStream& str)
  {
    char* buffer = m_buffer;
    unsigned int size = m_size;
    m_size += str.m_size;
    m_buffer = new char[m_size+1];
    if (m_buffer==0) THROW ("Memory Full.");

    memcpy (m_buffer, buffer, size);
    memcpy (&m_buffer[size], str.m_buffer, str.m_size);
    m_buffer[m_size] = 0;

    delete [] buffer;
    return *this;
  }

  /*!
   * \return the null-terminated string.
   */
  StringStream::operator const char*() const
  {
    return m_buffer;
  }
  /*!
   * \brief Compare this string to another string.
   * \param str is the other string.
   * \return true if the two strings are equal.
   */
  bool
  StringStream::operator == (const  StringStream& str) const
  {
    if (m_size != str.m_size) return false;
    return (memcmp (m_buffer, str.m_buffer, m_size)==0);
  }

  /*!
   * \brief Compare this string to another string.
   * \param str is the other string.
   * \return true if the two strings are not equal.
   */
  bool
  StringStream::operator != (const  StringStream& str) const
  {
    if (m_size != str.m_size) return true;
    return (memcmp (m_buffer, str.m_buffer, m_size)!=0);
  }

  /*!
   * \brief Create a new Logger.
   * \param prefix will prefix each line.
   * \param file is a shareable file.
   */
  Logger::Logger (const char* prefix, FILE*& logfile)
    : m_prefix(prefix), m_file (logfile)
  {
  }

  /*!
   * \brief destruct this logger.
   *  Do nothing - file will close anyway.
   */
  Logger::~Logger()
  {
  }

  /*!
   * \brief log a string.
   * \param str is the string to log.
   * \param endline is true if we print prefix and endline character.
   */
  void
  Logger::log (const StringStream& str, bool endline)
  {
    if (endline)
    {
      fprintf (m_file, "%s: %s\n", (const char*) m_prefix, (const char*) str);
    }
    else
    {
      fprintf (m_file, "%s", (const char*) str);
    }
  }
  /*!
   * \brief flush the stream.
   */
  void
  Logger::flush()
  {
    fflush (m_file);
  } 
}

