/*==========================================================
 * Program : sendsms.c                     Project : smslink
 * Author  : Philippe Andersson.
 * Date    : 11/03/05
 * Version : 0.24b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : SMS client.
 *
 * Modification History :
 * - 0.01a (28/09/98) : Initial release.
 * - 0.02a (21/10/98) : Adapted client to modifications in server
 *   consecutive to full GSM dialog implementation.
 * ++++ Switch to Beta ++++
 * - 0.03b (21/10/98) : First Beta release.
 * - 0.04b (23/10/98) : Added handling of the 'user' parameter.
 * - 0.05b (28/10/98) : Improved error reporting regarding
 *   host name resolution failure.
 * - 0.06b (03/11/98) : Added -f parameter (load message from
 *   a text file).
 * - 0.07b (06/02/99) : Added "-f -" option for stdin processing.
 *   Added Solaris compatibility. (contrib. by Philipp Klaus).
 * - 0.08b (16/02/99) : Adapted "slurping" of server announce
 *   after Ph. Klaus' modifications. Also solved a problem
 *   in the HP-UX port (PATH_MAX needs limits.h on HP-UX).
 * - 0.09b (20/05/99) : Corrected a bug that prevented comp-
 *   ilation on LINUX_LC6 platform.
 * - 0.10b (01/03/00) : Replaced the "old" slurp_n_catch() by
 *   a new version copied on get_gsm_answer(). This one should
 *   be more robust and less sensitive to the amount of data
 *   sent back by the server.
 * - 0.11b (28/09/00) : Added conditional compile control for
 *   FreeBSD platform (contributed by Stanley Hopcroft 
 *   <Stanley.Hopcroft@IPAustralia.Gov.AU>).
 * - 0.12b (29/09/00) : Missed one conditional compile for
 *   FreeBSD - fixed it.
 * - 0.13b (29/11/00) : Ported to Win32 + Cygwin1.1.
 * - 0.14b (12/02/01) : Fixed compile on Solaris platform.
 *   Thanks to Cristian Llaras <sistemes@myp.ad> for the fix.
 * - 0.15b (09/06/01) : Moved debug feature to a command-line
 *   parameter. Implemented a faster waiting loop mechanism to
 *   take advantage of the server "fast response" feature (client
 *   defaults to fast mode). Added -v and -h command-line options.
 *   Altered timeout values in GSM dialog. Cosmetics.
 * - 0.16b (11/06/01) : Replace the "slow" functions by their
 *   improved equivalent. Modify the "fast" ones to use the
 *   "early exit" functionality. Make the program default to
 *   "slow" (the "faster slow"). Added the -F option.
 * - 0.17b (05/09/01) : Fixed the dialogue w/ the server to
 *   accept the answer from "dd" mode as successfull. Added an
 *   option to report the device used by the server. Thanks
 *   to Paul Schaap <paul.schaap@dingoblue.com.au> for the fix
 *   and the devreport idea. Cosmetics.
 * - 0.18b (14/09/01) : Previous version was just so much
 *   gobbledigook. Here comes a full reengineering of the thing.
 *   Down to 14" in "poll,fast" mode, and 11" in "dd,fast" mode !
 *   Removed useless distinction between slow & fast mode.
 *   Merged slurp_n_catch() into get_gsm_answer(), made 
 *   allocation of space for sendlog fully dynamic. Improved
 *   on error reporting. Added message ID reporting, when
 *   provided. Added option "-p" and proper destination port
 *   handling.
 * - 0.19b (05/10/01) : Gained a few more secs against a fast
 *   server (fast mode, 19200 baud) by moving to usleep().
 *   Implemented slices of 100 millisecs. Down to 11" in
 *   "poll,fast".
 * - 0.20b (04/09/02) : Added support for multiple server-side
 *   action command (currently supports send, post and queue).
 * - 0.21b (11/06/03) : Added support for FreeBSD 5.0 (contributed by
 *   Greg May <greg@ult.co.za>).
 * - 0.22b (16/07/03) : Replaced outdated use of sys_errlist[] and
 *   sys_nerr by calls to strerror(). Thanks to Greg May
 *   (<greg@ult.co.za>) for reporting the bug (spotted on RH 9.0).
 * - 0.23b (03/06/04) : Added support for queued message priority
 *   with -e parameter. Improved error reporting in -Q mode.
 * - 0.24b (11/03/05) : Improved error reporting.
 *========================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>               /* for strcpy & friends */
#include <errno.h>
#ifdef HPUX
#  include <sys/unistd.h>                /* for getopt () */
#  include <limits.h>                     /* for PATH_MAX */
#else
#  include <unistd.h>                    /* for getopt () */
#  if (defined (LINUX_LC6) || defined (FREEBSD) || defined (FREEBSD5) || defined (WIN32) || defined (SOLARIS))
#    include <limits.h>
#    ifdef LINUX_LC6
#      include <linux/limits.h>
#    endif
#  endif
#endif                                      /* ifdef HPUX */
#include <netdb.h>                 /* for gethostbyname() */
#include <pwd.h>                        /* for getpwuid() */
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/time.h>           /* for the struct timeval */
#include <sys/ioctl.h>            /* for the ioctl() call */
#ifdef SOLARIS
#  include <sys/filio.h>
#endif

/*--------------------------------------Personal includes */
#include "sendsms.h"

/*========================================================*/
/**********             LOCAL DEFINES              ********/
/*========================================================*/

/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
int debug = FALSE;
char *server_actions[] = {"send", "post", "queue"};

/*========================================================*/
/**********               FUNCTIONS                ********/
/*========================================================*/
void syserr (char *msg)
{
  extern int errno;

  fprintf (stderr, "ERROR %s", msg);
#ifdef WIN32
  fprintf (stderr, " (%d)\n", errno);
#else
  if (errno > 0)
    fprintf (stderr, " (%d ; %s)\n", errno, strerror (errno));
  else
    fprintf (stderr, "\n");
#endif
  exit (1);
}                                            /* syserr () */
/*========================================================*/
void tellsock (int sock, char *string)
{
  int length;
  
  length = strlen (string);
  if (write (sock, string, length) != length)
    syserr ("sendsms: error while writing to socket");

}                                          /* tellsock () */
/*========================================================*/
void quote (char *string)
/* we take it for granted that we malloc'ed at least 2 char more
than max length */
{
  char *ptr;
  
  ptr = string + strlen (string);
  /* shift all chars 1 place to the right */
  while (ptr >= string) {
    ptr [1] = ptr[0];
    ptr--;
  }
  string[0] = '"';
  ptr = string + strlen (string);
  ptr[1] = '\0';
  ptr[0] = '"';
}                                             /* quote () */
/*========================================================*/
int desLF (char *string)
{
  char *ptr;
  
  if ((ptr = strchr (string, '\n')) != NULL) {
    ptr[0] = ' ';
  }
  
  return (strlen (string));
}                                             /* desLF () */
/*========================================================*/
int strappend (char **d, char *s)
{
  char *tmp;
  int l;
  
  /*--------------------------------------Initializations */
  tmp = NULL;
  
  /*-------------------------------------------Processing */
  if (*d) {
    l = (strlen (*d) + strlen (s));
    tmp = (char *) malloc ((l + 1) * sizeof (char));
    if (!tmp) {
      syserr ("sendsms: strappend(): can't malloc");
    }
    strcpy (tmp, *d);
    strcat (tmp, s);
    free (*d);
    *d = tmp;
  }
  else {
    l = strlen (s);
    *d = (char *) malloc ((l + 1) * sizeof (char));
    if (!*d) {
      syserr ("sendsms: strappend(): can't malloc");
    }
    strcpy (*d, s);
  }
  
  /*------------------------------------------Conclusions */
  return (l);
}                                         /* strappend () */
/*========================================================*/
int is_dotted_quad (char *string)
{
  int is_dq;
  char *s;
  int ntok = 0;
  char *ptok = NULL;
  long int ip_byte;
  char **endptr;
  char *dummyp;
  
  /*--------------------------------------Initializations */
  s = (char *) malloc ((strlen (string) + 1) * sizeof (char));
  strcpy (s, string);
  /*--------------------------------------------Main Loop */
  /* no point going any further if the 1st char. is not a digit */
  is_dq = isdigit (string[0]);
  if (is_dq) {
    ptok = strtok (s, ".");
    while (ptok != NULL) {
      ntok++;
      if (debug) {
        fprintf (stderr, "got token #%d : <%s>\n", ntok, ptok);
      }
      dummyp = ptok + strlen (ptok);
      endptr = &dummyp;
      ip_byte = strtol (ptok, endptr, 10);
      if (strlen (*endptr) != 0) {
        if (debug) {
	  fprintf (stderr, "error on token #%d : can't convert <%s>\n", ntok, *endptr);
        }
	is_dq = FALSE;
      }
      ptok = strtok (NULL, ".");
    }
    if (ntok != 4) {
      is_dq = FALSE;
    }
  }                                         /* if (is_dq) */
  /*------------------------------------------Conclusions */
  free (s);
  /*-------------------------------------------------Exit */
  return (is_dq);
}                                    /* is_dotted_quad () */
/*========================================================*/
unsigned long int resolve (char *host)
/* return the host ip address in network format */
{
  struct hostent *h_ent;
  struct in_addr target_ip;
  char **addrs;
  char *scratch;
  
  if (is_dotted_quad (host)) {
    if (debug) {
      fprintf (stderr, "I think I got an IP address\n");
    }
    if (inet_aton (host, &target_ip) == 0)
      syserr ("sendsms: invalid server IP address");
  }
  else {
    if (debug) {
      fprintf (stderr, "I think I got a host name\n");
    }
    if ((h_ent = gethostbyname (host)) == NULL) {
      scratch = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
      sprintf (scratch, "sendsms: can't resolve hostname (%s)", host);
      herror (scratch);
      free (scratch);
      exit (1);
    }
    /* lines below cf. "Beginning Linux Progr.", pp. 468-473 */
    addrs = h_ent->h_addr_list;
    target_ip = *(struct in_addr *)*addrs;
    if (debug) {
      fprintf (stderr, "server IP address is: [%s]\n", inet_ntoa (target_ip));
    }
  }                             /* if (isdigit (host[0])) */
  return (target_ip.s_addr);
}                                           /* resolve () */
/*========================================================*/
int load_from_file (char *msg, char *file)
{
  FILE *infile;
  char *buff;
  
  /*--------------------------------------Initializations */
  buff = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (!buff)
    syserr ("sendsms: can't malloc");
  msg[0] = '\0';
  
  /*------------------------------------------------------*/
  /* open file */
  if (file[0] == '-' && file[1] == '\0') {
    infile = stdin;
  } else {
    if ((infile = fopen (file, "r")) == NULL)
      syserr ("sendsms: can't open input file");
  }
  
  /* read from file */
  while (fgets (buff, BUFFSIZE, infile)) {
    desLF (buff);
    if ((strlen (msg) + strlen (buff)) > MAXMSGLEN) {
      strncat (msg, buff, (MAXMSGLEN - strlen (msg)));
      fprintf (stderr, "Warning: input file too long - truncation will occur\n");
    }
    else {
      strcat (msg, buff);
    }
  }                                           /* while () */
  
  /* close file */
  if (infile != stdin)
    fclose (infile);
  
  /*------------------------------------------------------*/
  return (strlen (msg));
}                                    /* load_from_file () */
/*========================================================*/
int get_gsm_answer (int fd, char **answer, char *catch, int resptime)
{
  int nread, retval, found;
  unsigned long slept, lresptime, slice;
  fd_set inputs;
  struct timeval timeout;
  char *buffer;
  
  /*--------------------------------------Initializations */
  nread = 0;
  slept = 0;
  lresptime = (resptime * 1000000);           /* in usec. */
  slice = 100000;                             /* in usec. */
  found = FALSE;
  buffer = NULL;
  
  FD_ZERO (&inputs);
  FD_SET (fd, &inputs);
  
  timeout.tv_sec = 10;
  timeout.tv_usec = 0;
  
  /* wait for data to arrive on the line */
  retval = select (FD_SETSIZE, &inputs, NULL, NULL, &timeout);
  switch (retval) {
    case 0:
      fprintf (stderr, "timeout while waiting for GSM answer.\n");
      break;
    
    case -1:
      syserr ("sendsms: call to select() failed");
      break;
      
    default:
      while ((!found) && (slept < lresptime)) {
        if (FD_ISSET (fd, &inputs)) {
          /* how much data do we have ? */
	  ioctl (fd, FIONREAD, &nread);
	  
	  /* free previous instance of buffer, if any */
	  if (buffer) {
	    free (buffer);
	  }
	  /* we know what the data size is - alloc space for it */
	  buffer = (char *) malloc ((nread + 1) * sizeof (char));
	  if (!buffer) {
	    syserr ("sendsms: get_gsm_answer(): can't allocate buffer space");
	  }
	  
	  /* now we can finally read this data */
	  nread = read (fd, buffer, nread);
	  switch (nread) {
	    case 0:
	      /* EOF */
	      if (debug) {
	        fprintf (stderr, "got no data from server.\n");
	      }
	      usleep (slice);
	      slept += slice;
	      break;

	    case -1:
	      syserr ("sendsms: error while reading answer from GSM");
	      break;

	    default:
	      buffer[nread] = '\0';
              if (debug) {
		fprintf (stderr, "pid<%d> Got : [%s] (%d char)\n", getpid (),
	        	buffer, nread);
              }
	      /* here we could pre-process it (remove Ctrl-Z) */
	      /* append it to answer */
              strappend (answer, buffer);
	      
	      /* search for catch */
	      if (strstr (*answer, catch)) {
	        found = TRUE;
		if (debug) {
		  fprintf (stderr, "Catch [%s] found !\n", catch);
		}
	      }
	      else {
	        /* wait for the rest of the server response */
		usleep (slice);
	        slept += slice;
	      }
	      break;
	  }                             /* switch (nread) */
        }
	else {
	  /* wait for data to present itself */
          usleep (slice);
	  slept += slice;
	}                              /* if (FD_ISSET... */
      }         /* while ((!found) && (slept < resptime)) */
      if (buffer) {
        free (buffer);
      }
      break;
  }                                    /* switch (retval) */

  if (*answer) {
    return (strlen (*answer));
  }
  else {
    return (0);
  }
}                                    /* get_gsm_answer () */
/*========================================================*/
char *extract_device (char *log)
{
  char *intro;
  char *start;
  char *end;
  char *device;
  int gotcha = FALSE;
  
  /* maybe the server is in "poll" mode */
  intro = strstr (log, "got ");
  if (! intro) {
    /* if not, then maybe it is in "dd" mode */
    intro = strstr (log, "resorting to dedicated device");
  }
  
  /* isolate device string */
  if (intro) {
    if ((start = strstr (intro, "<")) != NULL) {
      if ((end = strstr (start, ">")) != NULL) {
        gotcha = TRUE;
      }
    }
  }
  
  if (gotcha) {
    /* device found: allocate space (len + 1) & copy device string */
    device = (char *) malloc ((end - start) * sizeof (char));
    if (! device) {
      syserr ("sendsms: extract_device(): can't malloc");
    }
    strncpy (device, (start + 1), ((end - start) - 1));
    device[(end - start) - 1] = '\0';
  }
  else {
    /* catch string not found -- can't parse */
    device = (char *) malloc (3 * sizeof (char));
    if (! device) {
      syserr ("sendsms: extract_device(): can't malloc");
    }
    strcpy (device, "??");
  }
  
  return (device);

}                                    /* extract_device () */
/*========================================================*/
char *extract_msgid (char *log)
{
  char *intro;
  char *start;
  char *end;
  char *msgid;
  int gotcha = FALSE;
  
  /* maybe the server is in "poll" mode */
  intro = strstr (log, "message ID is ");
  
  /* a server in "dd" mode doesn't seem to report the msg ID */
  
  /* isolate message ID string */
  if (intro) {
    if ((start = strstr (intro, "<")) != NULL) {
      if ((end = strstr (start, ">")) != NULL) {
        gotcha = TRUE;
      }
    }
  }
  
  if (gotcha) {
    /* msg ID found: allocate space (len + 1) & copy ID string */
    msgid = (char *) malloc ((end - start) * sizeof (char));
    if (! msgid) {
      syserr ("sendsms: extract_msgid(): can't malloc");
    }
    strncpy (msgid, (start + 1), ((end - start) - 1));
    msgid[(end - start) - 1] = '\0';
  }
  else {
    /* catch string not found -- can't parse */
    msgid = (char *) malloc (3 * sizeof (char));
    if (! msgid) {
      syserr ("sendsms: extract_msgid(): can't malloc");
    }
    strcpy (msgid, "??");
  }
  
  return (msgid);

}                                     /* extract_msgid () */
/*========================================================*/
void report_success (char *log, int devrep)
{
  char *device;
  char *msgid;
  int outcome;
  
  /*---------------------------------------Initialization */
  device = NULL;
  msgid = NULL;
  outcome = -1;
  /*-------------------------------------------Processing */
  /* first determine action outcome */
  if (strstr (log, "message stored OK")) {
    /* message was queued */
    outcome = SMS_ACTION_QUEUE;
  }
  else if (strstr (log, "message sent OK")) {
    /* message was sent */
    outcome = SMS_ACTION_SEND;
  }
  else if (strstr (log, "Transmission Successful")) {
    /* message was sent */
    outcome = SMS_ACTION_SEND;
  }
  else {
    /* unknown -- internal error */
    syserr ("sendsms: can't determine outcome -- internal error");
  }
  
  /* then build reporting message */
  switch (outcome) {
    case SMS_ACTION_SEND: {
      if (devrep) {
	device = extract_device (log);
	msgid = extract_msgid (log);
	fprintf (stderr, "message successfully sent (device: <%s>, message ID: <%s>)\n",
        	device, msgid);
      }
      else {
	fprintf (stderr, "message successfully sent\n");
      }
      break;
    }
    
    case SMS_ACTION_QUEUE: {
      if (devrep) {
	msgid = extract_msgid (log);
	fprintf (stderr, "message queued for later delivery (message ID: <%s>)\n",
        	msgid);
      }
      else {
	fprintf (stderr, "message queued for later delivery\n");
      }
      break;
    }
    
    default: {
      fprintf (stderr, "unsupported outcome <%d>, aborting.\n", outcome);
      syserr ("sendsms: unknown outcome -- internal error");
      break;
    }
  }                                   /* switch (outcome) */
  /*-------------------------------------------Conclusion */
  if (device) {
    free (device);
  }
  if (msgid) {
    free (msgid);
  }
}                                    /* report_success () */
/*========================================================*/

/*========================================================*/
/**********           MAIN PROGRAM LOOP            ********/
/*========================================================*/

main (int argc, char *argv[])
{
  extern char *optarg;
  extern int optind, optopt;
  int c;                /* option char returned by getopt */
  int errflag = 0;       /* error in command line options */
  char *destgsm;
  char *user;
  char *smsc;
  char *message;
  char *msgfile;
  char *host_or_ip;
  int devreport;
  int destport;
  int action;
  int priority;
  struct servent *sent;
  struct servent ssent;
  struct in_addr server_ip;
  struct sockaddr_in sockaddr;
  int sockfd;
  int addrlen;
  char *cmdline;
  int retval;
  char *sendlog;
  struct passwd *pwduentry;

  /*--------------------------------------Initialisations */
  cmdline = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (!cmdline) {
    syserr ("sendsms: main(): can't malloc cmdline");
  }
  cmdline[0] = '\0';
  destgsm = (char *) malloc ((MAXPHNUMLEN + 1) * sizeof (char));
  if (!destgsm) {
    syserr ("sendsms: main(): can't malloc destgsm");
  }
  destgsm[0] = '\0';
  smsc = (char *) malloc ((MAXPHNUMLEN + 1) * sizeof (char));
  if (!smsc) {
    syserr ("sendsms: main(): can't malloc smsc");
  }
  smsc[0] = '\0';
  msgfile = (char *) malloc ((PATH_MAX + 1) * sizeof (char));
  if (!msgfile) {
    syserr ("sendsms: main(): can't malloc msgfile");
  }
  msgfile[0] = '\0';
  /* MAXUIDLEN + 3 to leave space for the quotes */
  user = (char *) malloc ((MAXUIDLEN + 3) * sizeof (char));
  if (!user) {
    syserr ("sendsms: main(): can't malloc user");
  }
  user[0] = '\0';
  /* MAXMSGLEN + 3 to leave space for the quotes */
  message = (char *) malloc ((MAXMSGLEN + 3) * sizeof (char));
  if (!message) {
    syserr ("sendsms: main(): can't malloc message");
  }
  message[0] = '\0';
  host_or_ip = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
  if (!host_or_ip) {
    syserr ("sendsms: main(): can't malloc host_or_ip");
  }
  host_or_ip[0] = '\0';
  sendlog = NULL;
  /*........................default values for parameters */
  if ((pwduentry = getpwuid (getuid ())) == NULL) {
    syserr ("sendsms: can't get user entry in passwd file");
  }
  if (strlen (pwduentry->pw_name) > MAXUIDLEN) {
    strncpy (user, pwduentry->pw_name, MAXUIDLEN);
    fprintf (stderr, "sendsms: WARNING: username too long (max. %d), truncated\n",
             MAXUIDLEN);
  }
  else {
    strcpy (user, pwduentry->pw_name);
  }
  quote (user);
  devreport = FALSE;
  destport = 0;
  priority = 0;
  action = SMS_ACTION_SEND;
  /*-------------------------Command line processing loop */
  while ((c = getopt (argc, argv, ":grvhSPQd:s:m:f:u:p:e:")) != -1) {
    switch (c) {
      case 'g':                               /* debug ON */
        debug = TRUE;
        break;
	
      case 'r':                     /* report used device */
        devreport = TRUE;
        break;
	
      case 'S':                         /* sends a "send" */
        action = SMS_ACTION_SEND;
        break;
	
      case 'P':                         /* sends a "post" */
        action = SMS_ACTION_POST;
        break;
	
      case 'Q':                        /* sends a "queue" */
        action = SMS_ACTION_QUEUE;
        break;
	
      case 'd':                 /* destination GSM number */
        if (strlen (optarg) > MAXPHNUMLEN) {
	  fprintf (stderr, "sendsms: -d parameter too long, max. is %d\n",
	          MAXPHNUMLEN);
	  errflag++;
	}
	else
	  strcpy (destgsm, optarg);
        break;

      case 's':                            /* SMSC number */
        if (strlen (optarg) > MAXPHNUMLEN) {
	  fprintf (stderr, "sendsms: -s parameter too long, max. is %d\n",
	          MAXPHNUMLEN);
	  errflag++;
	}
	else
          strcpy (smsc, optarg);
        break;

      case 'm':                        /* message to send */
        if (strlen (optarg) > MAXMSGLEN) {
	  fprintf (stderr, "sendsms: warning: message too long (max. %d char.) - truncated\n",
	          MAXMSGLEN);
          strncpy (message, optarg, MAXMSGLEN);
	}
	else
          strcpy (message, optarg);
	quote (message);
        break;

      case 'f':              /* file to load message from */
        if (strlen (optarg) > PATH_MAX) {
	  fprintf (stderr, "sendsms: warning: filename too long (max. %d char.)\n",
	          PATH_MAX);
	  errflag++;
	}
	else
          strcpy (msgfile, optarg);
        break;

      case 'u':                              /* sender ID */
        if (strlen (optarg) > MAXUIDLEN) {
	  fprintf (stderr, "sendsms: warning: username too long (max. %d char.) - truncated\n",
	          MAXUIDLEN);
          strncpy (user, optarg, MAXUIDLEN);
	}
	else
          strcpy (user, optarg);
	quote (user);
        break;

      case 'p':                     /* port on the server */
        destport = atoi (optarg);
        break;

      case 'e':                /* queued message priority */
        priority = atoi (optarg);
        break;

      case 'v':                           /* version info */
	printf ("sendsms ver. %s (%s), SMSLink Client for Unix\n", 
        	SMS_CLI_VERSION, SMS_CLI_DATE);
	printf ("(c) Les Ateliers du Heron, 1998-2006 for Scitex Europe, S.A.\n");
        exit (0);
        break;
	
      case 'h':                             /* usage help */
	printf ("\nsendsms ver. %s (%s), SMSLink Client for Unix\n", 
        	SMS_CLI_VERSION, SMS_CLI_DATE);
	printf ("(c) Les Ateliers du Heron, 1998-2004 for Scitex Europe, S.A.\n");
	printf ("\nUsage:\n");
	printf ("sendsms -d dest (-m msg | -f file) [options] server\n");
	printf ("\nOptions:\n");
	printf ("-d <dest>     = sets destination GSM number\n");
	printf ("-m <msg>      = sets message to be sent (max. %d char.)\n", MAXMSGLEN);
	printf ("-f <file>     = reads message to be sent from file ('-' for stdin)\n");
	printf ("-s <smsc>     = sets SMSC number (opt.)\n");
	printf ("-u <user>     = sets sender ID (max. %d char. - opt.)\n", MAXUIDLEN);
	printf ("-p <port>     = sets target comm. port on the server (if non-standard)\n");
	printf ("-e <priority> = sets queued message priority\n");
	printf ("-S            = use the server-side command \"send\" (default)\n");
	printf ("-P            = use the server-side command \"post\"\n");
	printf ("-Q            = use the server-side command \"queue\"\n");
	printf ("-g            = sets debugging ON [def. = OFF]\n");
	printf ("-r            = reports the device used by the server [def. = OFF]\n");
	printf ("-v            = prints version information and exits\n");
	printf ("-h            = prints this usage information and exits\n");
	printf ("<server>      = SMS Server name or ip addr.\n");
        exit (0);
        break;
	
      case ':':             /* parameter without argument */
        fprintf (stderr, "sendsms: Option -%c requires an argument.\n",
                optopt);
        errflag++;
        break;

      case '?':               /* unrecognized option char */
        fprintf (stderr, "sendsms: Unrecognized option: -%c.\n", optopt);
        errflag++;
        break;

      default:
        fprintf (stderr, "sendsms: getopt() returned char. code 0%o ??\n", c);
    }                                       /* switch (c) */
  }                                       /* while (c...) */
  /*......................................................*/
  /* now transfer the last element (server name) */
  if (optind < argc) {
    if (strlen (argv[optind]) > MINIBUFF) {
      fprintf (stderr, "sendsms: <server name> parameter too long, max. is %d\n",
	      MINIBUFF);
      errflag++;
    }
    else
      strcpy (host_or_ip, argv[optind]);
  }
  /*----------------------Check for required parameter(s) */
  /* -d, either -m or -f and server name or address */
  if (!errflag) {
    if (strlen (destgsm) == 0) {
      fprintf (stderr, "sendsms: Required parameter -d not present.\n");
      errflag++;
    }
    if ((strlen (message) == 0) && (strlen (msgfile) == 0)) {
      fprintf (stderr, "sendsms: You must provide either -m or -f.\n");
      errflag++;
    }
    if (strlen (host_or_ip) == 0) {
      fprintf (stderr, "sendsms: Required parameter <server name> not present.\n");
      errflag++;
    }
  }                                      /* if (!errflag) */

  /*-------------------Check for conflicting parameter(s) */
  /* -m with -f */
  if ((strlen (message) > 0) && (strlen (msgfile) > 0)) {
    fprintf (stderr, "sendsms: Conflicting parameters -m and -f.\n");
    errflag++;
  }
  
  /* -e in "send" mode */
  if ((priority != 0) && (action == SMS_ACTION_SEND)) {
    fprintf (stderr, "sendsms: priority meaningful in POST and QUEUE mode only, ignored.\n");
  }
  
  /*--------------------------------Validate parameter(s) */
  if ((priority < 0) || (priority > SMS_Q_PRI_CEIL)) {
    fprintf (stderr, "sendsms: Unsupported priority (%d).\n", priority);
    errflag++;
  }
  /*-----------------------On error, display usage & exit */
  if (errflag) {
    fprintf (stderr, "\n==> sendsms -h for usage details.\n");
    exit (2);
  }
  /* DEBUG ...............................................*/
  if (debug) {
    fprintf (stderr, "Debugging mode is turned ON.\n");
    fprintf (stderr, "Those are the current parameters :\n");
    fprintf (stderr, "dest     = [%s]\n", destgsm);
    fprintf (stderr, "smsc     = [%s]\n", smsc);
    fprintf (stderr, "message  = [%s]\n", message);
    fprintf (stderr, "msgfile  = [%s]\n", msgfile);
    fprintf (stderr, "user     = [%s]\n", user);
    if (destport) {
      fprintf (stderr, "port     = [%d]\n", destport);
    }
    if (priority) {
      fprintf (stderr, "priority = [%d]\n", priority);
    }
    fprintf (stderr, "raw host = [%s]\n", host_or_ip);
    fprintf (stderr, "action   = [%s]\n", server_actions[action]);
    fprintf (stderr, "Dev report [%s]\n", (devreport) ? "ON" : "OFF");
  }

  /*---------------------------Start real processing loop */
  /* first resolve server name */
  server_ip.s_addr = resolve (host_or_ip);
  
  /* load message from file if required */
  if (strlen (msgfile) > 0) {
    if (load_from_file (message, msgfile) == 0)
      syserr ("sendsms: error loading message from file or empty file");
    quote (message);
  }
  
  /* find destination port if not specified */
  if (destport == 0) {
    /* not provided on command-line -- get it from system */
    if ((sent = getservbyname ("sms", "tcp")) == NULL) {
      /* not defined in /etc/services, use compiled-in default */
      sent = &ssent;
      sent->s_port = htons (DEFSMSPORT);
      if (debug) {
	fprintf (stderr, "using compiled-in default port <%d> to connect to server\n",
	DEFSMSPORT);
      }
    }
    else {
      /* ok, found */
      if (debug) {
	fprintf (stderr, "found port <%d> for service sms\n", ntohs (sent->s_port));
      }
    }
  }
  else {
    /* store provided value in struct servent after conversion */
    sent = &ssent;
    sent->s_port = htons (destport);
    if (debug) {
      fprintf (stderr, "using provided destport <%d> to connect to server\n",
              destport);
    }
  }                                 /* if (destport == 0) */
  
  /* create the socket */
  if ((sockfd = socket (AF_INET, SOCK_STREAM, 0)) == -1)
    syserr ("sendsms: can't create socket");
  
  /* build the server socket address parameters */
  sockaddr.sin_family = AF_INET;
  sockaddr.sin_port = sent->s_port;
  sockaddr.sin_addr.s_addr = server_ip.s_addr;
  addrlen = sizeof (sockaddr);
  
  /* now connect to the server */
  if (connect (sockfd, (struct sockaddr *)&sockaddr, addrlen) == -1)
    syserr ("sendsms: can't connect to server");
    
  /*......................................Start of dialog */
  while (TRUE) {                            /* dummy loop */
    if (sendlog) {
      free (sendlog);
      sendlog = NULL;
    }

    /* slurp server announce and catch prompt */
    if (get_gsm_answer (sockfd, &sendlog, SERVPROMPT, 5)) {
      if (debug) {
	fprintf (stderr, "server announce:\n%s\n", sendlog);
      }
      if (strstr (sendlog, SERVPROMPT)) {
	retval = 0;                            /* success */
      }
      else {
	fprintf (stderr, "can't get server prompt\n");
	retval = 103;             /* failed to get prompt */
	break;
      }
    }
    else {
      fprintf (stderr, "no answer from server (announce)\n");
      retval = 101;                          /* no answer */
      break;
    }                          /* if (get_gsm_answer (... */

    /* set sender ID - get ok */
    sprintf (cmdline, "set user = %s\n", user);
    tellsock (sockfd, cmdline);

    if (sendlog) {
      free (sendlog);
      sendlog = NULL;
    }

    if (get_gsm_answer (sockfd, &sendlog, SERVPROMPT, 2)) {
      if (debug) {
	fprintf (stderr, "set user dialog:\n%s\n", sendlog);
      }
      if (strstr (sendlog, "Ok")) {
	retval = 0;                            /* success */
      }
      else {
	fprintf (stderr, "failed to set sender ID (user name)\n");
	retval = 104;                  /* set cmd failure */
	break;
      }
    }
    else {
      fprintf (stderr, "no answer from server (user)\n");
      retval = 101;                          /* no answer */
      break;
    }                          /* if (get_gsm_answer (... */

    /* set dest GSM - get ok */
    sprintf (cmdline, "set dest = %s\n", destgsm);
    tellsock (sockfd, cmdline);

    if (sendlog) {
      free (sendlog);
      sendlog = NULL;
    }

    if (get_gsm_answer (sockfd, &sendlog, SERVPROMPT, 2)) {
      if (debug) {
	fprintf (stderr, "set dest dialog:\n%s\n", sendlog);
      }
      if (strstr (sendlog, "Ok")) {
	retval = 0;                            /* success */
      }
      else {
	fprintf (stderr, "failed to set destination GSM number\n");
	retval = 104;                  /* set cmd failure */
	break;
      }
    }
    else {
      fprintf (stderr, "no answer from server (dest)\n");
      retval = 101;                          /* no answer */
      break;
    }                          /* if (get_gsm_answer (... */

    /* set smsc if needs be - get ok */
    if (strlen (smsc) > 0) {
      sprintf (cmdline, "set smsc = %s\n", smsc);
      tellsock (sockfd, cmdline);

      if (sendlog) {
	free (sendlog);
	sendlog = NULL;
      }

      if (get_gsm_answer (sockfd, &sendlog, SERVPROMPT, 2)) {
	if (debug) {
	  fprintf (stderr, "set smsc dialog:\n%s\n", sendlog);
	}
	if (strstr (sendlog, "Ok")) {
	  retval = 0;                          /* success */
	}
	else {
	  fprintf (stderr, "failed to set new SMSC number\n");
	  retval = 104;                /* set cmd failure */
	  break;
	}
      }
      else {
	fprintf (stderr, "no answer from server (smsc)\n");
	retval = 101;                        /* no answer */
	break;
      }                        /* if (get_gsm_answer (... */
    }

    /* set priority if needs be - get ok */
    if (priority) {
      sprintf (cmdline, "set prio = %d\n", priority);
      tellsock (sockfd, cmdline);

      if (sendlog) {
	free (sendlog);
	sendlog = NULL;
      }

      if (get_gsm_answer (sockfd, &sendlog, SERVPROMPT, 2)) {
	if (debug) {
	  fprintf (stderr, "set prio dialog:\n%s\n", sendlog);
	}
	if (strstr (sendlog, "Ok")) {
	  retval = 0;                          /* success */
	}
	else {
	  fprintf (stderr, "failed to set new priority\n");
	  retval = 104;                /* set cmd failure */
	  break;
	}
      }
      else {
	fprintf (stderr, "no answer from server (prio)\n");
	retval = 101;                        /* no answer */
	break;
      }                        /* if (get_gsm_answer (... */
    }

    /* set message - get ok */
    sprintf (cmdline, "set message = %s\n", message);
    tellsock (sockfd, cmdline);

    if (sendlog) {
      free (sendlog);
      sendlog = NULL;
    }

    if (get_gsm_answer (sockfd, &sendlog, SERVPROMPT, 2)) {
      if (debug) {
	fprintf (stderr, "set message dialog:\n%s\n", sendlog);
      }
      if (strstr (sendlog, "Ok")) {
	retval = 0;                            /* success */
      }
      else {
	fprintf (stderr, "failed to set message to be transmitted\n");
	retval = 104;                  /* set cmd failure */
	break;
      }
    }
    else {
      fprintf (stderr, "no answer from server (message)\n");
      retval = 101;                          /* no answer */
      break;
    }                          /* if (get_gsm_answer (... */

    /* send - get feedback (message sent) */
    sprintf (cmdline, "%s\n", server_actions[action]);
    if (debug) {
      fprintf (stderr, "Now issuing the <%s> server-side command.\n",
              server_actions[action]);
    }
    tellsock (sockfd, cmdline);

    if (sendlog) {
      free (sendlog);
      sendlog = NULL;
    }

    if (get_gsm_answer (sockfd, &sendlog, SERVPROMPT, 150)) {
      /* timeout needs to be as large as 150 here, because of the
       * huge time it takes to send and verify a PIN code (at a
       * min. 13 secs., plus the 90 secs. timeout while waiting
       * for a free GSM instance, plus the max. time to send in
       * in regular (= slow) mode. */
      if (debug) {
	fprintf (stderr, "SMS send dialog:\n%s\n", sendlog);
      }
      if (strstr (sendlog, "message sent OK") ||
          strstr (sendlog, "message stored OK")) {
	/* server in "poll" mode */
	report_success (sendlog, devreport);
	retval = 0;                            /* success */
      }
      else if (strstr (sendlog, "Transmission Successful")) {
	/* server in "dd" mode */
	report_success (sendlog, devreport);
	retval = 0;                            /* success */
      }
      else if (strstr (sendlog, "queueing facility disabled")) {
	fprintf (stderr, "failed to queue message\n");
	retval = 105;                /* queueing disabled */
	/* don't break -- try to close connection properly */
      }
      else {
	fprintf (stderr, "failed to send message\n");
	retval = 102;                   /* sending failed */
	/* don't break -- try to close connection properly */
      }
    }
    else {
      fprintf (stderr, "no answer from server (action)\n");
      retval = 101;                          /* no answer */
      break;
    }                          /* if (get_gsm_answer (... */

    /* close connection */
    sprintf (cmdline, "bye\n");
    tellsock (sockfd, cmdline);
    break;
  }                                       /* while (TRUE) */

  /*------------------------------------------Conclusions */
  /* close socket */
  if (close (sockfd) == -1)
    syserr ("sendsms: can't close socket");
  
  /* free what's need to be */
  free (destgsm);
  free (smsc);
  free (message);
  free (msgfile);
  free (host_or_ip);
  free (cmdline);
  if (sendlog) {
    free (sendlog);
  }
  /*------------------------------------------End program */
  exit (retval);
}                                              /* main () */

/*==========================================================
 * EOF : sendsms.c
 *===================*/
