/*==========================================================
 * Program : send_sms.cpp             
 * Author  : Melle Sprenger (mellie@xs4all.nl)
 * 
 * Date    : 23/06/2000
 * Version : 0.1
 * License : GNU GENERAL PUBLIC LICENSE
 *           Version 2, June 1991
 *
 * Comment : Win32 commandline client for SMS Link server
 *=========================================================*/

#include <condefs.h>
#include <iostream.h>
#include <winsock.h>
#include <stdio.h>
#include <winsock.h>
#include <conio.h>
#include <io.h>
#include <fcntl.h>

// Prototypes of the functions used
SOCKET OpenSocket(char *sServer,short iPortNumber);
int SendSMS(SOCKET socketSMS, char *sMessage, char *sUser, char *sDest, char *sSMSC);
int WaitForPrompt(SOCKET socketSMS, bool bCheckOk = true);
int SendToSocket(SOCKET socketSMS,char *sCommand);

// Defines of the error values used in this SMS client
#define SUCCESS           0
#define ERR_SOCK_TIMEOUT  1
#define ERR_SOCK_RECIEVE  2
#define ERR_SOCK_SEND     3
#define ERR_SOCK_INIT     4
#define ERR_SOCK_CONNECT  5
#define ERR_SOCK_SETOPT   6
#define ERR_SOCK_NOACCESS 7
#define ERR_SOCK_DISCONN  8
#define ERR_CFG_OPEN      9
#define ERR_PHONE_OPEN    10
#define ERR_WRONG_PARAMS  11
#define ERR_WINSOCK_DLL   12
#define ERR_HOSTNOTFOUND  13
#define ERR_SMS_ERROR     14

// SMS client specific defines
#define SOCKET_TIMEOUT 15         // Timeout value in seconds for socket
//#define DEBUG                   // uncomment this line out for more
                                  // screenoutput and user-input

// SMS server specific defines
#define SMS_PROMPT  "SMS> "            // SMS server prompt
#define SMS_SUCCESS "Ok"               // SMS server's string that indicates successful command execution
#define SMS_SENT    "message sent OK"  // SMS server's respond after an SMS has been sent

//------------------------- Main ---------------------------------------------//
void main(int argc, char **argv)
{
  int iReturn;                // return value
  FILE *hCfgFile;             // handle to config file
  FILE *hPhoneFile;           // handle to file containing multiple phonenumbers
  short iPortNumber;          // SMS server's port number
  char sCfgFile[256],         // The filename of the config file
       sServer[256],          // SMS server's IP address
       sSMSC[64],             // SMS Service Centre's phone number
       sMessage[170],         // Message we'll send
       sPhone[256],           // Second parameter, can be a phonenumber or a filename
       sDest[64],             // Destination phone number
       sUser[64],             // Username for SMS server
       sBuffer[256];          // Buffer used to read config file
  SOCKET  socketSMS;          // Handle to the socket used
  bool bPhoneNumber = true;   // Do we have a phonenumber, or a file with multiple numbers?


  printf("\n  Send_sms version 0.1, Copyright (C) 2000 Melle Sprenger");
  printf("\n  Send_sms comes with ABSOLUTELY NO WARRANTY.");
  printf("\n  This is free software, and you are welcome to redistribute it");
  printf("\n  under certain conditions; read license.txt for details.");

  // Construct the name of the config file
  strcpy(sCfgFile,argv[0]);
  sCfgFile[strlen(sCfgFile) - 3] = '\0';
  strcat(sCfgFile,"CFG");

  // Check commandline parameters
  if (argc != 3)
  {
    printf("\n\nUsage: send_sms \"msg\" \<phone\>\|\<list\>");
    printf("\n\n----------------------------------------------");
    printf("\n\n msg  : SMS message to send, between quotes");
    printf("\n\n phone: phone number to send SMS to.");
    printf("\n        (int. format, e.g. +31655108229)");
    printf("\n\n list : full path to file containing a list");
    printf("\n        of phone numbers to send SMS to");
    printf("\n\n----------------------------------------------");
    printf("\n\n\n press any key to exit...");
    getch();
    exit(ERR_WRONG_PARAMS);
  }

  // Set message to first commandline argument
  strcpy(sMessage,"\0");
  strcat(sMessage,argv[1]);

  // Set destination to second commandline argument
  strcpy(sPhone,"\0");
  strcpy(sPhone,argv[2]);

  // Here we could maybe place the local hostname as the user that is
  // sent to the SMS server TODO...
  strcpy(sUser,"w32smsclient");

  // Open the config file for read
  hCfgFile = fopen(sCfgFile, "r");
  if (hCfgFile == NULL)
     exit(ERR_CFG_OPEN);

  // The config file should consist of 3 lines, containing the following information:
  //
  // ipaddress sms server
  // sms_serv port
  // sms service centre
  //
  // This way of reading a config file is not really the best way to do it i guess.
  // we should probably use a proper .ini file and some more robust code for this.
  // for now this quick hack works though... TODO...
  int iStrPos,iChar;

  for (int iLine = 0; iLine < 3; iLine++)
  {
    iStrPos = 0;

    // Read a line, char by char and put it into sBuffer
    while ((iChar = (char)fgetc(hCfgFile)) != '\n')
    {
       sBuffer[iStrPos] = (char) iChar;
       iStrPos++;
    }
    // Close the sBuffer
    sBuffer[iStrPos] = '\0';

    // Put the data in sBuffer into the appropriate string
    switch(iLine)
    {
      case 0: // first line is the SMS server's IP address
        strcpy(sServer,sBuffer);
        break;
      case 1:  // second line is the SMS server's port number
        iPortNumber = atoi(sBuffer);
        break;
      case 2: // third line is the SMS Service Centre
        strcpy(sSMSC,sBuffer);
        break;
      default:
        break;
    }
  }
  // close the config file
  fclose(hCfgFile);

  // Check the sPhone string. If it is not a phonenumber (starting with "+")
  // we're gonna check if we can open the file
  if (sPhone[0] != '+')
  {
    bPhoneNumber = false;
    hPhoneFile = fopen(sPhone, "r");
    if (hPhoneFile == NULL)
    {
      printf("\nCould not open phonefile : %s",sDest);
      getch();
      exit(ERR_PHONE_OPEN);
    }
  }
  else
    strcpy(sDest,sPhone);

  // open a socket to the SMS server
  // after this call we should check if the socket is really there :-) TODO...
  socketSMS = OpenSocket(sServer,iPortNumber);
  if (socketSMS == NULL)
    exit(ERR_SOCK_INIT);
  // If the second parameter was a phonenumber send the SMS
  if (bPhoneNumber)
    iReturn = SendSMS(socketSMS, sMessage, sUser, sDest, sSMSC);

  // If the second parameter was a file, read it and send the SMS message
  // for every phonenumber in the file
  else
  {
    printf("\n\nSending multiple SMS's using %s\n", sPhone);

    // quick and dirty way to set strlen(sDest) bigger than 4
    strcpy(sDest,"what is the question to which the answer is 42?");

    // if the last read string from the file is shorter than 4, we'll assume we
    // have to quit. We should ofcourse find a more elegant way to read and
    // analyze the file with phonenumbers. Since the length of a phonenumber is
    // always > 4 this works for now :-)
    while (strlen(sDest) > 4)
    {
      iStrPos = 0;
      // Read a line, char by char and put it into sBuffer while we're not at EOF
      while ((iChar = (char)fgetc(hPhoneFile)) != '\n' && !feof(hPhoneFile))
      {
         sBuffer[iStrPos] = (char) iChar;
         iStrPos++;
      }
      // Close the sBuffer
      sBuffer[iStrPos] = '\0';
      // Copy the the read string from the phonenumber file into sDest
      strcpy(sDest,sBuffer);
      // Send the SMS message to the number read (only if we're not at EOF)
      if (!feof(hPhoneFile))
        iReturn = SendSMS(socketSMS, sMessage, sUser, sDest, sSMSC);
      else
        iReturn = 0;

      // If an error occured sending the SMS, we'll close and reopen the socket
      // just to be sure it will work for the next SMS to send (hopefully :-)
      if (iReturn != 0)
      {
        SendToSocket(socketSMS,"exit\r\n");
        closesocket(socketSMS);
        socketSMS = OpenSocket(sServer,iPortNumber);
      }
    }
  // close the phone file
  fclose(hPhoneFile);
  }
  // close the connection by sending the "exit" command
  SendToSocket(socketSMS,"exit\r\n");
  // Just to be sure we'll close the connection from our side too :-)
  closesocket(socketSMS);
  // Release WinSock and exit with the return code
  WSACleanup();
  #ifdef DEBUG
  printf("Exiting, lasterror = %i",iReturn);
  getch();
  #endif
  exit(iReturn);
}

//------------------------- SendSMS ------------------------------------------//
// function that sends the SMS message
//
// Arguments:
//               -socketSMS      Pointer to a socket connected to the SMS server
//               -sMessage       Message to send
//               -sUser          Username to send to the SMS server
//               -sDest          Destination phone number
//               -sSMSC          Phone number of the SMS Service Centre
//
// Return value: integer (0 if succesfull)
//
//----------------------------------------------------------------------------//

int SendSMS(SOCKET socketSMS,   // the open socket to the SMS Server
            char *sMessage,     // Message to send
            char *sUser,        // Username to send to SMS server
            char *sDest,        // Destination phone number
            char *sSMSC)        // SMS Service Centre
{
  // Print some info to the screen...
  printf("\nSending SMS message to %s...",sDest);
  // Start SMS sending sequence
  char sCmd[256];
  int i = 0;
  int iReturn;

  iReturn = SUCCESS;

  // Go through command sequence while return value is SUCCESS
  while (iReturn == SUCCESS)
  {
    i++;
    switch(i)  // set the command to send to the SMS server
    {
      case 1: // set username
        sprintf(sCmd,"set user = \"%s\"\r\n",sUser);
        break;
      case 2: // set message
        sprintf(sCmd,"set msg = \"%s\"\r\n",sMessage);
        break;
      case 3: // set destination phone number
        sprintf(sCmd,"set dest = %s\r\n",sDest);
        break;
      case 4: // set SMS Service Centre number
        sprintf(sCmd,"set smsc = %s\r\n",sSMSC);
        break;
      case 5: // send the message
        sprintf(sCmd,"send\r\n");
        break;
      default: // if the "send" command returns, we'll quit
        printf("done!");
        return(iReturn);
        break;
    }
    // if we got a prompt, send the command to the SMS server
    iReturn = SendToSocket(socketSMS,sCmd);
    // Command should have been set, now check if we get a prompt from the SMS Server
    if (iReturn == SUCCESS)
      iReturn = WaitForPrompt(socketSMS);
    if (iReturn == SUCCESS)
      printf(".");
  }
  // if the returncode is not SUCCESS we'll stop and return the error to main()
  printf("error!");
  return(iReturn);
}


//------------------------- WaitForPrompt ------------------------------------//
// This function reads the incoming data from the SMS server and waits for
// the SMS server's prompt. If no incoming data is detected for SOCKET_TIMEOUT
// seconds the function returns
//
// Arguments:
//               -socketSMS      Pointer to a socket connected to the SMS server
//               -bCheckOK       boolean to indicate wether the function should
//                               check if the recieved buffer before the prompt
//                               contains the string SMS_SUCCESS (default=true)
//
// Return value: integer (0 if succesfull)
//
//----------------------------------------------------------------------------//

int WaitForPrompt(SOCKET socketSMS, bool bCheckOk)
{
  char sBuf[256],         // string to hold the last recieved data
       sResponse[2048];   // string to hold the recieved data since the last prompt

  int iReturn;
  fd_set ReadAbleSock;            // Mask for select() function
  struct timeval  TimeOut;    // timeout structure
  struct timeval *pTimeOut;   // timeout structure

  // Set the socket timeout value
  TimeOut.tv_sec = SOCKET_TIMEOUT;
  TimeOut.tv_usec=0;
  pTimeOut = &TimeOut;
  // reset the sResponse string AND the sBuf string (NEVER forget to initialize
  // chars i guess... took me about an hour to figure this out
  strcpy(sResponse,"\0");
  strcpy(sBuf,"\0");

  // Get the data sent by the SMS server until we have a prompt
  while ( strstr(sBuf,SMS_PROMPT) == NULL)
  {
    // reset mask and set the socket we are using. If not our application could
    // cause other programs serious trouble :-) (although Win32 docs tell us something
    // completely different)
    FD_ZERO (&ReadAbleSock);
    FD_SET (socketSMS, &ReadAbleSock);

    // Poll the socket for pending data and check if it doesn't timeout
    iReturn = select(socketSMS+1, &ReadAbleSock, NULL, NULL, pTimeOut);

    // If select() returns an error we'll quit...
    if (iReturn == SOCKET_ERROR)
      return(ERR_SOCK_RECIEVE);

    // else if the socket is alive, hasn't timed out and there's data waiting,
    // get the data
    else if (FD_ISSET(socketSMS,&ReadAbleSock))
    {
      strcpy(sBuf,"\0");
      iReturn = recv(socketSMS,               // Connected socket
                     sBuf,                    // Receive buffer
                     sizeof(sBuf),            // Size of receive buffer
                     0);                      // Flags
      // If we couldn't recieve data from the socket we'll 'die'
      if (iReturn == SOCKET_ERROR)
        return(ERR_SOCK_RECIEVE);
      // If recv() returns "0" (no chars recieves) the server gracefully
      // ended out connection
      if (iReturn == 0)
        return(ERR_SOCK_DISCONN);
      // We close the buffer since we'll get garbage at the end otherwise
      sBuf[iReturn] = '\0';
      // print the buffer to screen an append it to the sResponse buffer
      #ifdef DEBUG
      printf("%s", sBuf);
      #endif
      strcat(sResponse,sBuf);
    }
    // if no data is waiting and select returns "0" the socket has timed out :-(
    else
      return(ERR_SOCK_TIMEOUT);
  }
  // check if the SMS server understood what we were telling him/her
  // In case the message is sent ok we might want to extract the message id from the
  // sResponse string too. It might also be a good idea to let this function somehow
  // return the sResponse string so we can evaluate that inside the SendSMS()
  // function and take actions there TODO..
  if (strstr(sResponse,SMS_SUCCESS) || strstr(sResponse, SMS_SENT) || !bCheckOk )
    return(SUCCESS);

   // if he/she did not, return an SMS_ERROR. We should probably look at the sResponse
  // string and see what went wrong here, so we can deal with it a little bit more
  // intelligent as opposed to just quitting with an SMS_ERROR  TODO...
  else
    return(ERR_SMS_ERROR);
}

//------------------------- SendToSocket -------------------------------------//
// Sends a string to the connected socket
//
// Arguments:
//               -socketSMS      Pointer to a socket connected to the SMS server
//               -sCommand       Command to send to the SMS server
//
// Return value: integer (0 if succesfull)
//
//----------------------------------------------------------------------------//

int SendToSocket(SOCKET socketSMS, char *sCommand)
{
  int iReturn;
  // print the command to screen so we can see wat's going on (local echo)
  #ifdef DEBUG
  printf("%s",sCommand);
  #endif
  // send the command to the socket
  iReturn = send(socketSMS,           // Connected socket
                 sCommand,            // Data buffer to send
                 strlen(sCommand),    // Length of data buffer
                 0);                  // Flags
  // if we got a socket error we'll just 'die'
  if (iReturn == SOCKET_ERROR)
    return(ERR_SOCK_SEND);
  //string sent successfully
  return(SUCCESS);
}

//------------------------- OpenSocket -------------------------------------//
// Initialises and opens the socket connection
//
// Arguments:
//               -sSMS           SMS server IP address
//               -iPortNumber    Portnumber to which the SMS server is listening
//
// Return value: Pointer to the connected socket
//
//----------------------------------------------------------------------------//
SOCKET OpenSocket(char *sServer, short iPortNumber)
{
  int iReturn;
  SOCKET socketSMS;
  WORD wVersionRequested;     // Windsock version we want
  WSADATA wsaData;            // Windows socket data buffer

  // Initialize WinSock and check for the correct version
  wVersionRequested = MAKEWORD(1,1);
  iReturn = WSAStartup(wVersionRequested, &wsaData);

  // If the version is not what we expected we'll just 'die' for now
  if (wsaData.wVersion != wVersionRequested)
      return(NULL);

  // Find the SMS server
  LPHOSTENT lpHostEntry;
  lpHostEntry = gethostbyname(sServer);
  if (lpHostEntry == NULL)
    return(NULL);

  // Create a socket connection to the SMS Server
  socketSMS = socket(AF_INET,             // Address family
                     SOCK_STREAM,         // Socket type
                     IPPROTO_TCP);        // Protocol
  if (socketSMS == INVALID_SOCKET)
    return(socketSMS);

  // Fill in the socket address structure
  SOCKADDR_IN saServer;
  saServer.sin_family = AF_INET;
  saServer.sin_addr = *((LPIN_ADDR)*lpHostEntry->h_addr_list);
  saServer.sin_port = htons(iPortNumber);

  printf("\n\nConnecting to server %s on port %d...", sServer, iPortNumber);
  // connect to the server
  iReturn = connect(socketSMS,                // Socket
                    (LPSOCKADDR)&saServer,    // Server address
                    sizeof(struct sockaddr)); // Length of server address structure

  // if we don't get a connection we'll just 'die' with the appropriate error
  // we probably should have some retry mechanism here, so a temporarily network
  // problem won't spoil the fun... TODO...
  if (iReturn == SOCKET_ERROR)
    return(NULL);
  printf("connected!\n");
  iReturn = WaitForPrompt(socketSMS, false);
  if (iReturn == 0)
    return(socketSMS);
  else
    return(NULL);
}

