/*==========================================================
 * Program : decode_pdu.c                  Project : smslink
 * Authors : Philippe Andersson.
 *           Franois Baligant <francois@euronet.be>.
 * Date    : 28/05/03
 * Version : 0.05b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : PDU Manipulation tool. Decodes PDU string.
 *
 * Modification History :
 * - 0.01a (12/01/02) : Initial release.
 * - 0.02a (23/01/02) : Added support for default GSM alphabet.
 * ++++ Switch to Beta ++++
 * - 0.03b (29/01/02) : Added CLI option to set the alphabet to
 *   use.
 * - 0.04b (02/03/02) : Added support for real GSM model emulation
 *   (incl. the ability to specify the model on the command-line).
 * - 0.05b (28/05/03) : Fixed a bug where the 'fromgsm' field in
 *   struct mbox_item wasn't properly initialized.
 *========================================================*/
#include <unistd.h>
#include <stdio.h>                         /* for fprintf */
#include <stdlib.h>                  /* for errno & stuff */
#include <errno.h>
#include <string.h>

#include "sms_serv.h"
#include "gsmdev.h"
#include "pdu.h"
#include "decode_pdu.h"

/*========================================================*/
/**********             LOCAL DEFINES              ********/
/*========================================================*/

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
int debug = 0;
char *progname = NULL;
int default_alphabet = DEFAULT_ALPHABET;
char *gsmdevfile = NULL;     /* unused - required to link */
                                  /* against gsmdevices.o */

/*========================================================*/
/**********           MAIN PROGRAM LOOP            ********/
/*========================================================*/
int main (int argc, char *argv[])
{
  extern char *optarg;
  extern int optind, optopt;
  int c;                /* option char returned by getopt */
  int errflag = 0;       /* error in command line options */
  int task = 0;
  char *string;
  char *csname = NULL;
  struct mbox_item message;
  cap_matrix capmatrix;
  char model[MAXMODLEN + 1];
  struct gsms_def gsm;

  /*---------------------------------------Store own name */
  progname = (char *) malloc ((strlen (argv[0]) + 1) * sizeof (char));
  if (! progname) {
    syserr("main(): can't malloc()");
  }
  strcpy (progname, argv[0]);
  l_basename (progname);
  
  /*--------------------------------------Initialisations */
  string = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (!string) {
    syserr("main(): can't malloc() string");
  }
  string[0] = '\0';
  
  model[0] = '\0';
  strcpy (model, "WMOD2B");

  message.msgid = 0;
  memset (message.fromgsm, 0, (MAXPHNUMLEN + 1));
  memset (message.date, 0, 9);
  memset (message.time, 0, 10);
  memset (message.text, 0, (MAXMSGLEN + 1));
  message.next = NULL;
  message.prev = NULL;

  /*-------------------------Command line processing loop */
  while ((c = getopt (argc, argv, ":ghvs:m:z:")) != -1) {
    switch (c) {
      case 'g':                               /* debug ON */
        debug = DEBUG_ALL;
        break;
	
      case 's':                                 /* string */
        if (strlen (optarg) > BIGBUFF) {
	  fprintf (stderr, "decode_pdu: -s parameter too long, max. is %d\n",
	          BIGBUFF);
	  errflag++;
	}
	else
	  strcpy (string, optarg);
        break;

      case 'm':                                  /* model */
        if (strlen (optarg) > MAXMODLEN) {
	  fprintf (stderr, "decode_pdu: -m parameter too long, max. is %d\n",
	          MAXMODLEN);
	  errflag++;
	}
	else
	  strcpy (model, optarg);
        break;

      case 'z':
        if (optarg) {
          if ((default_alphabet = getcharsetID (optarg)) == -1) {
            fprintf (stderr, "Requested charset %s not supported\n", optarg);
	    exit (-1);
	  }
        }
        break;

      case 'v':                           /* version info */
	printf ("decode_pdu ver. %s (%s), SMSLink PDU Manipulation Tool.\n", 
        	DECODE_VERSION, DECODE_DATE);
	printf ("(c) Les Ateliers du Heron, 1998-2002.\n");
        exit (0);
        break;
	
      case 'h':                             /* usage help */
	printf ("decode_pdu ver. %s (%s), SMSLink PDU Manipulation Tool.\n", 
        	DECODE_VERSION, DECODE_DATE);
	printf ("(c) Les Ateliers du Heron, 1998-2002.\n");
	printf ("\nUsage:\n");
	printf ("decode_pdu [options] -s string\n");
	printf ("\nWhere: string  = pdu string to decode\n");
	printf ("Other options:\n");
        printf ("-ms       selects GSM model to emulate (def.: %s)\n", model);
        printf ("-zs       sets character set for GSMTS processing\n");
	printf ("-g        sets debugging ON [def. = OFF]\n");
	printf ("-v        prints version information and exits\n");
	printf ("-h        prints this usage information and exits\n");
        printf ("\n");
        printf ("Supported character sets:\n");
	listsupportedcharsets ();
        printf ("    (* indicates compiled-in default).\n");
        printf ("\n");
        exit (0);
        break;
	
      case ':':             /* parameter without argument */
        fprintf (stderr, "decode_pdu: Option -%c requires an argument.\n",
                optopt);
        errflag++;
        break;

      case '?':               /* unrecognized option char */
        fprintf (stderr, "decode_pdu: Unrecognized option: -%c.\n", optopt);
        errflag++;
        break;

      default:
        fprintf (stderr, "decode_pdu: getopt() returned char. code 0%o ??\n", c);
    }                                       /* switch (c) */
  }                                       /* while (c...) */
  /*----------------------Check for required parameter(s) */
  /* -s */
  if (!errflag) {
    if (strlen (string) == 0) {
      fprintf (stderr, "decode_pdu: Missing required -s parameter.\n");
      errflag++;
    }
  }                                      /* if (!errflag) */

  /*-------------------Check for conflicting parameter(s) */
  /*-----------------------On error, display usage & exit */
  if (errflag) {
    fprintf (stderr, "\n==> decode_pdu -h for usage details.\n");
    exit (2);
  }
  /* DEBUG ...............................................*/
  if (debug) {
    fprintf (stderr, "Debugging mode is turned ON.\n");
    fprintf (stderr, "Those are the current parameters :\n");
    fprintf (stderr, "string   = [%s]\n", string);
    fprintf (stderr, "charset  = [%d]\n", default_alphabet);
    fprintf (stderr, "emulating GSM model = [%s]\n", model);
  }

  /* Log current charset */
  csname = getcharsetname (default_alphabet);
  if (default_alphabet == 0) {
    /* GSM default alphabet is not meant to be used on the server side */
    fprintf (stderr, "WARNING: charset set to GSM default alphabet.\n");
    fprintf (stderr, "         This may not have the intended result !\n");
  }
  else {
    printf ("Using charset %s\n", csname);
  }
  if (csname) {
    free (csname);
  }
  
  /*---------------------------Start real processing loop */
  strcpy (gsm.model, model);
  if ((translatemodname (&gsm, GSMCAPS_FILE)) == -1) {
    fprintf (stderr, "GSM Model <%s> unsupported - using driver %s\n",
             gsm.model, GSMCAPS_FILE);
    free (string);
    exit (-1);
  }
  
  printf ("Original string: [%s] (%d char.)\n", string, strlen (string));
  decode_pdu (string, &message, gsm.capmatrix, default_alphabet);
  printf ("Message ID:   [%d]\n", message.msgid);
  printf ("Sender addr.: [%s] (%d char.)\n", message.fromgsm, strlen (message.fromgsm));
  printf ("Message date: [%s] (%d char.)\n", message.date, strlen (message.date));
  printf ("Message time: [%s] (%d char.)\n", message.time, strlen (message.time));
  printf ("Message:      [%s] (%d char.)\n", message.text, strlen (message.text));

  /*------------------------------------------Conclusions */
  /* free what's need to be */
  free (string);

  /*------------------------------------------End program */
  exit (0);
}

/*==========================================================
 * EOF : decode_pdu.c
 *===================*/
