/*==========================================================
 * Program : encode_pdu.c                  Project : smslink
 * Authors : Philippe Andersson.
 *           Franois Baligant <francois@euronet.be>.
 * Date    : 29/01/02
 * Version : 0.03b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : PDU Manipulation tool. Encodes PDU string.
 *
 * Modification History :
 * - 0.01a (13/01/02) : Initial release.
 * - 0.02a (23/01/02) : Added support for default GSM alphabet.
 * ++++ Switch to Beta ++++
 * - 0.03b (29/01/02) : Added CLI option to set the alphabet to
 *   use.
 *========================================================*/
#include <unistd.h>
#include <stdio.h>                         /* for fprintf */
#include <stdlib.h>                  /* for errno & stuff */
#include <errno.h>
#include <string.h>

#include "sms_serv.h"
#include "pdu.h"
#include "encode_pdu.h"

/*========================================================*/
/**********             LOCAL DEFINES              ********/
/*========================================================*/

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
int debug = 0;
char *progname = NULL;
int default_alphabet = DEFAULT_ALPHABET;

/*========================================================*/
/**********           MAIN PROGRAM LOOP            ********/
/*========================================================*/
int main (int argc, char *argv[])
{
  extern char *optarg;
  extern int optind, optopt;
  int c;                /* option char returned by getopt */
  int errflag = 0;       /* error in command line options */
  int task = 0;
  char *string;
  char *csname = NULL;
  char *pdu = NULL;
  struct symbols symbols;

  /*---------------------------------------Store own name */
  progname = (char *) malloc ((strlen (argv[0]) + 1) * sizeof (char));
  if (! progname) {
    syserr("main(): can't malloc()");
  }
  strcpy (progname, argv[0]);
  l_basename (progname);
  
  /*--------------------------------------Initialisations */
  string = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (!string) {
    syserr("main(): can't malloc() string");
  }
  string[0] = '\0';
  
  /*-------------------------Command line processing loop */
  while ((c = getopt (argc, argv, ":ghvs:z:")) != -1) {
    switch (c) {
      case 'g':                               /* debug ON */
        debug = DEBUG_ALL;
        break;
	
      case 's':                                 /* string */
        if (strlen (optarg) > BIGBUFF) {
	  fprintf (stderr, "encode_pdu: -s parameter too long, max. is %d\n",
	          BIGBUFF);
	  errflag++;
	}
	else
	  strcpy (string, optarg);
        break;

      case 'z':
        if (optarg) {
          if ((default_alphabet = getcharsetID (optarg)) == -1) {
            fprintf (stderr, "Requested charset %s not supported\n", optarg);
	    exit (-1);
	  }
        }
        break;

      case 'v':                           /* version info */
	printf ("encode_pdu ver. %s (%s), SMSLink PDU Manipulation Tool.\n", 
        	ENCODE_VERSION, ENCODE_DATE);
	printf ("(c) Les Ateliers du Heron, 1998-2002.\n");
        exit (0);
        break;
	
      case 'h':                             /* usage help */
	printf ("encode_pdu ver. %s (%s), SMSLink PDU Manipulation Tool.\n", 
        	ENCODE_VERSION, ENCODE_DATE);
	printf ("(c) Les Ateliers du Heron, 1998-2002.\n");
	printf ("\nUsage:\n");
	printf ("encode_pdu [options] -s string\n");
	printf ("\nWhere: string  = pdu string to decode\n");
	printf ("Other options:\n");
        printf ("-zs       sets character set for GSMTS processing\n");
	printf ("-g        sets debugging ON [def. = OFF]\n");
	printf ("-v        prints version information and exits\n");
	printf ("-h        prints this usage information and exits\n");
        printf ("\n");
        printf ("Supported character sets:\n");
	listsupportedcharsets ();
        printf ("    (* indicates compiled-in default).\n");
        printf ("\n");
        exit (0);
        break;
	
      case ':':             /* parameter without argument */
        fprintf (stderr, "encode_pdu: Option -%c requires an argument.\n",
                optopt);
        errflag++;
        break;

      case '?':               /* unrecognized option char */
        fprintf (stderr, "encode_pdu: Unrecognized option: -%c.\n", optopt);
        errflag++;
        break;

      default:
        fprintf (stderr, "encode_pdu: getopt() returned char. code 0%o ??\n", c);
    }                                       /* switch (c) */
  }                                       /* while (c...) */
  /*----------------------Check for required parameter(s) */
  /* -s */
  if (!errflag) {
    if (strlen (string) == 0) {
      fprintf (stderr, "encode_pdu: Missing required -s parameter.\n");
      errflag++;
    }
  }                                      /* if (!errflag) */

  /*-------------------Check for conflicting parameter(s) */
  /*-----------------------On error, display usage & exit */
  if (errflag) {
    fprintf (stderr, "\n==> encode_pdu -h for usage details.\n");
    exit (2);
  }
  /* DEBUG ...............................................*/
  if (debug) {
    fprintf (stderr, "Debugging mode is turned ON.\n");
    fprintf (stderr, "Those are the current parameters :\n");
    fprintf (stderr, "string   = [%s]\n", string);
    fprintf (stderr, "charset  = [%d]\n", default_alphabet);
  }

  /* Log current charset */
  csname = getcharsetname (default_alphabet);
  if (default_alphabet == 0) {
    /* GSM default alphabet is not meant to be used on the server side */
    fprintf (stderr, "WARNING: charset set to GSM default alphabet.\n");
    fprintf (stderr, "         This may not have the intended result !\n");
  }
  else {
    printf ("Using charset %s\n", csname);
  }
  if (csname) {
    free (csname);
  }
  
  /*------------------------------Fill the symbols struct */
  symbols.message = string;
  symbols.destgsm = (char *) malloc ((MAXPHNUMLEN + 1) * sizeof (char));
  if (!symbols.destgsm) {
    syserr("main(): can't malloc() symbols.destgsm");
  }
  symbols.destgsm[0] = '\0';
  strcpy (symbols.destgsm, "0123456789");
  symbols.smsc = NULL;
  symbols.pdu = NULL;
  symbols.user = NULL;
  symbols.mode = 0;
  
  /*---------------------------Start real processing loop */
  printf ("Original string: [%s] (%d char.)\n", string, strlen (string));
  pdu = encode_pdu (&symbols, default_alphabet);
  printf ("PDU string: [%s]\n", pdu);

  /*------------------------------------------Conclusions */
  /* free what's need to be */
  free (string);
  free (symbols.destgsm);
  if (pdu) {
    free (pdu);
  }

  /*------------------------------------------End program */
  exit (0);
}

/*==========================================================
 * EOF : encode_pdu.c
 *===================*/
